package com.biz.crm.tpm.business.month.budget.local.service.internal;

import com.biz.crm.tpm.business.month.budget.sdk.constant.BudgetLockConstant;
import com.biz.crm.tpm.business.month.budget.sdk.service.MonthBudgetLockService;
import com.biz.crm.mn.common.base.service.RedisLockService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @author huojia
 * @date 2022年11月01日 14:51
 */
@Slf4j
@Service
public class MonthBudgetLockServiceImpl implements MonthBudgetLockService {

    @Autowired(required = false)
    private RedisLockService redisLockService;

    /**
     * 根据预算编码加锁
     *
     * @param monthBudgetCode
     * @param timeUnit
     * @param time
     * @return boolean
     * @author huojia
     * @date 2022/11/1 21:03
     **/
    @Override
    public boolean lock(String monthBudgetCode, TimeUnit timeUnit, int time) {
        Assert.hasLength("月度预算加锁失败，预算编码不能为空");
        if (ObjectUtils.isEmpty(timeUnit)) {
            timeUnit = TimeUnit.MILLISECONDS;
        }
        if (time <= 0) {
            time = BudgetLockConstant.DEFAULT_LOCK_TIME;
        }
        return this.redisLockService.tryLock(BudgetLockConstant.MONTH_BUDGET_LOCK + monthBudgetCode, timeUnit, time);
    }

    /**
     * 根据预算编码批量加锁
     *
     * @param monthBudgetCodeList
     * @param timeUnit
     * @param time
     * @return boolean
     * @author huojia
     * @date 2022/11/1 21:04
     **/
    @Override
    public boolean lock(List<String> monthBudgetCodeList, TimeUnit timeUnit, int time) {
        return this.lock(monthBudgetCodeList, timeUnit, time, BudgetLockConstant.DEFAULT_WAITE_TIME_FIVE);
    }

    /**
     * 是否锁定
     *
     * @param monthBudgetCode
     * @return boolean
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023-12-13 23:49
     */
    @Override
    public boolean isLock(String monthBudgetCode) {
        Assert.hasLength("月度预算锁验证失败，预算编码不能为空");
        return redisLockService.isLock(BudgetLockConstant.MONTH_BUDGET_LOCK + monthBudgetCode);
    }


    /**
     * 根据预算编码批量加锁
     *
     * @param monthBudgetCodeList
     * @param timeUnit
     * @param lockTime
     * @param waiteTime
     * @return boolean
     * @author huojia
     * @date 2022/11/1 21:04
     **/
    @Override
    public boolean lock(List<String> monthBudgetCodeList, TimeUnit timeUnit, int lockTime, int waiteTime) {
        Assert.notEmpty(monthBudgetCodeList, "月度预算加锁失败，预算编码不能为空");
        if (ObjectUtils.isEmpty(timeUnit)) {
            timeUnit = TimeUnit.MILLISECONDS;
        }
        if (lockTime <= 0) {
            lockTime = BudgetLockConstant.DEFAULT_LOCK_TIME;
        }
        if (waiteTime <= 0) {
            waiteTime = BudgetLockConstant.DEFAULT_WAITE_TIME_FIVE;
        }

        boolean isLock = true;
        List<String> successKeys = new ArrayList<>();
        try {
            // 循环加锁，并记录成功的key
            for (String monthBudgetCode : monthBudgetCodeList) {
                isLock = this.redisLockService.tryLock(BudgetLockConstant.MONTH_BUDGET_LOCK + monthBudgetCode, timeUnit, lockTime, waiteTime);
                if (!isLock) {
                    return false;
                }
                successKeys.add(monthBudgetCode);
            }
        } finally {
            // 存在加锁失败的情况，则先将成功的解锁
            if (!isLock && !CollectionUtils.isEmpty(successKeys)) {
                successKeys.forEach(key -> {
                    redisLockService.unlock(BudgetLockConstant.MONTH_BUDGET_LOCK + key);
                });
            }
        }
        return true;
    }


    /**
     * 是否锁定
     *
     * @param monthBudgetCodeList
     * @return boolean
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023-12-13 23:49
     */
    @Override
    public boolean isLock(List<String> monthBudgetCodeList) {
        Assert.notEmpty(monthBudgetCodeList, "月度预算锁验证失败，预算编码不能为空");
        return redisLockService.isBatchLock(BudgetLockConstant.MONTH_BUDGET_LOCK, monthBudgetCodeList);
    }

    /**
     * 根据预算编码解锁
     *
     * @param monthBudgetCode
     * @author huojia
     * @date 2022/11/1 21:04
     **/
    @Override
    public void unLock(String monthBudgetCode) {
        Assert.hasLength("月度预算解锁失败，预算编码不能为空");
        redisLockService.unlock(BudgetLockConstant.MONTH_BUDGET_LOCK + monthBudgetCode);
    }

    /**
     * 根据预算编码批量解锁
     *
     * @param monthBudgetCodeList
     * @author huojia
     * @date 2022/11/1 21:04
     **/
    @Override
    public void unLock(List<String> monthBudgetCodeList) {
        Assert.notEmpty(monthBudgetCodeList, "月度预算解锁失败，预算编码不能为空");
        monthBudgetCodeList.forEach(monthBudgetCode -> {
            redisLockService.unlock(BudgetLockConstant.MONTH_BUDGET_LOCK + monthBudgetCode);
        });
    }

}
