package com.biz.crm.tpm.business.month.budget.local.service.internal;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.tpm.business.month.budget.local.entity.SubComMonthBudgetDetailEntity;
import com.biz.crm.tpm.business.month.budget.local.entity.SubComMonthBudgetEntity;
import com.biz.crm.tpm.business.month.budget.local.repository.SubComMonthBudgetDetailRepository;
import com.biz.crm.tpm.business.month.budget.local.repository.SubComMonthBudgetRepository;
import com.biz.crm.tpm.business.month.budget.local.service.SubComMonthBudgetCallBackService;
import com.biz.crm.tpm.business.month.budget.local.service.SubComMonthBudgetDetailService;
import com.biz.crm.tpm.business.month.budget.sdk.constant.BudgetLockConstant;
import com.biz.crm.tpm.business.month.budget.sdk.dto.SubComMonthBudgetDetailDto;
import com.biz.crm.tpm.business.month.budget.sdk.eunm.BudgetOperationTypeEnum;
import com.biz.crm.tpm.business.month.budget.sdk.service.MonthBudgetLockService;
import com.biz.crm.workflow.sdk.dto.ProcessStatusDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 月度预算回调
 *
 * @author huojia
 * @date 2022年11月12日 17:50
 */
@Slf4j
@Service
public class SubComMonthBudgetCallBackServiceImpl implements SubComMonthBudgetCallBackService {

    @Resource
    private NebulaToolkitService nebulaToolkitService;

    @Resource
    private SubComMonthBudgetRepository subComMonthBudgetRepository;

    @Resource
    private MonthBudgetLockService monthBudgetLockService;

    @Resource
    private SubComMonthBudgetDetailRepository subComMonthBudgetDetailRepository;

    @Resource
    private SubComMonthBudgetDetailService subComMonthBudgetDetailService;

    /**
     * 月度预算调整审批通过
     *
     * @param dto
     * @author huojia
     * @date 2022/11/14 9:54
     **/
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void adjustPass(ProcessStatusDto dto) {
        Validate.notNull(dto, "审批通过回调失败，请求参数不能为空！");
        Validate.notNull(dto.getProcessNo(), "审批通过回调失败，流程编码不能为空！");
        List<SubComMonthBudgetDetailEntity> subComMonthBudgetDetailEntityList = subComMonthBudgetDetailRepository.listByProcessNo(dto.getProcessNo());
        Validate.notEmpty(subComMonthBudgetDetailEntityList, "审批通过回调失败，对应分子公司月度预算审批数据不存在！");
        List<String> monthBudgetCodeList = subComMonthBudgetDetailEntityList.stream().map(SubComMonthBudgetDetailEntity::getMonthBudgetCode).collect(Collectors.toList());
        boolean lock = true;
        try {
            lock = monthBudgetLockService.lock(monthBudgetCodeList, TimeUnit.SECONDS, BudgetLockConstant.LOCK_TIME_FIVE);
            Validate.isTrue(lock, "预算调整失败，当前调入、调出预算正在操作中，请稍后！");
            // 修改审批状态，调整预算
            subComMonthBudgetDetailEntityList.forEach(subComMonthBudgetDetailEntity -> {
                subComMonthBudgetDetailEntity.setProcessStatus(ProcessStatusEnum.PASS.getDictCode());
                SubComMonthBudgetEntity subComMonthBudgetEntity = subComMonthBudgetRepository.getByMonthBudgetCode(subComMonthBudgetDetailEntity.getMonthBudgetCode(), null);
                // 调入方对应金额要改变
                if (BudgetOperationTypeEnum.ADJUST_IN.getCode().equals(subComMonthBudgetDetailEntity.getOperationType())) {
                    subComMonthBudgetDetailEntity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                    // 调整金额增加
                    subComMonthBudgetEntity.setAdjustAmount(
                            Optional.ofNullable(subComMonthBudgetEntity.getAdjustAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(subComMonthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    // 审批中金额减少
                    subComMonthBudgetEntity.setApprovingAmount(
                            Optional.ofNullable(subComMonthBudgetEntity.getApprovingAmount()).orElse(BigDecimal.ZERO)
                                    .subtract(Optional.ofNullable(subComMonthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    // 余额增加
                    subComMonthBudgetEntity.setCurrentBalanceAmount(
                            Optional.ofNullable(subComMonthBudgetEntity.getCurrentBalanceAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(subComMonthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    subComMonthBudgetDetailEntity.setBeforeAmount(
                            Optional.ofNullable(subComMonthBudgetEntity.getCurrentBalanceAmount()).orElse(BigDecimal.ZERO)
                                    .subtract(Optional.ofNullable(subComMonthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    subComMonthBudgetDetailEntity.setBalanceAmount(subComMonthBudgetEntity.getCurrentBalanceAmount());
                }
                // 调出方只用更新审批中金额
                if (BudgetOperationTypeEnum.ADJUST_OUT.getCode().equals(subComMonthBudgetDetailEntity.getOperationType())) {
                    // 审批中金额减少
                    subComMonthBudgetEntity.setApprovingAmount(
                            Optional.ofNullable(subComMonthBudgetEntity.getApprovingAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(subComMonthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO)));
                }
                subComMonthBudgetRepository.updateById(subComMonthBudgetEntity);
            });
            subComMonthBudgetDetailRepository.updateBatchById(subComMonthBudgetDetailEntityList);
        } finally {
            if (lock) {
                monthBudgetLockService.unLock(monthBudgetCodeList);
            }
        }
    }


    /**
     * 构建预算明细
     *
     * @param subComMonthBudgetEntity
     * @param operationAmount
     * @param operationType
     * @param businessCode
     * @return com.biz.crm.tpm.business.month.budget.sdk.dto.SubComMonthBudgetDetailDto
     * @author huojia
     * @date 2023/1/3 1:38
     **/
    private SubComMonthBudgetDetailDto buildDetail(SubComMonthBudgetEntity subComMonthBudgetEntity, BigDecimal operationAmount, String operationType, String businessCode) {
        SubComMonthBudgetDetailDto subComMonthBudgetDetailDto = nebulaToolkitService.copyObjectByWhiteList(subComMonthBudgetEntity, SubComMonthBudgetDetailDto.class, LinkedHashSet.class, ArrayList.class);
        subComMonthBudgetDetailDto.setId(null);
        subComMonthBudgetDetailDto.setBudgetItemCode(businessCode);
        subComMonthBudgetDetailDto.setInitialAmount(subComMonthBudgetEntity.getBudgetAmount());
        subComMonthBudgetDetailDto.setOperationType(operationType);
        // 根据不同操作类型，设置不同操作前金额
        if (BudgetOperationTypeEnum.ADJUST_IN.getCode().equals(operationType)
                || BudgetOperationTypeEnum.ADD.getCode().equals(operationType)
                || BudgetOperationTypeEnum.UNFREEZE.getCode().equals(operationType)) {
            subComMonthBudgetDetailDto.setBeforeAmount(subComMonthBudgetEntity.getCurrentBalanceAmount().subtract(operationAmount));
        } else if (BudgetOperationTypeEnum.ADJUST_OUT.getCode().equals(operationType)
                || BudgetOperationTypeEnum.SUBTRACT.getCode().equals(operationType)
                || BudgetOperationTypeEnum.FREEZE.getCode().equals(operationType)) {
            subComMonthBudgetDetailDto.setBeforeAmount(subComMonthBudgetEntity.getCurrentBalanceAmount().add(operationAmount));
        }
        subComMonthBudgetDetailDto.setCurOperationAmount(operationAmount);
        subComMonthBudgetDetailDto.setBalanceAmount(subComMonthBudgetEntity.getCurrentBalanceAmount());
        return subComMonthBudgetDetailDto;
    }

    /**
     * 月度预算调整驳回、追回
     *
     * @param dto
     * @author huojia
     * @date 2022/11/14 10:31
     **/
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void adjustRejectAndRecover(ProcessStatusDto dto) {
        Validate.notNull(dto, "回调失败，请求参数不能为空！");
        Validate.notNull(dto.getProcessNo(), "回调失败，流程编码不能为空！");
        List<SubComMonthBudgetDetailEntity> subComMonthBudgetDetailEntityList = subComMonthBudgetDetailRepository.listByProcessNo(dto.getProcessNo());
        Validate.notEmpty(subComMonthBudgetDetailEntityList, "回调失败，对应月度预算审批数据不存在！");
        List<String> monthBudgetCodeList = subComMonthBudgetDetailEntityList.stream().map(SubComMonthBudgetDetailEntity::getMonthBudgetCode).collect(Collectors.toList());
        boolean lock = true;
        try {
            lock = monthBudgetLockService.lock(monthBudgetCodeList, TimeUnit.SECONDS, BudgetLockConstant.LOCK_TIME_FIVE);
            Validate.isTrue(lock, "回调失败，预算调整失败，当前调入、调出预算正在操作中，请稍后！");
            // 修改审批状态，调整预算
            subComMonthBudgetDetailEntityList.forEach(subComMonthBudgetDetailEntity -> {
                subComMonthBudgetDetailEntity.setProcessStatus(dto.getProcessStatus());
                SubComMonthBudgetEntity subComMonthBudgetEntity = subComMonthBudgetRepository.getByMonthBudgetCode(subComMonthBudgetDetailEntity.getMonthBudgetCode(), null);
                // 调入方
                if (BudgetOperationTypeEnum.ADJUST_IN.getCode().equals(subComMonthBudgetDetailEntity.getOperationType())) {
                    // 审批中金额减少
                    subComMonthBudgetEntity.setApprovingAmount(
                            Optional.ofNullable(subComMonthBudgetEntity.getApprovingAmount()).orElse(BigDecimal.ZERO)
                                    .subtract(Optional.ofNullable(subComMonthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO)));
                    subComMonthBudgetDetailEntity.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
                }
                // 调出方预算恢复
                if (BudgetOperationTypeEnum.ADJUST_OUT.getCode().equals(subComMonthBudgetDetailEntity.getOperationType())) {
                    // 调整金额恢复
                    subComMonthBudgetEntity.setAdjustAmount(
                            Optional.ofNullable(subComMonthBudgetEntity.getAdjustAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(subComMonthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    // 审批中金额增加
                    subComMonthBudgetEntity.setApprovingAmount(
                            Optional.ofNullable(subComMonthBudgetEntity.getApprovingAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(subComMonthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    // 余额恢复
                    subComMonthBudgetEntity.setCurrentBalanceAmount(
                            Optional.ofNullable(subComMonthBudgetEntity.getCurrentBalanceAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(subComMonthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                }
                subComMonthBudgetRepository.updateById(subComMonthBudgetEntity);
                if (BudgetOperationTypeEnum.ADJUST_OUT.getCode().equals(subComMonthBudgetDetailEntity.getOperationType())) {
                    SubComMonthBudgetDetailDto subComMonthBudgetDetailDto = this.buildDetail(subComMonthBudgetEntity, subComMonthBudgetDetailEntity.getCurOperationAmount(), BudgetOperationTypeEnum.RELEASE.getCode(), null);
                    subComMonthBudgetDetailService.create(subComMonthBudgetDetailDto);
                }
                /*if (BudgetOperationTypeEnum.ADJUST_IN.getCode().equals(subComMonthBudgetDetailEntity.getOperationType())) {
                    SubComMonthBudgetDetailDto subComMonthBudgetDetailDto = this.buildDetail(subComMonthBudgetEntity, subComMonthBudgetDetailEntity.getCurOperationAmount(), BudgetOperationTypeEnum.RELEASE.getCode(), null);
                    subComMonthBudgetDetailService.create(subComMonthBudgetDetailDto);
                }*/
            });
            subComMonthBudgetDetailRepository.updateBatchById(subComMonthBudgetDetailEntityList);
        } finally {
            if (lock) {
                monthBudgetLockService.unLock(monthBudgetCodeList);
            }
        }
    }

    /**
     * 预算变更审批通过
     *
     * @param dto
     * @author huojia
     * @date 2022/11/14 11:10
     **/
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void changePass(ProcessStatusDto dto) {
        Validate.notNull(dto, "审批通过回调失败，请求参数不能为空！");
        Validate.notNull(dto.getProcessNo(), "审批通过回调失败，流程编码不能为空！");
        List<SubComMonthBudgetDetailEntity> subComMonthBudgetDetailEntityList = subComMonthBudgetDetailRepository.listByProcessNo(dto.getProcessNo());
        Validate.notEmpty(subComMonthBudgetDetailEntityList, "审批通过回调失败，对应月度预算审批数据不存在！");
        List<String> monthBudgetCodeList = subComMonthBudgetDetailEntityList.stream().map(SubComMonthBudgetDetailEntity::getMonthBudgetCode).collect(Collectors.toList());
        boolean lock = true;
        try {
            lock = monthBudgetLockService.lock(monthBudgetCodeList, TimeUnit.SECONDS, BudgetLockConstant.LOCK_TIME_FIVE);
            Validate.isTrue(lock, "回调失败，预算变更失败，当前变更预算正在操作中，请稍后！");
            subComMonthBudgetDetailEntityList.forEach(subComMonthBudgetDetailEntity -> {
                subComMonthBudgetDetailEntity.setProcessStatus(dto.getProcessStatus());
                SubComMonthBudgetEntity subComMonthBudgetEntity = subComMonthBudgetRepository.getByMonthBudgetCode(subComMonthBudgetDetailEntity.getMonthBudgetCode(), null);
                // 追加
                if (BudgetOperationTypeEnum.ADD.getCode().equals(subComMonthBudgetDetailEntity.getOperationType())) {
                    subComMonthBudgetDetailEntity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                    // 调整金额追加
                    subComMonthBudgetEntity.setAdjustAmount(
                            Optional.ofNullable(subComMonthBudgetEntity.getAdjustAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(subComMonthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    // 审批中金额减少
                    subComMonthBudgetEntity.setApprovingAmount(
                            Optional.ofNullable(subComMonthBudgetEntity.getApprovingAmount()).orElse(BigDecimal.ZERO)
                                    .subtract(Optional.ofNullable(subComMonthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    // 可用余额追加
                    subComMonthBudgetEntity.setCurrentBalanceAmount(
                            Optional.ofNullable(subComMonthBudgetEntity.getCurrentBalanceAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(subComMonthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    subComMonthBudgetDetailEntity.setBeforeAmount(
                            Optional.ofNullable(subComMonthBudgetEntity.getCurrentBalanceAmount()).orElse(BigDecimal.ZERO)
                                    .subtract(Optional.ofNullable(subComMonthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    subComMonthBudgetDetailEntity.setBalanceAmount(subComMonthBudgetEntity.getCurrentBalanceAmount());
                }
                // 削减
                if (BudgetOperationTypeEnum.SUBTRACT.getCode().equals(subComMonthBudgetDetailEntity.getOperationType())) {
                    // 审批中金额增加
                    subComMonthBudgetEntity.setApprovingAmount(
                            Optional.ofNullable(subComMonthBudgetEntity.getApprovingAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(subComMonthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                }
                subComMonthBudgetRepository.updateById(subComMonthBudgetEntity);
            });
            subComMonthBudgetDetailRepository.updateBatchById(subComMonthBudgetDetailEntityList);
        } finally {
            if (lock) {
                monthBudgetLockService.unLock(monthBudgetCodeList);
            }
        }
    }

    /**
     * 预算变更审批驳回
     *
     * @param dto
     * @author huojia
     * @date 2022/11/14 11:10
     **/
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void changeRejectAndRecover(ProcessStatusDto dto) {
        Validate.notNull(dto, "回调失败，请求参数不能为空！");
        Validate.notNull(dto.getProcessNo(), "回调失败，流程编码不能为空！");
        List<SubComMonthBudgetDetailEntity> subComMonthBudgetDetailEntityList = subComMonthBudgetDetailRepository.listByProcessNo(dto.getProcessNo());
        Validate.notEmpty(subComMonthBudgetDetailEntityList, "回调失败，对应月度预算审批数据不存在！");
        List<String> monthBudgetCodeList = subComMonthBudgetDetailEntityList.stream().map(SubComMonthBudgetDetailEntity::getMonthBudgetCode).collect(Collectors.toList());
        boolean lock = true;
        try {
            lock = monthBudgetLockService.lock(monthBudgetCodeList, TimeUnit.SECONDS, BudgetLockConstant.LOCK_TIME_FIVE);
            Validate.isTrue(lock, "预算调整失败，当前调入、调出预算正在操作中，请稍后！");
            // 修改审批状态，调整预算
            subComMonthBudgetDetailEntityList.forEach(subComMonthBudgetDetailEntity -> {
                subComMonthBudgetDetailEntity.setProcessStatus(dto.getProcessStatus());
                SubComMonthBudgetEntity subComMonthBudgetEntity = subComMonthBudgetRepository.getByMonthBudgetCode(subComMonthBudgetDetailEntity.getMonthBudgetCode(), null);
                // 追加
                if (BudgetOperationTypeEnum.ADD.getCode().equals(subComMonthBudgetDetailEntity.getOperationType())) {
                    // 审批中金额减少
                    subComMonthBudgetEntity.setApprovingAmount(
                            Optional.ofNullable(subComMonthBudgetEntity.getApprovingAmount()).orElse(BigDecimal.ZERO)
                                    .subtract(Optional.ofNullable(subComMonthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    subComMonthBudgetDetailEntity.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
                }
                // 削减
                if (BudgetOperationTypeEnum.SUBTRACT.getCode().equals(subComMonthBudgetDetailEntity.getOperationType())) {
                    // 调整金额追加
                    subComMonthBudgetEntity.setAdjustAmount(
                            Optional.ofNullable(subComMonthBudgetEntity.getAdjustAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(subComMonthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    // 审批中金额增加
                    subComMonthBudgetEntity.setApprovingAmount(
                            Optional.ofNullable(subComMonthBudgetEntity.getApprovingAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(subComMonthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    // 可用余额追加
                    subComMonthBudgetEntity.setCurrentBalanceAmount(
                            Optional.ofNullable(subComMonthBudgetEntity.getCurrentBalanceAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(subComMonthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                }
                subComMonthBudgetRepository.updateById(subComMonthBudgetEntity);
                if (BudgetOperationTypeEnum.SUBTRACT.getCode().equals(subComMonthBudgetDetailEntity.getOperationType())) {
                    SubComMonthBudgetDetailDto subComMonthBudgetDetailDto = this.buildDetail(subComMonthBudgetEntity, subComMonthBudgetDetailEntity.getCurOperationAmount(), BudgetOperationTypeEnum.RELEASE.getCode(), null);
                    subComMonthBudgetDetailService.create(subComMonthBudgetDetailDto);
                }
                /*if (BudgetOperationTypeEnum.ADD.getCode().equals(subComMonthBudgetDetailEntity.getOperationType())) {
                    SubComMonthBudgetDetailDto subComMonthBudgetDetailDto = this.buildDetail(subComMonthBudgetEntity, subComMonthBudgetDetailEntity.getCurOperationAmount(), BudgetOperationTypeEnum.RELEASE.getCode(), null);
                    subComMonthBudgetDetailService.create(subComMonthBudgetDetailDto);
                }*/
            });
            subComMonthBudgetDetailRepository.updateBatchById(subComMonthBudgetDetailEntityList);
        } finally {
            if (lock) {
                monthBudgetLockService.unLock(monthBudgetCodeList);
            }
        }
    }

    /**
     * 预算划拨审批通过
     *
     * @param dto
     * @author huojia
     * @date 2022/11/14 11:10
     **/
    @Override
    public void transferPass(ProcessStatusDto dto) {

    }

    /**
     * 预算划拨驳回
     *
     * @param dto
     * @author huojia
     * @date 2022/11/14 11:11
     **/
    @Override
    public void transferRejectAndRecover(ProcessStatusDto dto) {

    }
}
