package com.biz.crm.tpm.business.payment.receipt.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.mn.third.system.sd.sdk.dto.PaymentReceiptQueryStatusDto;
import com.biz.crm.tpm.business.payment.receipt.local.service.PaymentReceiptService;
import com.biz.crm.tpm.business.payment.receipt.sdk.constant.PaymentReceiptConstants;
import com.biz.crm.tpm.business.payment.receipt.sdk.dto.PaymentReceiptAuditPayDto;
import com.biz.crm.tpm.business.payment.receipt.sdk.dto.PaymentReceiptDto;
import com.biz.crm.tpm.business.payment.receipt.sdk.dto.PaymentReceiptProcessDto;
import com.biz.crm.tpm.business.payment.receipt.sdk.dto.PaymentReceiptShouldDto;
import com.biz.crm.tpm.business.payment.receipt.sdk.service.PaymentReceiptSdkService;
import com.biz.crm.tpm.business.payment.receipt.sdk.vo.PaymentReceiptShouldVo;
import com.biz.crm.tpm.business.payment.receipt.sdk.vo.PaymentReceiptVo;
import com.biz.crm.tpm.business.payment.receipt.sdk.vo.ReimburseFundsReasonMappingVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author: chenlong
 * @date: 2022/12/5 11:33
 * @description: 付款单管理
 */
@RestController
@RequestMapping("/v1/paymentReceipt")
@Slf4j
@Api(tags = "付款单管理")
public class PaymentReceiptController {

    @Autowired(required = false)
    private PaymentReceiptSdkService paymentReceiptSdkService;
    @Autowired(required = false)
    private RedisLockService redisLockService;
    @Autowired(required = false)
    private PaymentReceiptService paymentReceiptService;

    /**
     * 分页查询所有数据
     *
     * @param pageable 分页对象
     * @param dto      查询dto
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询所有数据")
    @GetMapping("findByConditions")
    public Result<Page<PaymentReceiptVo>> findByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                           @ApiParam(name = "dto", value = "付款单主表Dto") PaymentReceiptDto dto) {
        try {
            Page<PaymentReceiptVo> page = this.paymentReceiptSdkService.findByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 通过主键查询单条数据
     *
     * @param id 主键
     * @return 单条数据
     */
    @ApiOperation(value = "通过主键查询单条数据")
    @GetMapping("{id}")
    public Result<PaymentReceiptVo> findById(@PathVariable @ApiParam(name = "id", value = "主键id") String id) {
        try {
            PaymentReceiptVo vo = this.paymentReceiptSdkService.findById(id);
            return Result.ok(vo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 通过编码查询单条数据
     *
     * @param code 编码
     * @return 单条数据
     */
    @ApiOperation(value = "通过编码查询单条数据")
    @GetMapping("detail/{code}")
    public Result<PaymentReceiptVo> findByCode(@PathVariable @ApiParam(name = "code", value = "编码") String code) {
        try {
            PaymentReceiptVo vo = this.paymentReceiptSdkService.findByCode(code);
            return Result.ok(vo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "移动端付款单审批查询")
    @GetMapping("mobilePaymentReceiptApproved")
    public Result<?> mobilePaymentReceiptApproved(@RequestParam("processNo") String processNo) {
        try {
            return Result.ok(this.paymentReceiptSdkService.mobilePaymentReceiptApproved(processNo));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 新增数据
     *
     * @param dto dto对象
     * @return 新增结果
     */
    @ApiOperation(value = "新增数据")
    @PostMapping
    public Result<PaymentReceiptVo> create(@ApiParam(name = "dto", value = "付款单主表Dto") @RequestBody PaymentReceiptDto dto) {
        try {
            PaymentReceiptVo vo = this.paymentReceiptSdkService.create(dto);
            return Result.ok(vo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    /**
     * 汇总应付明细数据
     *
     * @param dto dto对象
     * @return 汇总应付明细数据
     */
    @ApiOperation(value = "汇总应付明细数据")
    @PostMapping("/summaryShouldInfo")
    public Result<List<PaymentReceiptShouldDto>> summaryShouldInfo(@ApiParam(name = "dto", value = "付款单主表Dto") @RequestBody PaymentReceiptDto dto) {
        try {
            List<PaymentReceiptShouldDto> paymentReceiptShouldDtos = this.paymentReceiptSdkService.summaryShouldInfo(dto);
            return Result.ok(paymentReceiptShouldDtos);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 修改数据
     *
     * @param dto dto对象
     * @return 修改结果
     */
    @ApiOperation(value = "修改数据")
    @PatchMapping
    public Result<PaymentReceiptVo> update(@ApiParam(name = "dto", value = "付款单主表Dto") @RequestBody PaymentReceiptDto dto) {
        try {
            PaymentReceiptVo vo = this.paymentReceiptSdkService.update(dto);
            return Result.ok(vo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 删除数据
     *
     * @param ids 主键结合
     * @return 删除结果
     */
    @ApiOperation(value = "删除数据")
    @DeleteMapping("delete")
    public Result<?> delete(@RequestParam("ids") List<String> ids) {
        try {
            this.paymentReceiptSdkService.delete(ids);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 启用
     *
     * @param ids 主键列表
     * @return 启用结果
     */
    @ApiOperation(value = "启用", httpMethod = "PATCH")
    @PatchMapping("/enable")
    public Result<?> enable(@RequestBody List<String> ids) {
        try {
            this.paymentReceiptSdkService.enableBatch(ids);
            return Result.ok("启用成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 禁用
     *
     * @param ids 主键列表
     * @return 禁用结果
     */
    @ApiOperation(value = "禁用", httpMethod = "PATCH")
    @PatchMapping("/disable")
    public Result<?> disable(@RequestBody List<String> ids) {
        try {
            this.paymentReceiptSdkService.disableBatch(ids);
            return Result.ok("禁用成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据核销明细编码,获取应付明细详细数据
     *
     * @param codeList 主键列表
     * @return 应付明细详细数据列表
     */
    @ApiOperation(value = "根据核销明细编码,获取应付明细详细数据")
    @GetMapping("getShouldDetailData")
    public Result<LinkedList<PaymentReceiptShouldVo>> getShouldDetailData(@ApiParam(name = "dto", value = "codeList") LinkedList<String> codeList) {
        try {
            LinkedList<PaymentReceiptShouldVo> list = this.paymentReceiptSdkService.getShouldDetailData(codeList);
            return Result.ok(list);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 分页查询可付款的核销付款信息
     *
     * @param pageable 分页对象
     * @param dto      查询dto
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询可付款的核销付款信息")
    @GetMapping("findAuditPayByConditions")
    public Result<Page<PaymentReceiptShouldVo>> findAuditPayByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                         @ApiParam(name = "dto", value = "核销付款信息Dto") PaymentReceiptAuditPayDto dto) {
        try {
            Page<PaymentReceiptShouldVo> page = this.paymentReceiptSdkService.findAuditPayByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 分页查询可付款的核销付款信息(无付款方式条件)
     *
     * @param pageable 分页对象
     * @param dto      查询dto
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询可付款的核销付款信息")
    @GetMapping("findAuditPayByConditionsWithoutPayType")
    public Result<Page<PaymentReceiptShouldVo>> findAuditPayByConditionsWithoutPayType(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                         @ApiParam(name = "dto", value = "核销付款信息Dto") PaymentReceiptAuditPayDto dto) {
        try {
            Page<PaymentReceiptShouldVo> page = this.paymentReceiptSdkService.findAuditPayByConditionsWithoutPayType(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 提交审批流
     *
     * @param processDto 工作流提交dto
     * @return Result<?>
     */
    @ApiOperation(value = "提交审批流")
    @PostMapping("submitApproval")
    public Result<?> submitApproval(@ApiParam(name = "processDto", value = "提交审批dto") @RequestBody PaymentReceiptProcessDto processDto) {
        List<PaymentReceiptDto> paymentReceiptDtos = processDto.getPaymentReceiptDtos();
        Validate.isTrue(!CollectionUtils.isEmpty(paymentReceiptDtos), "请选择数据！");
        List<String> ids = paymentReceiptDtos.stream().map(o -> o.getId()).collect(Collectors.toList());
        Validate.notNull(ids, "请选择数据！");
        boolean lockSuccess = false;
        Map<String,Boolean> lockMap = new HashMap<>();
        try {
            //提交审批前加锁一个小时
            lockSuccess = this.redisLockService.batchLock(PaymentReceiptConstants.PAYMENT_RECEIPT_LOCK_APPROVE,ids,TimeUnit.HOURS,1);
            if (lockSuccess){
                this.paymentReceiptService.submitApproval(processDto);
            }else {
                throw new UnsupportedOperationException("数据加锁异常，请重试");
            }
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (lockSuccess){
                this.redisLockService.batchUnLock(PaymentReceiptConstants.PAYMENT_RECEIPT_LOCK_APPROVE,ids);
            }
        }
    }

    /**
     * 手动推送EC
     * @param ids 付款单ID
     * @return
     */
    @ApiOperation(value = "手动推送CE")
    @PostMapping("/pushPaymentReceiptToCE")
    public Result<?> pushPaymentReceiptToEC(@ApiParam(name = "ids",value = "付款单ID") @RequestBody List<String> ids){
        try {
            Map<String, String> map = this.paymentReceiptSdkService.pushPaymentReceiptToCE(ids);
            if(Objects.nonNull(map)&&!map.isEmpty()){
                StringBuilder sb = new StringBuilder();
                for (Map.Entry<String, String> entry : map.entrySet()) {
                    sb.append("[").append(entry.getKey()).append("]:").append(entry.getValue()).append("\n");
                }
                return Result.error(sb.toString());
            }
            return Result.ok();
        }catch (Exception e){
            e.printStackTrace();
            return Result.error(e.getMessage());
        }
    }

    /**
     * 手动推送SAP
     * @param ids 付款单ID
     * @return
     */
    @ApiOperation(value = "手动推送SAP")
    @PostMapping("/pushPaymentReceiptToSAP")
    public Result<?> pushPaymentReceiptToSAP(@ApiParam(name = "id",value = "付款单ID") @RequestBody List<String> ids){
        try {
            Map<String, String> map = this.paymentReceiptSdkService.pushPaymentReceiptToSAP(ids);
            if(Objects.nonNull(map)&&!map.isEmpty()){
                StringBuilder sb = new StringBuilder();
                for (Map.Entry<String, String> entry : map.entrySet()) {
                    sb.append("[").append(entry.getKey()).append("]:").append(entry.getValue()).append("\n");
                }
                return Result.error(sb.toString());
            }
            return Result.ok();
        }catch (Exception e){
            e.printStackTrace();
            return Result.error(e.getMessage());
        }
    }

    /**
     * 查询EC状态
     * @param paymentReceiptQueryStatusDto 请求参数
     * @return
     */
    @ApiOperation(value = "查询CE状态")
    @PostMapping("/queryPaymentReceiptECStatus")
    public Result<?> queryPaymentReceiptECStatus(@ApiParam(name = "paymentReceiptQueryStatusDto",value = "请求参数") @RequestBody PaymentReceiptQueryStatusDto paymentReceiptQueryStatusDto){
        try {
            this.paymentReceiptSdkService.queryPaymentReceiptECStatus(paymentReceiptQueryStatusDto);
            return Result.ok("查询成功");
        }catch (Exception e){
            e.printStackTrace();
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "根据报销项目编码查询报销项目数据")
    @GetMapping("/findReimburseFundsReasonMappingByCode")
    public Result<ReimburseFundsReasonMappingVo> findReimburseFundsReasonMappingByCode(@RequestParam("code") String code){
        try {
            return Result.ok(this.paymentReceiptSdkService.findReimburseFundsReasonMappingByCode(code));
        }catch (Exception e){
            e.printStackTrace();
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("核销付款单生成查询")
    @GetMapping("getPaymentReceiptByAuditCode")
    public Result<?> getPaymentReceiptByAuditCode(@RequestParam("auditCode") String auditCode){

        try {
            PaymentReceiptVo paymentReceipt = this.paymentReceiptSdkService.getPaymentReceiptByAuditCode(auditCode);
            return Result.ok(paymentReceipt);
        }catch (Exception e){
            log.error("查询异常:{}", e);
            return Result.error(e.getMessage());
        }
    }
}
