package com.biz.crm.tpm.business.payment.receipt.local.repository;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.json.JSONUtil;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.tpm.business.payment.receipt.local.entity.PaymentReceipt;
import com.biz.crm.tpm.business.payment.receipt.local.enums.CommitStatusEnum;
import com.biz.crm.tpm.business.payment.receipt.local.mapper.PaymentReceiptMapper;
import com.biz.crm.tpm.business.payment.receipt.sdk.dto.PaymentReceiptDto;
import com.biz.crm.tpm.business.payment.receipt.sdk.vo.PaymentReceiptVo;
import com.biz.crm.tpm.business.payment.receipt.sdk.vo.ReimburseFundsReasonMappingVo;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * @author: chenlong
 * @date: 2022/12/5 11:34
 * @description: 付款单表(PaymentReceipt)表数据库访问层
 */
@Slf4j
@Component
public class PaymentReceiptRepository extends ServiceImpl<PaymentReceiptMapper, PaymentReceipt> {

    @Autowired(required = false)
    private PaymentReceiptMapper paymentReceiptMapper;

    /**
     * 重构修改方法
     *
     * @param entity     实体类
     * @param tenantCode 租户编码
     */
    public void updateByIdAndTenantCode(PaymentReceipt entity, String tenantCode) {
        if (null == entity) {
            return;
        }
        if (StringUtils.isEmpty(tenantCode) || StringUtils.isEmpty(entity.getId())) {
            return;
        }
        this.lambdaUpdate()
                .eq(PaymentReceipt::getTenantCode, tenantCode)
                .in(PaymentReceipt::getId, entity.getId())
                .setEntity(entity);
    }

    /**
     * 重构修改方法
     *
     * @param ids        ID集合
     * @param tenantCode 租户编码
     */
    public void updateByIdsAndTenantCode(List<String> ids, String tenantCode, String processNo) {
        if (CollectionUtils.isEmpty(ids)) {
            return;
        }
        UpdateWrapper<PaymentReceipt> wrapper = new UpdateWrapper<>();
        wrapper.eq("tenant_code", tenantCode)
                .in("id", ids)
                .set("process_state", ProcessStatusEnum.COMMIT.getDictCode())
                .set("process_no", processNo);
        this.update(wrapper);

    }

    /**
     * 更新Sap转款提交状态
     *
     * @param commitStatusEnum
     * @param id
     */
    public void updateSapTransferCommitStatus(CommitStatusEnum commitStatusEnum, String id) {
        if (StringUtils.isEmpty(id)) {
            return;
        }
        this.lambdaUpdate()
                .eq(PaymentReceipt::getId, id)
                .set(PaymentReceipt::getSapTransferCommitStatus, commitStatusEnum.getCode()).update();
    }

    /**
     * 更新Sap转款提交状态
     *
     * @param commitStatusEnum
     * @param ids
     */
    public void updateSapTransferCommitStatusBatch(CommitStatusEnum commitStatusEnum, List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.lambdaUpdate()
                .in(PaymentReceipt::getId, ids)
                .set(PaymentReceipt::getSapTransferCommitStatus, commitStatusEnum.getCode())
                .update();
    }

    /**
     * 更新CE提交状态
     *
     * @param commitStatusEnum
     * @param id
     */
    public void updateCECommitStatus(CommitStatusEnum commitStatusEnum, String id) {
        if (StringUtils.isEmpty(id)) {
            return;
        }
        this.lambdaUpdate()
                .eq(PaymentReceipt::getId, id)
                .set(PaymentReceipt::getCeCommitStatus, commitStatusEnum.getCode())
                .update();
    }

    /**
     * 更新CE提交状态
     *
     * @param commitStatusEnum
     * @param ids
     */
    public void updateCECommitStatusBatch(CommitStatusEnum commitStatusEnum, List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.lambdaUpdate()
                .in(PaymentReceipt::getId, ids)
                .set(PaymentReceipt::getCeCommitStatus, commitStatusEnum.getCode())
                .update();
    }


    /**
     * 重构批量修改方法
     *
     * @param entitys    实体类集合
     * @param tenantCode 租户编码
     */
    public void updateBatchByIdAndTenantCode(Collection<PaymentReceipt> entitys, String tenantCode) {
        if (CollectionUtils.isEmpty(entitys)) {
            return;
        }
        if (StringUtils.isEmpty(tenantCode)) {
            return;
        }
        entitys.forEach(item -> {
            if (StringUtils.isNotBlank(item.getId())) {
                LambdaUpdateWrapper<PaymentReceipt> lambdaUpdateWrapper = Wrappers.lambdaUpdate();
                lambdaUpdateWrapper.eq(PaymentReceipt::getTenantCode, tenantCode);
                lambdaUpdateWrapper.eq(PaymentReceipt::getId, item.getId());
                this.baseMapper.update(item, lambdaUpdateWrapper);
            }
        });
    }

    /**
     * 修改审批状态方法
     *
     * @param code          付款单编码
     * @param processStatus 审批状态
     */
    public void updateProcessStatus(String code, String processStatus) {
        if (StringUtils.isBlank(code)) {
            return;
        }
        this.lambdaUpdate()
                .eq(PaymentReceipt::getPaymentReceiptCode, code)
                .set(PaymentReceipt::getProcessState, processStatus)
                .update();
    }

    /**
     * 修改审批状态方法
     *
     * @param codes         付款单编码集合
     * @param processStatus 审批状态
     */
    public void updateProcessStatusBatch(List<String> codes, String processStatus) {
        if (CollectionUtils.isEmpty(codes)) {
            return;
        }
        this.lambdaUpdate()
                .in(PaymentReceipt::getPaymentReceiptCode, codes)
                .set(PaymentReceipt::getProcessState, processStatus)
                .update();
    }

    /**
     * 修改审批状态方法
     *
     * @param ids           付款单ID集合
     * @param processStatus 审批状态
     */
    public void updateProcessStatusBatchByIds(List<String> ids, String processStatus) {
        if (CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.lambdaUpdate()
                .in(PaymentReceipt::getId, ids)
                .set(PaymentReceipt::getProcessState, processStatus)
                .update();
    }

    /**
     * 根据编码获取付款单
     *
     * @param code 付款单编码
     */
    public PaymentReceipt findByCode(String code) {
        if (StringUtils.isBlank(code)) {
            return null;
        }
        return this.lambdaQuery()
                .eq(PaymentReceipt::getPaymentReceiptCode, code)
                .eq(PaymentReceipt::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .one();
    }

    /**
     * 根据编码获取付款单
     *
     * @param codes 付款单编码
     */
    public List<PaymentReceipt> findByCodes(List<String> codes) {
        if (CollectionUtils.isEmpty(codes)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .in(PaymentReceipt::getPaymentReceiptCode, codes)
                .eq(PaymentReceipt::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 分页查询数据
     *
     * @param pageable 分页对象
     * @param dto      查询Dto
     * @return Page<PaymentReceiptPayVo>
     */
    public Page<PaymentReceiptVo> findByConditions(Pageable pageable, PaymentReceiptDto dto) {
        Page<PaymentReceiptVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.paymentReceiptMapper.findByConditions(page, dto);
    }

    public List<ReimburseFundsReasonMappingVo> findAuditReasonCodeFundsOutflowMapping(List<String> codes) {
        return this.paymentReceiptMapper.findAuditReasonCodeFundsOutflowMapping(codes);
    }

    public void updatePaymentReceiptProcess(String processNo, String processStatus, String paymentReceiptCode) {
        PaymentReceipt paymentReceipt = this.paymentReceiptMapper
                .selectOne(Wrappers.lambdaQuery(PaymentReceipt.class)
                        .eq(PaymentReceipt::getPaymentReceiptCode, paymentReceiptCode));
        if (ObjectUtil.isNull(paymentReceipt)) {
            return;
        }

        log.info("付款单信息:{}", JSONUtil.toJsonStr(paymentReceipt));

        paymentReceipt.setProcessNo(processNo);
        paymentReceipt.setProcessState(processStatus);
        this.paymentReceiptMapper.updateById(paymentReceipt);
    }

    public void deleteByCode(List<String> codes) {

        if (CollUtil.isEmpty(codes)) {
            return;
        }
        this.baseMapper.delete(Wrappers.lambdaQuery(PaymentReceipt.class).in(PaymentReceipt::getPaymentReceiptCode, codes));
    }

    public void updateDelStatusById(List<String> ids, String code) {
        if (CollectionUtils.isEmpty(ids)
                || StringUtils.isEmpty(code)) {
            return;
        }
        this.lambdaUpdate()
                .in(PaymentReceipt::getId, ids)
                .set(PaymentReceipt::getDelFlag, code)
                .update();
    }
}
