package com.biz.crm.tpm.business.payment.receipt.local.repository;

import cn.hutool.core.collection.CollUtil;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.tpm.business.payment.receipt.local.entity.PaymentReceiptShould;
import com.biz.crm.tpm.business.payment.receipt.local.mapper.PaymentReceiptShouldMapper;
import com.biz.crm.tpm.business.payment.receipt.sdk.dto.PaymentReceiptAuditPayDto;
import com.biz.crm.tpm.business.payment.receipt.sdk.vo.PaymentReceiptShouldVo;
import com.biz.crm.tpm.business.payment.receipt.sdk.vo.ReimburseFundsReasonMappingVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

/**
 * @author: chenlong
 * @date: 2022/12/5 19:44
 * @description: 付款单-应付明细表(PaymentReceiptShould)表数据库访问层
 */
@Component
public class PaymentReceiptShouldRepository extends ServiceImpl<PaymentReceiptShouldMapper, PaymentReceiptShould> {

    @Autowired(required = false)
    private PaymentReceiptShouldMapper paymentReceiptShouldMapper;

    /**
     * 重构批量修改方法
     *
     * @param entitys    实体类集合
     * @param tenantCode 租户编码
     */
    public void updateBatchByIdAndTenantCode(Collection<PaymentReceiptShould> entitys, String tenantCode) {
        if (CollectionUtils.isEmpty(entitys)) {
            return;
        }
        if (StringUtils.isEmpty(tenantCode)) {
            return;
        }
        entitys.forEach(item -> {
            if (StringUtils.isNotBlank(item.getId())) {
                LambdaUpdateWrapper<PaymentReceiptShould> lambdaUpdateWrapper = Wrappers.lambdaUpdate();
                lambdaUpdateWrapper.eq(PaymentReceiptShould::getTenantCode, tenantCode);
                lambdaUpdateWrapper.eq(PaymentReceiptShould::getId, item.getId());
                this.baseMapper.update(item, lambdaUpdateWrapper);
            }
        });
    }

    /**
     * 分页查询可付款的核销付款信息
     *
     * @param pageable 分页对象
     * @param dto      查询dto
     * @return Page<PaymentReceiptShouldVo>
     */
    public Page<PaymentReceiptShouldVo> findAuditPayByConditions(Pageable pageable, PaymentReceiptAuditPayDto dto) {
        Page<PaymentReceiptShouldVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.paymentReceiptShouldMapper.findAuditPayByConditions(page, dto);
    }

    /**
     * 分页查询可付款的核销付款信息(不加付款方式条件)
     *
     * @param pageable 分页对象
     * @param dto      查询dto
     * @return Page<PaymentReceiptShouldVo>
     */
    public Page<PaymentReceiptShouldVo> findAuditPayByConditionsWithoutPayType(Pageable pageable, PaymentReceiptAuditPayDto dto) {
        Page<PaymentReceiptShouldVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.paymentReceiptShouldMapper.findAuditPayByConditionsWithoutPayType(page, dto);
    }

    public LinkedList<PaymentReceiptShouldVo> findAuditPaymentByCode(String auditCode) {
        return this.paymentReceiptShouldMapper.findAuditPaymentByCode(auditCode, TenantUtils.getTenantCode());
    }

    /**
     * 获取核销付款信息和已付金额
     *
     * @param tenantCode 租户编码
     * @param codes      核销付款信息id列表
     * @return List<PaymentReceiptShouldVo>
     */
    public List<PaymentReceiptShouldVo> findAuditPayByAuditPayCode(String tenantCode, List<String> codes) {
        if (StringUtils.isEmpty(tenantCode) || CollectionUtils.isEmpty(codes)) {
            return null;
        }
        return this.paymentReceiptShouldMapper.findAuditPayByAuditPayCode(tenantCode, codes);
    }

    /**
     * 获取核销付款信息和已付金额
     *
     * @param auditPayCode
     * @param paymentReceiptCode
     * @return List<ReimburseFundsReasonMappingVo>
     */
    public List<ReimburseFundsReasonMappingVo> findAuditReimburseMapping(String auditPayCode, String paymentReceiptCode) {
        if (StringUtils.isEmpty(auditPayCode) || StringUtils.isEmpty(paymentReceiptCode)) {
            return null;
        }
        return this.paymentReceiptShouldMapper.findAuditReimburseMapping(auditPayCode, paymentReceiptCode);
    }

    public List<PaymentReceiptShouldVo> findPaymentReceiptShouldByAudit(List<String> auditCodes){
        List<PaymentReceiptShouldVo> resultList = Lists.newArrayList();
        for (String auditCode : auditCodes) {
            LinkedList<PaymentReceiptShouldVo> list = this.paymentReceiptShouldMapper.findAuditPaymentByCode(auditCode, TenantUtils.getTenantCode());
            if (CollUtil.isNotEmpty(list)){
                resultList.addAll(list);
            }
        }
        return resultList;
    }

    public List<PaymentReceiptShould> findShouldByAuditCode(String auditCode){
        return this.paymentReceiptShouldMapper
                .selectList(Wrappers.lambdaQuery(PaymentReceiptShould.class)
                        .eq(PaymentReceiptShould::getAuditCode, auditCode));

    }

    public List<PaymentReceiptShould> findShouldByAuditCodes(List<String> auditCodes){

        return this.paymentReceiptShouldMapper.selectList(Wrappers.lambdaQuery(PaymentReceiptShould.class).in(PaymentReceiptShould::getAuditCode, auditCodes));
    }

    public void deleteShould(List<String> paymentReceiptCodes){

        if (CollUtil.isEmpty(paymentReceiptCodes)){
            return;
        }

        this.baseMapper
                .delete(Wrappers.lambdaQuery(PaymentReceiptShould.class)
                        .in(PaymentReceiptShould::getPaymentReceiptCode, paymentReceiptCodes));
    }

    /**
     * 根据核销编码查应付明细
     *
     * @param codeSet
     * @return
     */
    public List<PaymentReceiptShouldVo> findPaymentReceiptShouldByAuditCode(Set<String> codeSet) {
        return this.paymentReceiptShouldMapper.findPaymentReceiptShouldByAuditCode(codeSet, TenantUtils.getTenantCode());
    }
}
