package com.biz.crm.tpm.business.payment.receipt.local.service.internal;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.local.entity.UuidEntity;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mn.common.rocketmq.service.RocketMqProducer;
import com.biz.crm.mn.common.rocketmq.vo.MqMessageVo;
import com.biz.crm.tpm.business.payment.receipt.local.entity.PaymentReceipt;
import com.biz.crm.tpm.business.payment.receipt.local.enums.CommitStatusEnum;
import com.biz.crm.tpm.business.payment.receipt.local.enums.YesOrNoEnum;
import com.biz.crm.tpm.business.payment.receipt.local.repository.PaymentReceiptRepository;
import com.biz.crm.tpm.business.payment.receipt.local.service.PaymentReceiptService;
import com.biz.crm.tpm.business.payment.receipt.sdk.constant.PaymentReceiptConstants;
import com.biz.crm.tpm.business.payment.receipt.sdk.dto.PaymentReceiptDto;
import com.biz.crm.tpm.business.payment.receipt.sdk.dto.PaymentReceiptProcessDto;
import com.biz.crm.tpm.business.payment.receipt.sdk.service.PaymentReceiptSdkService;
import com.biz.crm.tpm.business.payment.receipt.sdk.vo.PaymentReceiptVo;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessDto;
import com.biz.crm.workflow.sdk.dto.ProcessStatusDto;
import com.biz.crm.workflow.sdk.service.ProcessBatchBusinessService;
import com.biz.crm.workflow.sdk.service.ProcessBusinessService;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author: chenlong
 * @date: 2022/12/7 10:21
 * @description: 付款单表(PaymentReceipt)表服务接口实现类
 */
@Service("PaymentReceiptService")
@Slf4j
public class PaymentReceiptServiceImpl implements PaymentReceiptService {

    @Autowired(required = false)
    private PaymentReceiptRepository paymentReceiptRepository;
    @Autowired(required = false)
    private ProcessBusinessService processBusinessService;
    @Autowired(required = false)
    private ProcessBatchBusinessService processBatchBusinessService;
    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private PaymentReceiptSdkService paymentReceiptSdkService;

    protected DictDataVoService dictDataVoService;

    @Autowired(required = false)
    private RocketMqProducer rocketMqProducer;

    @Value("${rocketmq.environment}")
    private String rocketmqEnvironment;
    @Autowired(required = false)
    private GenerateCodeService generateCodeService;






    /**
     * 付款单流程审批通过(批量)
     *
     * @param dto 流程参数
     */
    @Override
    public void processPassSendData(ProcessStatusDto dto) {
        log.info("付款单审批通过 发送消息到消息队列");
        //更新审批状态
        if (CollectionUtil.isEmpty(dto.getBusinessNoList())){
            return;
        }

        //异步推送数据
        MqMessageVo mqMessage = new MqMessageVo();
        //数据
        mqMessage.setMsgBody(JSON.toJSONString(dto.getBusinessNoList()));
        //MQ标签
        mqMessage.setTag(PaymentReceiptConstants.TPM_PAYMENT_RECEIPT_PUSH_DATA_TAG);
        mqMessage.setTopic(PaymentReceiptConstants.TPM_PAYMENT_RECEIPT_PUSH_DATA_TOPIC.concat(rocketmqEnvironment));
        rocketMqProducer.sendMqMsg(mqMessage, 10);

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void pushDataAfterProcessPass(List<String> codeList){
        if (CollectionUtil.isEmpty(codeList)){
            return;
        }
        List<PaymentReceiptVo> paymentReceiptVos = paymentReceiptSdkService.findByCodes(codeList);
        for (PaymentReceiptVo paymentReceiptVo : paymentReceiptVos) {
            if (StringUtils.equals(YesOrNoEnum.YES.getCode(), paymentReceiptVo.getIsFactoring())) {
                //是挂账保理
                //先调用SAP-FICO 【上账调用促销费用凭证接口（ZM_ITF_ACC_DOCUMENT_CREATE）】，再调用CE付款单接口【TPM180-用款业务接口】
                Result<String> result = this.paymentReceiptSdkService.pushPaymentReceiptToSAP(paymentReceiptVo);
                //推送CE
                if (result.isSuccess()) {
                    Result<String> ceResult = this.paymentReceiptSdkService.pushPaymentReceiptToCe(paymentReceiptVo);
                }
            } else {
                //不是挂账保理
                // 直接调用CE付款单接口【TPM180-用款业务接口】
                Result<String> result = this.paymentReceiptSdkService.pushPaymentReceiptToCe(paymentReceiptVo);
            }
        }

    }

    /**
     * 付款单 流程结束
     *
     * @param dto 流程参数
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void updateProcessStatusBatch(ProcessStatusDto dto) {
        paymentReceiptRepository.updateProcessStatusBatch(dto.getBusinessNoList(), dto.getProcessStatus());
    }

    /**
     * 提交工作流
     *
     * @param processDto 参数
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void submitApproval(PaymentReceiptProcessDto processDto) {
        ProcessBusinessDto processBusiness = processDto.getProcessBusiness();
        Validate.isTrue(Objects.nonNull(processBusiness), "流程参数不能为空");
        List<PaymentReceiptDto> paymentReceiptDtoList = processDto.getPaymentReceiptDtos();
        List<String> ids = paymentReceiptDtoList.stream().map(o -> o.getId()).collect(Collectors.toList());
        List<PaymentReceipt> paymentReceipts = this.paymentReceiptRepository.lambdaQuery()
                .in(PaymentReceipt::getId, ids)
                .eq(PaymentReceipt::getTenantCode, TenantUtils.getTenantCode())
                .eq(PaymentReceipt::getDelFlag, DelFlagStatusEnum.NORMAL.getCode()).list();
        Validate.isTrue(!CollectionUtils.isEmpty(paymentReceipts), "未找到对应的折扣率");
        Collection<PaymentReceiptDto> paymentReceiptDtos = nebulaToolkitService.copyCollectionByWhiteList(paymentReceipts, PaymentReceipt.class, PaymentReceiptDto.class, HashSet.class, ArrayList.class);
//        PaymentReceiptDto rateDto = nebulaToolkitService.copyObjectByWhiteList(rate, PaymentReceiptDto.class, HashSet.class, ArrayList.class);
        ArrayList<PaymentReceiptDto> paymentReceiptDtoArrayList = Lists.newArrayList(paymentReceiptDtos);
        submitApprovalHandle(paymentReceiptDtoArrayList, processDto);
    }

    /**
     * 提交工作流处理
     *
     * @param dtos
     */
    public void submitApprovalHandle(List<PaymentReceiptDto> dtos, PaymentReceiptProcessDto processDto) {
        List<String> codes = dtos.stream().map(o -> o.getPaymentReceiptCode()).collect(Collectors.toList());
//        processBusiness.setBusinessNo(dto.getPaymentReceiptCode());
//        JSONObject jsonObject = JsonUtils.toJSONObject(dtos);
        ProcessBusinessDto processBusiness = processDto.getProcessBusiness();
        processBusiness.setBusinessFormJson(JsonUtils.obj2JsonString(processDto));
        processBusiness.setBusinessCode(PaymentReceiptConstants.PROCESS_NAME_PAYMENT_RECEIPT);
        processBusiness.setBusinessNoList(codes);
        processBusiness.setBusinessNo(UUID.randomUUID().toString().replace("-",""));
        ProcessBusinessVo processBusinessVo = this.processBatchBusinessService.processStart(processBusiness);
        List<String> ids = dtos.stream().map(o -> o.getId()).collect(Collectors.toList());
        this.paymentReceiptRepository.updateByIdsAndTenantCode(ids, TenantUtils.getTenantCode(),processBusinessVo.getProcessNo());
    }
}
