package com.biz.crm.tpm.business.prepayment.details.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.tpm.business.prepayment.details.sdk.dto.PrepaymentDetailsDto;
import com.biz.crm.tpm.business.prepayment.details.sdk.service.PrepaymentDetailsService;
import com.biz.crm.tpm.business.prepayment.details.sdk.vo.BeachPrepaymentDetailsVo;
import com.biz.crm.tpm.business.prepayment.details.sdk.vo.PrepaymentDetailsVo;
import com.google.common.collect.Lists;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Set;

/**
 * @author xiaoyoujun
 * @date 2022年11月19日 11:16
 */
@RestController
@RequestMapping("/v1/prepayment/details")
@Slf4j
@Api(tags = "活动预付明细")
public class PrepaymentDetailsController {

    /**
     * 服务对象
     */
    @Autowired(required = false)
    private PrepaymentDetailsService prepaymentDetailsService;

    /**
     * 分页查询所有数据
     *
     * @param pageable 分页对象
     * @param prepaymentDetailsDto 查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询所有数据")
    @GetMapping("findByPrepayments")
    public Result<Page<PrepaymentDetailsVo>> findByPrepayments(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                               @ApiParam(name = "prepaymentDetailsDto", value = "活动预付明细") PrepaymentDetailsDto prepaymentDetailsDto) {
        try {
            Page<PrepaymentDetailsVo> page =  this.prepaymentDetailsService.findByPrepayments(pageable,prepaymentDetailsDto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    /**
     * 根据供应商编码查询供应商预付明细
     *
     * @param supplierCodes 供应商编码
     * @return 所有数据
     */
    @ApiOperation(value = "根据供应商编码查询供应商预付明细")
    @GetMapping("findBySupplierCode")
    public Result<List<BeachPrepaymentDetailsVo>> findBySupplierCode(@ApiParam(name = "supplierCode", value = "供应商编码", required = true) Set<String> supplierCodes) {
        try {
            return Result.ok(prepaymentDetailsService.findBySupplierCode(supplierCodes));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据活动明细编码、客户编码（活动明细编码 有值以活动明细为准，没值就以客户编码查询出来的为准） 取活动预付管理明细上的待冲销金额汇总
     *
     * @param dtoList
     * @return
     */
    @ApiOperation("获取待冲销金额汇总")
    @PostMapping("getAmountWrittenOff")
    public Result<List<PrepaymentDetailsVo>> getAmountWrittenOff(@ApiParam(name = "dtoList", value = "dto集合") @RequestBody List<PrepaymentDetailsDto> dtoList){
        try {
            return Result.ok(prepaymentDetailsService.getAmountWrittenOff(dtoList));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据预付编码和供应商编码批量查询供应商预付明细的预付金额和已冲销预付金额
     *
     * @param dtoList
     * @return
     */
    @ApiOperation("获取预付金额和已冲销预付金额")
    @PostMapping("getPrepayReversedAmount")
    public Result<List<PrepaymentDetailsVo>> getPrepayReversedAmount(@ApiParam(name = "dtoList", value = "dto集合") @RequestBody List<PrepaymentDetailsDto> dtoList){
        try {
            return Result.ok(prepaymentDetailsService.getPrepayReversedAmount(dtoList));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据预付编码，供应商编码冲销或取消冲销预付单
     *
     * @param dtoList
     * @return
     */
    @ApiOperation("冲销或取消冲销预付单")
    @PostMapping("reversedOrBack")
    public Result<?> reversedOrBack(@ApiParam(name = "dtoList", value = "dto集合") @RequestBody List<PrepaymentDetailsDto> dtoList){
        try {
            prepaymentDetailsService.reversedOrBack(dtoList);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }
}
