package com.biz.crm.tpm.business.prepayment.details.local.repository;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.tpm.business.prepayment.details.local.constant.PrepaymentDetailsConstant;
import com.biz.crm.tpm.business.prepayment.details.local.entity.PrepaymentDetailsEntity;
import com.biz.crm.tpm.business.prepayment.details.local.mapper.PrepaymentDetailsMapper;
import com.biz.crm.tpm.business.prepayment.details.sdk.dto.PrepaymentDetailsDto;
import com.biz.crm.tpm.business.prepayment.details.sdk.vo.BeachPrepaymentDetailsVo;
import com.biz.crm.tpm.business.prepayment.details.sdk.vo.PrepaymentDetailsVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Component
public class PrepaymentDetailsRepository extends ServiceImpl<PrepaymentDetailsMapper, PrepaymentDetailsEntity> {

    @Resource
    private PrepaymentDetailsMapper prepaymentDetailsMapper;
    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    /**
     * 分页查询数据
     *
     * @param pageable             分页对象
     * @param prepaymentDetailsDto 实体对象
     * @return
     */
    public Page<PrepaymentDetailsVo> findByPrepayments(Pageable pageable, PrepaymentDetailsDto prepaymentDetailsDto) {
        Page<PrepaymentDetailsVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        Page<PrepaymentDetailsVo> pageList = this.prepaymentDetailsMapper.findByPrepayments(page, prepaymentDetailsDto);
        return pageList;
    }

    /**
     * 根据供应商编码查询供应商预付明细
     *
     * @param supplierCodes
     * @return
     */
    public List<BeachPrepaymentDetailsVo> findBySupplierCode(Set<String> supplierCodes) {
        List<PrepaymentDetailsEntity> list = lambdaQuery().in(PrepaymentDetailsEntity::getSupplierCode, supplierCodes)
                .eq(PrepaymentDetailsEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode()).list();
        List<PrepaymentDetailsVo> prepaymentDetailsVoList = CollectionUtils.isEmpty(list) ? new ArrayList<>() : new ArrayList<>(nebulaToolkitService.copyCollectionByWhiteList(
                list, PrepaymentDetailsEntity.class, PrepaymentDetailsVo.class, LinkedHashSet.class, ArrayList.class));

        List<BeachPrepaymentDetailsVo> beachPrepaymentDetailsVoList = new ArrayList<>();
        Map<String, List<PrepaymentDetailsVo>> collect = prepaymentDetailsVoList.stream().collect(Collectors.groupingBy(PrepaymentDetailsVo::getSupplierCode));
        collect.forEach((key,value)->{
            BeachPrepaymentDetailsVo beachPrepaymentDetailsVo = new BeachPrepaymentDetailsVo();
            beachPrepaymentDetailsVo.setSupplierCode(key);
            beachPrepaymentDetailsVo.setPrepaymentDetailsList(value);
            beachPrepaymentDetailsVoList.add(beachPrepaymentDetailsVo);
        });

        return beachPrepaymentDetailsVoList;
    }

    /**
     * 获取待冲销金额汇总
     *
     * @param prepaymentDetailsDto
     * @param type
     * @return
     */
    public BigDecimal getAmountWrittenOff(PrepaymentDetailsDto prepaymentDetailsDto, String type) {
        QueryWrapper<PrepaymentDetailsEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("del_flag", DelFlagStatusEnum.NORMAL.getCode())
                    .select("IFNULL(sum(amount_written_off),0) as total");
        if (PrepaymentDetailsConstant.ACTIVITY_DETAIL_NO.equals(type)) {
            queryWrapper.eq("activity_detail_no", prepaymentDetailsDto.getActivityDetailNo());
        } else {
            queryWrapper.eq("customer_code", prepaymentDetailsDto.getCustomerCode());
        }
        Map<String, Object> map = this.getMap(queryWrapper);
        return (BigDecimal) map.get("total");
    }

    public Map<String,BigDecimal> findSumAmountWrittenOffByActivityDetailNos(List<String> activityDetailNos) {
        HashMap<String, BigDecimal> map = Maps.newHashMap();
        if(CollectionUtils.isEmpty(activityDetailNos)){
            return map;
        }
        List<PrepaymentDetailsEntity> list = this.lambdaQuery().in(PrepaymentDetailsEntity::getActivityDetailNo, activityDetailNos)
                .eq(PrepaymentDetailsEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(PrepaymentDetailsEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
        Map<String, List<PrepaymentDetailsEntity>> collect = list.stream().collect(Collectors.groupingBy(PrepaymentDetailsEntity::getActivityDetailNo));
        collect.forEach((key,value)->{
            map.put(key,value.stream().map(PrepaymentDetailsEntity::getAmountWrittenOff).filter(Objects::nonNull).reduce(BigDecimal::add).orElse(BigDecimal.ZERO));
        });
        return map;
    }

    public Map<String,BigDecimal> findSumAmountWrittenOffByCustomerCodes(List<String> customerCodes) {
        HashMap<String, BigDecimal> map = Maps.newHashMap();
        if(CollectionUtils.isEmpty(customerCodes)){
            return map;
        }
        List<PrepaymentDetailsEntity> list = this.lambdaQuery().in(PrepaymentDetailsEntity::getCustomerCode, customerCodes)
                .eq(PrepaymentDetailsEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(PrepaymentDetailsEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
        Map<String, List<PrepaymentDetailsEntity>> collect = list.stream().collect(Collectors.groupingBy(PrepaymentDetailsEntity::getCustomerCode));
        collect.forEach((key,value)->{
            map.put(key,value.stream().map(PrepaymentDetailsEntity::getAmountWrittenOff).filter(Objects::nonNull).reduce(BigDecimal::add).orElse(BigDecimal.ZERO));
        });
        return map;
    }

    /**
     * 获取预付金额和已冲销预付金额
     *
     * @param prepaymentDetailsDto
     * @return
     */
    public List<PrepaymentDetailsVo> getPrepayReversedAmount(PrepaymentDetailsDto prepaymentDetailsDto) {
        List<PrepaymentDetailsEntity> list = lambdaQuery().eq(PrepaymentDetailsEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(PrepaymentDetailsEntity::getPrepaidCoding, prepaymentDetailsDto.getPrepaidCoding())
                .eq(PrepaymentDetailsEntity::getSupplierCode, prepaymentDetailsDto.getSupplierCode()).list();
        return CollectionUtils.isEmpty(list) ? new ArrayList<>() : new ArrayList<>(nebulaToolkitService.copyCollectionByWhiteList(
                list, PrepaymentDetailsEntity.class, PrepaymentDetailsVo.class, LinkedHashSet.class, ArrayList.class));
    }

    /**
     * 冲销或取消冲销预付单
     *
     * @param dtoList
     */
    public void reversedOrBack(List<PrepaymentDetailsDto> dtoList) {
        if(CollectionUtils.isEmpty(dtoList)){
            return;
        }
        for (PrepaymentDetailsDto prepaymentDetailsDto : dtoList) {
            PrepaymentDetailsEntity one = this.lambdaQuery().eq(PrepaymentDetailsEntity::getPrepaidCoding, prepaymentDetailsDto.getPrepaidCoding())
                    .eq(PrepaymentDetailsEntity::getSupplierCode, prepaymentDetailsDto.getSupplierCode()).one();

            this.lambdaUpdate().eq(PrepaymentDetailsEntity::getPrepaidCoding,prepaymentDetailsDto.getPrepaidCoding())
                    .eq(PrepaymentDetailsEntity::getSupplierCode,prepaymentDetailsDto.getSupplierCode())
                    .set(PrepaymentDetailsEntity::getAmountWrittenOff,Optional.ofNullable(one.getAmountWrittenOff()).orElse(BigDecimal.ZERO).subtract(prepaymentDetailsDto.getReversedAmount()))
                    .set(PrepaymentDetailsEntity::getReversedAmount,Optional.ofNullable(one.getReversedAmount()).orElse(BigDecimal.ZERO).add(prepaymentDetailsDto.getReversedAmount()));
        }
//        prepaymentDetailsMapper.reversedOrBack(dtoList);
    }

    /**
     * 修改冲销金额
     *
     * @param dtoList
     */
    public void updateReversedAmount(List<PrepaymentDetailsDto> dtoList) {
        for (PrepaymentDetailsDto prepaymentDetailsDto : dtoList) {

            List<PrepaymentDetailsEntity> list = this.lambdaQuery()
                    .eq(StringUtils.isNotBlank(prepaymentDetailsDto.getActivityDetailNo()), PrepaymentDetailsEntity::getActivityDetailNo, prepaymentDetailsDto.getActivityDetailNo())
                    .eq(StringUtils.isBlank(prepaymentDetailsDto.getActivityDetailNo()) && StringUtils.isNotEmpty(prepaymentDetailsDto.getCustomerCode()), PrepaymentDetailsEntity::getCustomerCode, prepaymentDetailsDto.getCustomerCode())
                    .eq(PrepaymentDetailsEntity::getTenantCode, TenantUtils.getTenantCode())
                    .eq(PrepaymentDetailsEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                    .list();

            Validate.notNull(list, "未找到对应的活动预付明细");

            list = list.stream().sorted(Comparator.comparing(PrepaymentDetailsEntity::getCreateTime)).collect(Collectors.toList());
            //结案核销准备冲销的金额
            BigDecimal auditReversedAmount = prepaymentDetailsDto.getReversedAmount();

            List<PrepaymentDetailsEntity> updateList = new ArrayList<>();
            Set<String> prepaidCodingSet = new HashSet<>();
            for (PrepaymentDetailsEntity entity : list) {
                PrepaymentDetailsEntity entityUpdate = new PrepaymentDetailsEntity();
                entityUpdate.setId(entity.getId());
                prepaidCodingSet.add(entity.getPrepaidCoding());
                if(auditReversedAmount.compareTo(entity.getAmountWrittenOff())<=0){
                    entityUpdate.setReversedAmount(entity.getReversedAmount().add(auditReversedAmount));
                    entityUpdate.setAmountWrittenOff(entity.getAmountWrittenOff().subtract(auditReversedAmount));
                    updateList.add(entityUpdate);
                    break;
                }else {
                    entityUpdate.setReversedAmount(entity.getReversedAmount().add(entity.getAmountWrittenOff()));
                    entityUpdate.setAmountWrittenOff(BigDecimal.ZERO);
                    updateList.add(entityUpdate);
                }
            }

            this.updateBatchById(updateList);
            refreshPrepaymentAmount(new ArrayList<>(prepaidCodingSet));
        }
    }

    /**
     * 刷新活动预付单冲销金额
     *
     * @param prepaidCodingList
     */
    public void refreshPrepaymentAmount(List<String> prepaidCodingList) {
        List<List<String>> partition = Lists.partition(prepaidCodingList, 500);
        for (List<String> part : partition) {
            prepaymentDetailsMapper.refreshPrepaymentAmount(part);
        }
    }

    public List<PrepaymentDetailsEntity> findForWithholding(String beginDate, String endDate, PrepaymentDetailsDto prepaymentDetailsDto) {
        if(prepaymentDetailsDto != null) {
            return this.lambdaQuery()
                    .ge(StringUtils.isNotEmpty(beginDate), PrepaymentDetailsEntity::getPrepaymentDate, beginDate)
                    .lt(StringUtils.isNotEmpty(endDate), PrepaymentDetailsEntity::getPrepaymentDate, endDate)
                    .eq(PrepaymentDetailsEntity::getBusinessUnitCode, prepaymentDetailsDto.getBusinessUnitCode())
                    .eq(StringUtils.isNotEmpty(prepaymentDetailsDto.getCustomerCode()), PrepaymentDetailsEntity::getCustomerCode, prepaymentDetailsDto.getCustomerCode())
                    .eq(StringUtils.isNotEmpty(prepaymentDetailsDto.getSalesOrgCode()), PrepaymentDetailsEntity::getSalesOrgCode, prepaymentDetailsDto.getSalesOrgCode())
                    .eq(PrepaymentDetailsEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                    .ne(PrepaymentDetailsEntity::getBusinessUnitCode, BusinessUnitEnum.HEADQUARTERS.getCode())
                    .ne(PrepaymentDetailsEntity::getAmountWrittenOff, BigDecimal.ZERO)
                    .list();
        } else {
            return this.lambdaQuery()
                    .ge(StringUtils.isNotEmpty(beginDate), PrepaymentDetailsEntity::getPrepaymentDate, beginDate)
                    .lt(StringUtils.isNotEmpty(endDate), PrepaymentDetailsEntity::getPrepaymentDate, endDate)
                    .eq(PrepaymentDetailsEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                    .ne(PrepaymentDetailsEntity::getBusinessUnitCode, BusinessUnitEnum.HEADQUARTERS.getCode())
                    .ne(PrepaymentDetailsEntity::getAmountWrittenOff, BigDecimal.ZERO)
                    .list();
        }

    }

    public List<PrepaymentDetailsVo> findUpwardForWithholding(String beginDate, String endDate, PrepaymentDetailsDto dto, List<String> customerCodeInStrList) {
        return prepaymentDetailsMapper.findUpwardForWithholding(beginDate, endDate, dto, customerCodeInStrList);
    }
}
