package com.biz.crm.tpm.business.prepayment.details.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgSubComOrgService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgSubComOrgVo;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.tpm.business.prepayment.details.local.constant.PrepaymentDetailsConstant;
import com.biz.crm.tpm.business.prepayment.details.local.entity.PrepaymentDetailsEntity;
import com.biz.crm.tpm.business.prepayment.details.local.repository.PrepaymentDetailsRepository;
import com.biz.crm.tpm.business.prepayment.details.sdk.constant.PrepaymentSdkConstant;
import com.biz.crm.tpm.business.prepayment.details.sdk.dto.PrepaymentDetailsDto;
import com.biz.crm.tpm.business.prepayment.details.sdk.service.PrepaymentDetailsService;
import com.biz.crm.tpm.business.prepayment.details.sdk.vo.BeachPrepaymentDetailsVo;
import com.biz.crm.tpm.business.prepayment.details.sdk.vo.PrepaymentDetailsVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.logging.impl.LogKitLogger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 活动预付明细(PrepaymentDetailsEntity)服务实现类
 *
 * @author xiaoyoujun
 * @date 2022年11月19日 11:30
 */
@Service("prepaymentDetailsService")
@Slf4j
public class PrepaymentDetailsServiceImpl implements PrepaymentDetailsService {

    @Autowired(required = false)
    private PrepaymentDetailsRepository prepaymentDetailsRepository;
    
    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    /**
     * 分页查询数据
     *
     * @param pageable             分页对象
     * @param prepaymentDetailsDto 实体对象
     * @return
     */
    @Override
    public Page<PrepaymentDetailsVo> findByPrepayments(Pageable pageable, PrepaymentDetailsDto prepaymentDetailsDto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(prepaymentDetailsDto)) {
            prepaymentDetailsDto = new PrepaymentDetailsDto();
        }
        return this.prepaymentDetailsRepository.findByPrepayments(pageable, prepaymentDetailsDto);
    }

    /**
     * 新增数据
     *
     * @param dtoList 集合实体对象
     * @return 新增结果
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void create(List<PrepaymentDetailsDto> dtoList) {
        Assert.notEmpty(dtoList, "数据不能为空!");
        Collection<PrepaymentDetailsEntity> entityList = this.nebulaToolkitService.copyCollectionByWhiteList(dtoList,
                PrepaymentDetailsDto.class, PrepaymentDetailsEntity.class, LinkedHashSet.class, ArrayList.class);
        this.prepaymentDetailsRepository.saveOrUpdateBatch(entityList);
    }

    /**
     * 根据供应商编码查询供应商预付明细
     *
     * @param supplierCodes
     * @return
     */
    @Override
    public List<BeachPrepaymentDetailsVo> findBySupplierCode(Set<String> supplierCodes) {
        return prepaymentDetailsRepository.findBySupplierCode(supplierCodes);
    }

    /**
     * 获取待冲销金额汇总
     *
     * @param dtoList
     * @return
     */
    @Override
    public List<PrepaymentDetailsVo> getAmountWrittenOff(List<PrepaymentDetailsDto> dtoList) {
        if (CollectionUtils.isEmpty(dtoList)) {
            return new ArrayList<>();
        }
        List<PrepaymentDetailsVo> voList = new ArrayList<>();
        for (PrepaymentDetailsDto dto : dtoList) {
            PrepaymentDetailsVo vo = new PrepaymentDetailsVo();
            BigDecimal amountWrittenOff;
            if (StringUtils.isNotBlank(dto.getActivityDetailNo())) {
                amountWrittenOff = prepaymentDetailsRepository.getAmountWrittenOff(dto, PrepaymentDetailsConstant.ACTIVITY_DETAIL_NO);
            } else if (StringUtils.isNotBlank(dto.getCustomerCode())) {
                amountWrittenOff = prepaymentDetailsRepository.getAmountWrittenOff(dto, PrepaymentDetailsConstant.CUSTOMER_CODE);
            } else {
                continue;
            }
            vo.setActivityDetailNo(dto.getActivityDetailNo());
            vo.setCustomerCode(dto.getCustomerCode());
            vo.setAmountWrittenOff(amountWrittenOff);
            voList.add(vo);
        }
        return voList;
    }


    @Override
    public List<PrepaymentDetailsVo> getAmountWrittenOff2(List<PrepaymentDetailsDto> dtoList) {
        if (CollectionUtils.isEmpty(dtoList)) {
            return new ArrayList<>();
        }
        List<String> activityDetailNos = dtoList.stream().map(PrepaymentDetailsDto::getActivityDetailNo).filter(StringUtils::isNotEmpty).distinct().collect(Collectors.toList());
        List<String> customerCodes = dtoList.stream().map(PrepaymentDetailsDto::getCustomerCode).filter(StringUtils::isNotEmpty).distinct().collect(Collectors.toList());
        Map<String, BigDecimal> activityDetailPrepaymentAmountMap = new HashMap<>();
        Map<String, BigDecimal> customerCodePrepaymentAmountMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(activityDetailNos)) {
            activityDetailPrepaymentAmountMap = prepaymentDetailsRepository.findSumAmountWrittenOffByActivityDetailNos(activityDetailNos);
        }
        if (!CollectionUtils.isEmpty(customerCodes)) {
            customerCodePrepaymentAmountMap = prepaymentDetailsRepository.findSumAmountWrittenOffByCustomerCodes(customerCodes);
        }
        List<PrepaymentDetailsVo> voList = new ArrayList<>();
        for (PrepaymentDetailsDto dto : dtoList) {
            PrepaymentDetailsVo vo = new PrepaymentDetailsVo();
            BigDecimal amountWrittenOff;
            if (StringUtils.isNotBlank(dto.getActivityDetailNo())) {
                amountWrittenOff = activityDetailPrepaymentAmountMap.get(dto.getActivityDetailNo());
            } else if (StringUtils.isNotBlank(dto.getCustomerCode())) {
                amountWrittenOff = customerCodePrepaymentAmountMap.get(dto.getCustomerCode());
            } else {
                continue;
            }
            vo.setActivityDetailNo(dto.getActivityDetailNo());
            vo.setCustomerCode(dto.getCustomerCode());
            vo.setAmountWrittenOff(amountWrittenOff);
            voList.add(vo);
        }
        return voList;
    }

    /**
     * 获取预付金额和已冲销预付金额
     *
     * @param dtoList
     * @return
     */
    @Override
    public List<PrepaymentDetailsVo> getPrepayReversedAmount(List<PrepaymentDetailsDto> dtoList) {
        if (CollectionUtils.isEmpty(dtoList)) {
            return new ArrayList<>();
        }
        List<PrepaymentDetailsVo> voListAll = new ArrayList<>();
        for (PrepaymentDetailsDto dto : dtoList) {
            Validate.notBlank(dto.getPrepaidCoding(), "预付编码, 不能为空");
            Validate.notBlank(dto.getSupplierCode(), "供应商编码, 不能为空");
            voListAll.addAll(prepaymentDetailsRepository.getPrepayReversedAmount(dto));
        }
        return voListAll;
    }

    /**
     * 冲销或取消冲销预付单
     *
     * @param dtoList
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void reversedOrBack(List<PrepaymentDetailsDto> dtoList) {
        if (CollectionUtils.isEmpty(dtoList)) {
            return;
        }
        Set<String> lockList = new HashSet<>();
        for (PrepaymentDetailsDto dto : dtoList) {
            Validate.notBlank(dto.getPrepaidCoding(), "预付编码, 不能为空");
            Validate.notBlank(dto.getSupplierCode(), "供应商编码, 不能为空");
            lockList.add(dto.getPrepaidCoding() + dto.getSupplierCode());
        }
        boolean lockSuccess = redisLockService.batchLock(PrepaymentSdkConstant.TPM_PREPAYMENT_DETAILS_LOCK, new ArrayList<>(lockList), TimeUnit.MINUTES, 5);
        Assert.isTrue(lockSuccess, "其他人正在操作数据,加锁失败,请稍后重试!");
        try {
            prepaymentDetailsRepository.reversedOrBack(dtoList);
        } finally {
            redisLockService.batchUnLock(PrepaymentSdkConstant.TPM_PREPAYMENT_DETAILS_LOCK, new ArrayList<>(lockList));
        }
    }

    /**
     * 修改冲销金额
     *
     * @param dtoList
     * @return
     */
    @Override
    public void updateReversedAmount(List<PrepaymentDetailsDto> dtoList) {
        if (CollectionUtils.isEmpty(dtoList)) {
            return;
        }
        for (PrepaymentDetailsDto dto : dtoList) {
            Validate.notNull(dto.getReversedAmount(), "冲销金额, 不能为空");
            if (StringUtils.isBlank(dto.getActivityDetailNo()) && StringUtils.isBlank(dto.getCustomerCode())) {
                throw new IllegalArgumentException("活动明细编号和客户编码不能同时为空");
            }
        }
        prepaymentDetailsRepository.updateReversedAmount(dtoList);
    }

    @Override
    public List<PrepaymentDetailsVo> findForWithholding(String beginDate, String endDate, PrepaymentDetailsDto prepaymentDetailsDto) {
        List<PrepaymentDetailsEntity> entityList = prepaymentDetailsRepository.findForWithholding(beginDate, endDate, prepaymentDetailsDto);
        return (List<PrepaymentDetailsVo>) nebulaToolkitService.copyCollectionByWhiteList(entityList, PrepaymentDetailsEntity.class, PrepaymentDetailsVo.class, LinkedHashSet.class, ArrayList.class);
    }

    @Autowired(required = false)
    private SalesOrgSubComOrgService salesOrgSubComOrgService;

    @Autowired
    private CustomerVoService customerVoService;

    @Override
    public List<PrepaymentDetailsVo> findUpwardForWithholding(String beginDate, String endDate, PrepaymentDetailsDto dto) {
        //业务启用后  改成分页查询
//        List<SalesOrgSubComOrgVo> salesOrgSubList = salesOrgSubComOrgService.findAll();
//        Set<String> customerErpCodeList = salesOrgSubList.stream().map(SalesOrgSubComOrgVo::getSubComOrgCode).collect(Collectors.toSet());
//        List<CustomerVo> customerList = customerVoService.findByErpCodeList(new ArrayList<>(customerErpCodeList));
//        List<String> customerCodeList = customerList.stream().map(CustomerVo::getCustomerCode).collect(Collectors.toList());
//        //拼接客户编码查询逻辑
//        List<String> customerCodeInStrList = new ArrayList<>();
//        int pageNum = 0;
//        int pageSize = 500;
//        for (; pageNum * pageSize < customerCodeList.size(); pageNum++) {
//            if (pageNum * pageSize + pageSize > customerCodeList.size()) {
//                customerCodeInStrList.add("('" + StringUtils.join(customerCodeList.subList(pageNum * pageSize, customerCodeList.size()), "','") + "')");
//            } else {
//                customerCodeInStrList.add("('" + StringUtils.join(customerCodeList.subList(pageNum * pageSize, pageNum * pageSize + pageSize), "','") + "')");
//            }
//        }
//        return prepaymentDetailsRepository.findUpwardForWithholding(beginDate, endDate, dto, customerCodeInStrList);
        return new ArrayList<>();
    }


}
