package com.biz.crm.tpm.business.promotion.plan.local.controller;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.CurrentMonthSaleDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.CurrentMonthSaleService;
import com.biz.crm.tpm.business.promotion.plan.sdk.vo.CurrentMonthSaleVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;

import java.util.List;

/**
 * 当月销售实体类(CurrentMonthSale)表相关的http接口
 *
 * @author yaoyongming
 * @since 2022-11-04 10:54:33
 */
@RestController
@RequestMapping("/v1/currentMonthSale")
@Slf4j
@Api(tags = "促销规划：当月销售")
public class CurrentMonthSaleController {
    /**
     * 服务对象
     */
    @Autowired(required = false)
    private CurrentMonthSaleService currentMonthSaleService;

    /**
     * 分页查询所有数据
     *
     * @param pageable         分页对象
     * @param currentMonthSale 查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询所有数据")
    @GetMapping("findByConditions")
    public Result<Page<CurrentMonthSaleVo>> findByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                             @ApiParam(name = "currentMonthSale", value = "当月销售实体类") CurrentMonthSaleDto currentMonthSale) {
        try {
            Page<CurrentMonthSaleVo> page = this.currentMonthSaleService.findByConditions(pageable, currentMonthSale);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 1、获取明细列表缓存分页接口
     *
     * @param pageable 分页参数
     * @param cacheKey 缓存key
     * @return
     */
    @ApiOperation(value = "获取明细列表缓存分页接口")
    @GetMapping("findCachePageList")
    public Result<Page<CurrentMonthSaleVo>> findCachePageList(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                              @ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                                              @ApiParam(name = "promotionPlanCode", value = "促销规划编码") @RequestParam(required = false) String promotionPlanCode) {
        try {
            Page<CurrentMonthSaleVo> page = this.currentMonthSaleService.findCachePageList(pageable, cacheKey, promotionPlanCode);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 2、新增一行接口：保存当前页数据后，在缓存中行首插入一条数据并返回第一页数据
     *
     * @param pageable 分页参数
     * @param cacheKey 缓存key
     * @param itemList 要保存的当前页数据
     * @return
     */
    @ApiOperation(value = "新增一行接口，保存当前页数据后，在缓存中行首插入一条数据并返回第一页数据")
    @PostMapping("addItemCache")
    public Result<Page<CurrentMonthSaleVo>> addItemCache(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                         @ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                                         @ApiParam(value = "当前页数据") @RequestBody List<CurrentMonthSaleDto> itemList) {
        try {
            this.currentMonthSaleService.addItemCache(cacheKey, itemList);
            Page<CurrentMonthSaleVo> page = this.currentMonthSaleService.findCachePageList(pageable, cacheKey, null);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    /**
     * 3、复制行接口：保存当前页数据后，在缓存中行首复制选中数据并返回第一页数据
     *
     * @param pageable 分页参数
     * @param cacheKey 缓存key
     * @return
     */
    @ApiOperation(value = "复制行接口，保存当前页数据后，在缓存中行首复制选中数据并返回第一页数据")
    @PostMapping("copyItemListCache")
    public Result<Page<CurrentMonthSaleVo>> copyItemListCache(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                              @ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                                              @ApiParam(value = "当前页数据，包含要复制的行勾选信息") @RequestBody List<CurrentMonthSaleDto> itemList) {
        try {
            this.currentMonthSaleService.copyItemListCache(cacheKey, itemList);
            Page<CurrentMonthSaleVo> page = this.currentMonthSaleService.findCachePageList(pageable, cacheKey, null);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 4、保存当前页数据到缓存并返回指定页数据接口
     *
     * @param pageable 分页参数
     * @param cacheKey 缓存key
     * @return
     */
    @ApiOperation(value = "保存当前页数据到缓存并返回指定页数据接口")
    @PostMapping("saveCurrentPageCache")
    public Result<Page<CurrentMonthSaleVo>> saveCurrentPageCache(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                 @ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                                                 @ApiParam(value = "当前页数据") @RequestBody List<CurrentMonthSaleDto> saveList) {
        try {
            this.currentMonthSaleService.saveCurrentPageCache(cacheKey, saveList);
            Page<CurrentMonthSaleVo> page = this.currentMonthSaleService.findCachePageList(pageable, cacheKey, null);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 7、删除行接口：删除有选中标记的行，更新未标记数据，并返回指定页数据接口
     *
     * @param pageable 分页参数
     * @param cacheKey 缓存key
     * @return
     */
    @ApiOperation(value = "多行删除并返回指定页数据接口")
    @PostMapping("deleteCacheList")
    public Result<Page<CurrentMonthSaleVo>> deleteCacheList(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                            @ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                                            @ApiParam(value = "当前页数据，包含要删除的行勾选信息") @RequestBody List<CurrentMonthSaleDto> itemList) {
        try {
            this.currentMonthSaleService.deleteCacheList(cacheKey, itemList);
            Page<CurrentMonthSaleVo> page = this.currentMonthSaleService.findCachePageList(pageable, cacheKey, null);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 6、清理缓存接口
     *
     * @param cacheKey 缓存key
     */
    @ApiOperation(value = "清理缓存接口")
    @PostMapping("clearCache")
    public Result clearCache(@ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey) {
        try {
            this.currentMonthSaleService.clearCache(cacheKey);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

}
