package com.biz.crm.tpm.business.promotion.plan.local.controller;


import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetBusinessPolicyQueryVo;
import com.biz.crm.tpm.business.promotion.plan.sdk.constant.PromotionPlanConstant;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.PromotionPlanCostProgressDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.PromotionPlanDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.PromotionPlanForEventPrepaymentDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.PromotionPlanSubmitDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.*;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.tpm.business.promotion.plan.sdk.vo.*;
import com.biz.crm.mn.common.base.service.RedisLockService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.*;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 促销规划实体类(PromotionPlanEntity)表相关的http接口
 *
 * @author yaoyongming
 * @since 2022-11-02 18:14:37
 */
@RestController
@RequestMapping("/v1/promotionPlan")
@Slf4j
@Api(tags = "促销规划")
public class PromotionPlanController {
    /**
     * 服务对象
     */
    @Autowired(required = false)
    private PromotionPlanService promotionPlanService;
    /**
     * redis服务
     */
    @Autowired(required = false)
    private RedisLockService redisLockService;

    @Autowired(required = false)
    private GeneralExpensesService generalExpensesService;

    @Autowired(required = false)
    private CurrentMonthSaleService currentMonthSaleService;

    @Autowired(required = false)
    private CurrentMonthDeliveryService currentMonthDeliveryService;

    @Autowired(required = false)
    private PromotionPlanCostProgressService promotionPlanCostProgressService;

    /**
     * 分页查询所有数据
     *
     * @param pageable      分页对象
     * @param promotionPlan 查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询所有数据")
    @GetMapping("findByConditions")
    public Result<Page<PromotionPlanVo>> findByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                          @ApiParam(name = "promotionPlan", value = "促销规划实体类") PromotionPlanDto promotionPlan) {
        try {
            Page<PromotionPlanVo> page = this.promotionPlanService.findByConditions(pageable, promotionPlan);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 通过主键查询单条数据
     *
     * @param promotionPlanCode 主键
     * @return 单条数据
     */
    @ApiOperation(value = "通过主键查询单条数据")
    @GetMapping("findById")
    public Result<PromotionPlanVo> findById(@RequestParam("promotionPlanCode") @ApiParam(name = "promotionPlanCode", value = "促销规划编码", required = true) String promotionPlanCode) {
        try {
            PromotionPlanVo promotionPlan = this.promotionPlanService.findById(promotionPlanCode);
            return Result.ok(promotionPlan);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 通过编码查询单条数据
     *
     * @param promotionPlanCode 编码
     * @return 单条数据
     */
    @ApiOperation(value = "通过编码查询单条数据")
    @GetMapping("findByCode")
    public Result<PromotionPlanVo> findByCode(@RequestParam("promotionPlanCode") @ApiParam(name = "promotionPlanCode", value = "促销规划编码", required = true) String promotionPlanCode) {
        try {
            PromotionPlanVo promotionPlan = this.promotionPlanService.findByCode(promotionPlanCode);
            return Result.ok(promotionPlan);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 新增数据
     *
     * @param promotionPlan 实体对象
     * @return 新增结果
     */
    @ApiOperation(value = "新增数据")
    @PostMapping
    public Result<?> create(@ApiParam(name = "promotionPlan", value = "促销规划实体类") @RequestBody PromotionPlanDto promotionPlan,
                            @ApiParam(name = "cacheKeySale", value = "当月销售缓存键")@RequestParam(required = false) String cacheKeySale,
                            @ApiParam(name = "cacheKeyDelivery", value = "当月送货缓存键")@RequestParam(required = false) String cacheKeyDelivery,
                            @ApiParam(name = "cacheKeyPurchase", value = "采销库存缓存键")@RequestParam(required = false) String cacheKeyPurchase,
                            @ApiParam(name = "cacheKeyGeneral", value = "费用申请缓存键")@RequestParam(required = false) String cacheKeyGeneral,
                            @ApiParam(name = "cacheKeyOther", value = "其他费用缓存键")@RequestParam(required = false) String cacheKeyOther) {
        try {
            this.promotionPlanService.create(promotionPlan, cacheKeySale, cacheKeyDelivery, cacheKeyPurchase, cacheKeyGeneral, cacheKeyOther);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 修改数据
     *
     * @param promotionPlan 实体对象
     * @return 修改结果
     */
    @ApiOperation(value = "修改数据")
    @PatchMapping
    public Result<?> update(@ApiParam(name = "promotionPlan", value = "促销规划实体类") @RequestBody PromotionPlanDto promotionPlan,
                            @ApiParam(name = "cacheKeySale", value = "当月销售缓存键")@RequestParam(required = false) String cacheKeySale,
                            @ApiParam(name = "cacheKeyDelivery", value = "当月送货缓存键")@RequestParam(required = false) String cacheKeyDelivery,
                            @ApiParam(name = "cacheKeyPurchase", value = "采销库存缓存键")@RequestParam(required = false) String cacheKeyPurchase,
                            @ApiParam(name = "cacheKeyGeneral", value = "费用申请缓存键")@RequestParam(required = false) String cacheKeyGeneral,
                            @ApiParam(name = "cacheKeyOther", value = "其他费用缓存键")@RequestParam(required = false) String cacheKeyOther) {
        try {
            this.promotionPlanService.update(promotionPlan, cacheKeySale, cacheKeyDelivery, cacheKeyPurchase, cacheKeyGeneral, cacheKeyOther);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 删除数据
     *
     * @param ids 主键结合
     * @return 删除结果
     */
    @ApiOperation(value = "删除数据")
    @DeleteMapping
    public Result<?> delete(@ApiParam(name = "ids", value = "主键集合") @RequestParam("ids") List<String> ids) {
        try {
            this.promotionPlanService.delete(ids);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 分页查询费用申请
     *
     * @param pageable      分页对象
     * @param promotionPlan 查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询费用申请")
    @GetMapping("findByConditionsGeneral")
    public Result<Page<GeneralExpensesVo>> findByConditionsGeneral(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                            @ApiParam(name = "promotionPlan", value = "促销规划实体类") PromotionPlanDto promotionPlan) {
        try {
            Page<GeneralExpensesVo> page = this.promotionPlanService.findByConditionsGeneral(pageable, promotionPlan);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    /**
     * 根据预估计算费用
     *
     * @param promotionPlan
     * @param cacheKeySale
     * @param cacheKeyDelivery
     * @param cacheKeyGeneral
     * @return
     */
    @ApiOperation(value = "根据预估计算费用")
    @PostMapping("generateGeneralExpenses")
    public Result<Page<GeneralExpensesVo>> generateGeneralExpenses(
                            @ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                            @ApiParam(name = "promotionPlan", value = "促销规划实体类")@RequestBody PromotionPlanDto promotionPlan,
                            @ApiParam(name = "cacheKeySale", value = "当月销售缓存键")@RequestParam(required = false) String cacheKeySale,
                            @ApiParam(name = "cacheKeyDelivery", value = "当月送货缓存键")@RequestParam(required = false) String cacheKeyDelivery,
                            @ApiParam(name = "cacheKeyPurchase", value = "采销库存缓存键")@RequestParam(required = false) String cacheKeyPurchase,
                            @ApiParam(name = "cacheKeyGeneral", value = "费用申请缓存键")@RequestParam(required = false) String cacheKeyGeneral) {
        try {
            this.promotionPlanService.generateGeneralExpenses(promotionPlan, cacheKeySale, cacheKeyDelivery, cacheKeyPurchase, cacheKeyGeneral);
            Page<GeneralExpensesVo> page = this.generalExpensesService.findCachePageList(pageable, cacheKeyGeneral, null);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 提交审批流
     *
     * @param promotionPlanSubmitDto
     * @return
     */
    @ApiOperation(value = "提交审批流")
    @PostMapping("submitApproval")
    public Result<?> submitApproval(@ApiParam(name = "promotionPlanSubmitDto", value = "提交审批dto") @RequestBody PromotionPlanSubmitDto promotionPlanSubmitDto){
        String lockKey = "";
        boolean lockSuccess = false;
        try {
            Validate.notNull(promotionPlanSubmitDto.getId(), "请选择数据！");
            //提交审批前加锁一个小时
            lockKey = PromotionPlanConstant.LOCK_PROMOTION_PLAN_APPROVE + promotionPlanSubmitDto.getId();
            lockSuccess = redisLockService.tryLock(lockKey, TimeUnit.MINUTES, 3);
            Assert.isTrue(lockSuccess, "其他人正在操作数据,加锁失败,请稍后重试!");
            this.promotionPlanService.submitApproval(promotionPlanSubmitDto);
            return Result.ok();
        }catch (Exception e){
            log.error(e.getMessage(),e);
            return Result.error(e.getMessage());
        } finally {
            if (lockSuccess){
                redisLockService.unlock(lockKey);
            }
        }
    }

    /**
     * 保存并提交审批
     *
     * @param promotionPlanSubmitDto
     * @return
     */
    @ApiOperation(value = "保存并提交审批")
    @PostMapping("submitAndSetPass")
    public Result<?> submitAndSetPass(@ApiParam(name = "promotionPlanSubmitDto", value = "保存并提交审批dto") @RequestBody PromotionPlanSubmitDto promotionPlanSubmitDto,
                                      @ApiParam(name = "cacheKeySale", value = "当月销售缓存键")@RequestParam(required = false) String cacheKeySale,
                                      @ApiParam(name = "cacheKeyDelivery", value = "当月送货缓存键")@RequestParam(required = false) String cacheKeyDelivery,
                                      @ApiParam(name = "cacheKeyPurchase", value = "采销库存缓存键")@RequestParam(required = false) String cacheKeyPurchase,
                                      @ApiParam(name = "cacheKeyGeneral", value = "费用申请缓存键")@RequestParam(required = false) String cacheKeyGeneral,
                                      @ApiParam(name = "cacheKeyOther", value = "其他费用缓存键")@RequestParam(required = false) String cacheKeyOther){
        String lockKey = "";
        boolean lockSuccess = false;
        try {
            if (StringUtils.isNotBlank(promotionPlanSubmitDto.getId())) {
                //提交审批前加锁一个小时
                lockKey = PromotionPlanConstant.LOCK_PROMOTION_PLAN_APPROVE + promotionPlanSubmitDto.getId();
                lockSuccess = redisLockService.tryLock(lockKey, TimeUnit.MINUTES, 3);
                Assert.isTrue(lockSuccess, "其他人正在操作数据,加锁失败,请稍后重试!");
            }
            this.promotionPlanService.submitAndSetPass(promotionPlanSubmitDto, cacheKeySale, cacheKeyDelivery, cacheKeyPurchase, cacheKeyGeneral, cacheKeyOther);
            return Result.ok();
        } catch (Exception e){
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (lockSuccess){
                redisLockService.unlock(lockKey);
            }
        }
    }


    /**
     * 获取预算信息
     *
     * @param cacheKeyGeneral
     * @return
     */
    @ApiOperation(value = "获取预算信息")
    @GetMapping("getMonthBudget")
    public Result<List<MonthBudgetBusinessPolicyQueryVo>> getMonthBudget(@ApiParam(name = "cacheKeyGeneral", value = "费用申请缓存键")@RequestParam(required = false) String cacheKeyGeneral,
                                                                         @ApiParam(name = "cacheKeySale", value = "当月销售缓存键")@RequestParam(required = false) String cacheKeySale,
                                                                         @ApiParam(name = "promotionPlanCode", value = "促销规划编码")@RequestParam(required = false) String promotionPlanCode) {
        try {
            List<MonthBudgetBusinessPolicyQueryVo> monthBudgetVoList = this.promotionPlanService.getMonthBudget(cacheKeyGeneral, cacheKeySale, promotionPlanCode);
            return Result.ok(monthBudgetVoList);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 结果计算
     *
     * @param promotionPlan
     * @return
     */
    @ApiOperation(value = "结果计算")
    @PostMapping("calculationResult")
    public Result<List<PromotionPlanResultVoStr>> calculationResult(@ApiParam(name = "promotionPlan", value = "结果计算dto") @RequestBody PromotionPlanDto promotionPlan,
                                                                    @ApiParam(name = "cacheKeySale", value = "当月销售缓存键")@RequestParam(required = false) String cacheKeySale,
                                                                    @ApiParam(name = "cacheKeyDelivery", value = "当月送货缓存键")@RequestParam(required = false) String cacheKeyDelivery,
                                                                    @ApiParam(name = "cacheKeyPurchase", value = "采销库存缓存键")@RequestParam(required = false) String cacheKeyPurchase,
                                                                    @ApiParam(name = "cacheKeyGeneral", value = "费用申请缓存键")@RequestParam(required = false) String cacheKeyGeneral,
                                                                    @ApiParam(name = "cacheKeyOther", value = "其他费用缓存键")@RequestParam(required = false) String cacheKeyOther,
                                                                    @ApiParam(name = "type", value = "结果计算类型")@RequestParam String type) {
        try {
            List<PromotionPlanResultVoStr> resultVoList = this.promotionPlanService.calculationResult(promotionPlan, cacheKeySale, cacheKeyDelivery, cacheKeyPurchase, cacheKeyGeneral, cacheKeyOther, type);
            return Result.ok(resultVoList);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    /**
     * 当月销售计算折扣、销管报销、行政+人力
     *
     * @param promotionPlan
     * @return
     */
    @ApiOperation(value = "当月销售计算折扣、销管报销、行政+人力")
    @PostMapping("calculationCurrentMonthSaleFee")
    public Result<Page<CurrentMonthSaleVo>> calculationCurrentMonthSaleFee(
            @ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
            @ApiParam(name = "promotionPlan", value = "促销规划实体类") @RequestBody PromotionPlanDto promotionPlan,
            @ApiParam(name = "cacheKeySale", value = "当月销售缓存键") @RequestParam(required = false) String cacheKeySale,
            @ApiParam(name = "cacheKeyDelivery", value = "当月送货缓存键") @RequestParam(required = false) String cacheKeyDelivery,
            @ApiParam(name = "cacheKeyPurchase", value = "采销库存缓存键")@RequestParam(required = false) String cacheKeyPurchase,
            @ApiParam(name = "cacheKeyGeneral", value = "费用申请缓存键") @RequestParam(required = false) String cacheKeyGeneral,
            @ApiParam(name = "cacheKeyOther", value = "其他费用缓存键") @RequestParam(required = false) String cacheKeyOther) {
        try {
            this.promotionPlanService.calculationCurrentMonthSaleFee(promotionPlan, cacheKeySale, cacheKeyDelivery, cacheKeyPurchase, cacheKeyGeneral, cacheKeyOther);
            Page<CurrentMonthSaleVo> page = this.currentMonthSaleService.findCachePageList(pageable, cacheKeySale, promotionPlan.getPromotionPlanCode());
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    /**
     * 当月送货计算折扣、销管报销、行政+人力
     *
     * @param promotionPlan
     * @return
     */
    @ApiOperation(value = "当月送货计算折扣、销管报销、行政+人力")
    @PostMapping("calculationCurrentMonthDeliveryFee")
    public Result<Page<CurrentMonthDeliveryVo>> calculationCurrentMonthDeliveryFee(
            @ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
            @ApiParam(name = "promotionPlan", value = "促销规划实体类") @RequestBody PromotionPlanDto promotionPlan,
            @ApiParam(name = "cacheKeySale", value = "当月销售缓存键") @RequestParam(required = false) String cacheKeySale,
            @ApiParam(name = "cacheKeyDelivery", value = "当月送货缓存键") @RequestParam(required = false) String cacheKeyDelivery,
            @ApiParam(name = "cacheKeyPurchase", value = "采销库存缓存键")@RequestParam(required = false) String cacheKeyPurchase,
            @ApiParam(name = "cacheKeyGeneral", value = "费用申请缓存键") @RequestParam(required = false) String cacheKeyGeneral,
            @ApiParam(name = "cacheKeyOther", value = "其他费用缓存键") @RequestParam(required = false) String cacheKeyOther) {
        try {
            this.promotionPlanService.calculationCurrentMonthDeliveryFee(promotionPlan , cacheKeySale, cacheKeyPurchase, cacheKeyDelivery, cacheKeyGeneral, cacheKeyOther);
            Page<CurrentMonthDeliveryVo> page = this.currentMonthDeliveryService.findCachePageList(pageable, cacheKeyDelivery, promotionPlan.getPromotionPlanCode());
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 活动预付电商选择活动明细
     * @param pageable
     * @param dto
     * @return
     */
    @GetMapping("/findPromotionPlanPageListForEventPrepayment")
    @ApiOperation(value = "活动预付电商选择活动明细")
    public Result<Page<PromotionPlanForEventPrepaymentVo>> findPromotionPlanPageListForEventPrepayment(
            @ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
            @ApiParam(name = "promotionPlanForEventPrepaymentVo", value = "活细案查询Vo") PromotionPlanForEventPrepaymentDto dto){
        try {
            Page<PromotionPlanForEventPrepaymentVo> list = this.promotionPlanService.findPromotionPlanPageListForEventPrepayment(pageable, dto);
            return Result.ok(list);
        }catch (Exception e){
            e.printStackTrace();
            return Result.error(e.getMessage());
        }
    }

    /**
     * 费用进度监控查询
     *
     * @param pageable      分页对象
     * @param dto 查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "费用进度监控查询")
    @GetMapping("findCostProgressByConditions")
    public Result<Page<PromotionPlanCostProgressVo>> findCostProgressByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                          @ApiParam(name = "promotionPlan", value = "费用进度监控查询dto") PromotionPlanCostProgressDto dto) {
        try {
            Page<PromotionPlanCostProgressVo> page = this.promotionPlanCostProgressService.findCostProgressByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }
}
