package com.biz.crm.tpm.business.promotion.plan.local.controller;

import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.tpm.business.promotion.plan.sdk.constant.PromotionPlanConstant;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.PromotionPlanWholeSummaryDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.PromotionPlanWholeSummaryService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.concurrent.TimeUnit;

/**
 * @author: yaoyongming
 * @date: 2023/9/11 18:27
 */
@RestController
@RequestMapping("/v1/promotionPlanWholeSummary")
@Slf4j
@Api(tags = "促销规划整体一览表")
public class PromotionPlanWholeSummaryController {

    @Autowired(required = false)
    private PromotionPlanWholeSummaryService promotionPlanWholeSummaryService;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    /**
     * 重新生成促销规划整体一览表
     *
     * @param yearMonthLy
     * @return
     */
    @ApiOperation(value = "重新生成促销规划整体一览表")
    @GetMapping("createPromotionPlanWholeSummary")
    public Result<?> createPromotionPlanWholeSummary(@ApiParam(name = "yearMonthLy", value = "年月") @RequestParam("yearMonthLy") String yearMonthLy){
        String lockKey = "";
        boolean lockSuccess = false;
        try {
            lockKey = PromotionPlanConstant.LOCK_PROMOTION_PLAN_WHOLE_SUMMARY_APPROVE + yearMonthLy;
            lockSuccess = redisLockService.tryLock(lockKey, TimeUnit.MINUTES, 10);
            if (!lockSuccess) {
                throw new RuntimeException("数据处理中，请稍后");
            }
            this.promotionPlanWholeSummaryService.createPromotionPlanWholeSummary(yearMonthLy);
            return Result.ok();
        }catch (Exception e){
            log.error(e.getMessage(),e);
            return Result.error(e.getMessage());
        } finally {
            if (lockSuccess){
                redisLockService.unlock(lockKey);
            }
        }
    }

    /**
     * 根据年月获取一览表明细
     *
     * @param yearMonthLy
     * @return
     */
    @ApiOperation(value = "根据年月获取一览表明细")
    @GetMapping("getPromotionPlanWholeSummaryDetail")
    public Result<?> getPromotionPlanWholeSummaryDetail(@ApiParam(name = "yearMonthLy", value = "年月") @RequestParam("yearMonthLy") String yearMonthLy){
        try {
            return Result.ok(this.promotionPlanWholeSummaryService.getPromotionPlanWholeSummaryDetail(yearMonthLy));
        }catch (Exception e){
            log.error(e.getMessage(),e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 提交审批流
     *
     * @param dto
     * @return
     */
    @ApiOperation(value = "提交审批流")
    @PostMapping("submitApproval")
    public Result<?> submitApproval(@ApiParam(name = "PromotionPlanWholeSummaryDto", value = "提交审批dto") @RequestBody PromotionPlanWholeSummaryDto dto){
        String lockKey = "";
        boolean lockSuccess = false;
        try {
            Validate.notNull(dto.getId(), "请选择数据！");
            //提交审批前加锁一个小时
            lockKey = PromotionPlanConstant.LOCK_PROMOTION_PLAN_WHOLE_SUMMARY_APPROVE + dto.getId();
            lockSuccess = redisLockService.tryLock(lockKey, TimeUnit.HOURS, 1);
            if (!lockSuccess) {
                throw new RuntimeException("数据处理中，请稍后");
            }
            this.promotionPlanWholeSummaryService.submitApproval(dto);
            return Result.ok();
        }catch (Exception e){
            log.error(e.getMessage(),e);
            return Result.error(e.getMessage());
        } finally {
            if (lockSuccess){
                redisLockService.unlock(lockKey);
            }
        }
    }
}
