package com.biz.crm.tpm.business.promotion.plan.local.repository;


import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mn.common.base.dto.audit.AutoAuditParamsDto;
import com.biz.crm.tpm.business.month.budget.sdk.eunm.YesOrNoEnum;
import com.biz.crm.tpm.business.promotion.plan.local.entity.GeneralExpensesEntity;
import com.biz.crm.tpm.business.promotion.plan.local.mapper.GeneralExpensesMapper;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.GeneralExpensesDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.PromotionPlanCostProgressDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.PromotionPlanDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.vo.GeneralExpensesVo;
import com.biz.crm.tpm.business.promotion.plan.sdk.vo.PromotionPlanCostProgressVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;


/**
 * 费用申请实体类(GeneralExpenses)表数据库访问层
 *
 * @author yaoyongming
 * @since 2022-11-03 16:02:55
 */
@Component
public class GeneralExpensesRepository extends ServiceImpl<GeneralExpensesMapper, GeneralExpensesEntity> {

    @Autowired(required = false)
    private GeneralExpensesMapper generalExpensesMapper;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    /**
     * 分页查询数据
     *
     * @param pageable        分页对象
     * @param generalExpenses 实体对象
     * @return
     */
    public Page<GeneralExpensesVo> findByConditions(Pageable pageable, GeneralExpensesDto generalExpenses) {
        Page<GeneralExpensesVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        Page<GeneralExpensesVo> pageList = this.generalExpensesMapper.findByConditions(page, generalExpenses);
        return pageList;
    }

    /**
     * 分页查询数据
     *
     * @param pageable        分页对象
     * @param promotionPlan 实体对象
     * @return
     */
    public Page<GeneralExpensesVo> findByConditionsGeneral(Pageable pageable, PromotionPlanDto promotionPlan) {
        Page<GeneralExpensesVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        Page<GeneralExpensesVo> pageList = this.generalExpensesMapper.findByConditionsGeneral(page, promotionPlan);
        return pageList;
    }

    /**
     * 根据PromotionPlanCode查询
     *
     * @param code
     * @return
     */
    public List<GeneralExpensesDto> findByPlanCode(String code) {
        List<GeneralExpensesEntity> itemList = this.lambdaQuery().eq(GeneralExpensesEntity::getPromotionPlanCode, code)
                .eq(GeneralExpensesEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode()).list();
        return (List<GeneralExpensesDto>) nebulaToolkitService.copyCollectionByWhiteList(itemList, GeneralExpensesEntity.class, GeneralExpensesDto.class, HashSet.class, ArrayList.class);
    }

    /**
     *
     *
     * @param codes
     * @return
     */
    public List<GeneralExpensesDto> findByPlanDetailCodes(List<String> codes) {
        List<GeneralExpensesEntity> itemList = this.lambdaQuery().in(GeneralExpensesEntity::getExpensesCode, codes)
                .eq(GeneralExpensesEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode()).list();
        return (List<GeneralExpensesDto>) nebulaToolkitService.copyCollectionByWhiteList(itemList, GeneralExpensesEntity.class, GeneralExpensesDto.class, HashSet.class, ArrayList.class);
    }

    /**
     * 根据expensesCode查询
     *
     * @param expensesCode
     * @return
     */
    public List<GeneralExpensesEntity> findByExpensesCode(String expensesCode) {
        return this.lambdaQuery().eq(GeneralExpensesEntity::getExpensesCode, expensesCode).list();
    }

    /**
     * 根据expensesCode查询Ext
     *
     * @param expensesCodeList
     * @return
     */
    public List<GeneralExpensesVo> findByExpensesCodeExt(List<String> expensesCodeList) {
        return this.generalExpensesMapper.findByExpensesCode(expensesCodeList);
    }

    /**
     * 根据PromotionPlanCode删除
     *
     * @param code
     */
    public void deleteByPromotionPlanCode(String code) {
        this.lambdaUpdate().eq(GeneralExpensesEntity::getPromotionPlanCode, code).remove();
    }

    /**
     * 根据PromotionPlanCodeList逻辑删除
     *
     * @param codeList
     */
    public void removeByPromotionPlanCodeList(List<String> codeList) {
        this.lambdaUpdate().in(GeneralExpensesEntity::getPromotionPlanCode, codeList).set(GeneralExpensesEntity::getDelFlag, DelFlagStatusEnum.DELETE.getCode()).update();
    }

    /**
     * 更新结案金额
     *
     * @param dtoList
     */
    public void updateAuditAmount(List<GeneralExpensesDto> dtoList) {
        this.generalExpensesMapper.updateAuditAmount(dtoList);
    }

    public Page<GeneralExpensesVo> findForWithholding(Page<GeneralExpensesVo> page, String processStatus, String upAccountStatus) {
        return this.generalExpensesMapper.findForWithholding(page, processStatus, upAccountStatus);
    }

    public BigDecimal getContractAmount(String promotionPlanCode){
        return this.generalExpensesMapper.getContractAmount(promotionPlanCode);
    }

    public int getTotalNum(PromotionPlanDto promotionPlanDto){

        Integer total = this.generalExpensesMapper.getGeneralExpenseTotal(promotionPlanDto);

        return total == null ? 0 : total;
    }

    public List<GeneralExpensesEntity> findListByExpensesCodes(Set<String> activityDetailCodes) {

        if(CollectionUtils.isEmpty(activityDetailCodes)){
            return Lists.newArrayList();
        }
        List<GeneralExpensesEntity> list = this.lambdaQuery().in(GeneralExpensesEntity::getExpensesCode, activityDetailCodes)
                .eq(GeneralExpensesEntity::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
                .list();

        return list;
    }

    /**
     * 查询商务政策关联的促销规划是否完全核销
     * @param businessPolicyCode
     * @return
     */
    public String findOtherDetailWholeAudit(String businessPolicyCode) {
        if(StringUtils.isEmpty(businessPolicyCode)){
            return YesOrNoEnum.YES.getCode();
        }
        List<GeneralExpensesEntity> list = this.lambdaQuery().eq(GeneralExpensesEntity::getCommercePolicyCode, businessPolicyCode).eq(GeneralExpensesEntity::getDelFlag,DelFlagStatusEnum.NORMAL.getCode()).list();
        for (GeneralExpensesEntity generalExpensesEntity : list) {
            if(!YesOrNoEnum.YES.getCode().equals(generalExpensesEntity.getWholeAudit())){
                return YesOrNoEnum.NO.getCode();
            }
        }
        return YesOrNoEnum.YES.getCode();
    }

    /**
     * 费用进度监控条件查询
     *
     * @param pageable
     * @param dto
     * @return
     */
    public Page<PromotionPlanCostProgressVo> findCostProgressByConditions(Pageable pageable, PromotionPlanCostProgressDto dto) {
        Page<PromotionPlanCostProgressVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        Page<PromotionPlanCostProgressVo> pageList = this.generalExpensesMapper.findCostProgressByConditions(page, dto);
        return pageList;
    }

    /**
     * 根据商务政策查询
     *
     * @param businessPolicyCodeList
     * @return
     */
    public List<GeneralExpensesVo> findByBusinessPolicyCode(List<String> businessPolicyCodeList) {
        return this.generalExpensesMapper.findByBusinessPolicyCode(businessPolicyCodeList);
    }

    public List<GeneralExpensesVo> findDataByExpensesCodes(Set<String> set) {
        return this.generalExpensesMapper.findDataByExpensesCodes(set);
    }

    public List<GeneralExpensesVo> findByExpensesCodes(Set<String> detailCodes) {
        if(CollectionUtils.isEmpty(detailCodes)){
            return Lists.newArrayList();
        }
        return this.generalExpensesMapper.findByExpensesCodes(detailCodes);
    }

    public boolean isOtherDetailWholeAudit2(String businessPolicyCode, List<String> expensesCodes) {
        if(StringUtils.isEmpty(businessPolicyCode)){
            return false;
        }
        Integer count = this.lambdaQuery().eq(GeneralExpensesEntity::getCommercePolicyCode, businessPolicyCode)
                .eq(GeneralExpensesEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .ne(GeneralExpensesEntity::getWholeAudit, YesOrNoEnum.YES.getCode())
                .notIn(!CollectionUtils.isEmpty(expensesCodes), GeneralExpensesEntity::getExpensesCode, expensesCodes)
                .count();
        if(count>0){
            return false;
        }
        return true;
    }

    public Page<GeneralExpensesVo> findCanAutoAuditItemPage(Page<GeneralExpensesVo> page, AutoAuditParamsDto autoAuditParams) {

        Page<GeneralExpensesVo> itemPage = this.getBaseMapper().findCanAutoAuditItemPage(page, autoAuditParams);
        List<GeneralExpensesVo> itemList = itemPage.getRecords();
        if (CollUtil.isEmpty(itemList)){
            return itemPage;
        }

        String endCaseForm = autoAuditParams.getEndCaseForm();
        List<String> endCaseFormList = null;
        if (CharSequenceUtil.isNotEmpty(endCaseForm)){
            endCaseFormList = Arrays.asList(endCaseForm.split(","));
        }

        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM");

        List<GeneralExpensesVo> resultList = new ArrayList<>();

        for (GeneralExpensesVo item : itemList) {
            //根据结案形式进行过滤
            if (CollUtil.isNotEmpty(endCaseFormList)){
                String auditForm = item.getAuditWayCode();
                if (CharSequenceUtil.isNotEmpty(autoAuditParams.getEndCaseForm())) {
                    continue;
                }
                List<String> list = Arrays.asList(auditForm.split(","));
                List<String> filterEndCaseFormList = com.google.common.collect.Lists.newArrayList();
                for (String s : list) {
                    if (endCaseFormList.contains(s)){
                        filterEndCaseFormList.add(s);
                    }
                }
                if (CollUtil.isEmpty(filterEndCaseFormList)){
                    continue;
                }
            }

            resultList.add(item);
        }
        itemPage.setRecords(resultList);

        return itemPage;
    }

    public void updateAutoAuditFlag(List<String> codes, String autoAudit) {
        if(CollectionUtils.isEmpty(codes)){
            return;
        }
        List<List<String>> partition = com.google.common.collect.Lists.partition(codes, 800);
        partition.forEach(list -> this.baseMapper.updateAutoAuditFlag(list, autoAudit));
    }
}

