package com.biz.crm.tpm.business.promotion.plan.local.service.internal;


import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.tpm.business.promotion.plan.local.repository.CurrentMonthDeliveryRepository;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.CurrentMonthDeliveryDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.CurrentMonthDeliveryService;
import com.biz.crm.tpm.business.promotion.plan.sdk.vo.CurrentMonthDeliveryVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.beans.factory.annotation.Autowired;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.Pageable;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.biz.crm.tpm.business.promotion.plan.sdk.constant.PromotionPlanConstant.*;

/**
 * 当月送货实体类(CurrentMonthDeliveryEntity)表服务实现类
 *
 * @author yaoyongming
 * @since 2022-11-02 18:14:36
 */
@Slf4j
@Service("currentMonthDeliveryService")
public class CurrentMonthDeliveryServiceImpl implements CurrentMonthDeliveryService {

    @Autowired(required = false)
    private CurrentMonthDeliveryRepository currentMonthDeliveryRepository;

    @Autowired(required = false)
    private RedisService redisService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    /**
     * 分页查询数据
     *
     * @param pageable             分页对象
     * @param currentMonthDelivery 实体对象
     * @return
     */
    @Override
    public Page<CurrentMonthDeliveryVo> findByConditions(Pageable pageable, CurrentMonthDeliveryDto currentMonthDelivery) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(currentMonthDelivery)) {
            currentMonthDelivery = new CurrentMonthDeliveryDto();
        }
        return this.currentMonthDeliveryRepository.findByConditions(pageable, currentMonthDelivery);
    }

    /**
     * 1、获取明细列表分页接口
     *
     * @param pageable          分页参数
     * @param cacheKey          缓存key
     * @param promotionPlanCode 促销规划编码
     * @return
     */
    @Override
    public Page<CurrentMonthDeliveryVo> findCachePageList(Pageable pageable, String cacheKey, String promotionPlanCode) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        Page<CurrentMonthDeliveryVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        page.setTotal(0);
        page.setRecords(Lists.newArrayList());

        if (redisService.hasKey(redisCacheKey)) {
            //redis里面有的话直接从redis里面取
            Long total = redisService.lSize(redisCacheKey);
            page.setTotal(total);
            List<Object> objects = redisService.lRange(redisCacheKey, page.offset(), page.offset() + page.getSize() - 1);
            if (!CollectionUtils.isEmpty(objects)) {
                List<CurrentMonthDeliveryVo> voList = (List<CurrentMonthDeliveryVo>) nebulaToolkitService.copyCollectionByBlankList((List<CurrentMonthDeliveryDto>) (List) objects, CurrentMonthDeliveryDto.class, CurrentMonthDeliveryVo.class, HashSet.class, ArrayList.class);
                page.setRecords(voList);
            }
        } else {
            //redis里面没有
            if (StringUtils.isNotEmpty(promotionPlanCode)) {
                //从数据库里面查出来放到缓存里面
                List<CurrentMonthDeliveryDto> dtoList = currentMonthDeliveryRepository.findByPlanCode(promotionPlanCode);
                if (!CollectionUtils.isEmpty(dtoList)) {
                    redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, dtoList.toArray());
                }
                page.setTotal(dtoList.size());
                long start = page.offset();
                if (page.getTotal() > start) {
                    long end = page.offset() + page.getSize();
                    if (page.getTotal() < end) {
                        end = page.getTotal();
                    }
                    List<CurrentMonthDeliveryVo> voList = (List<CurrentMonthDeliveryVo>) nebulaToolkitService.copyCollectionByBlankList(dtoList.subList((int) page.offset(), (int) end), CurrentMonthDeliveryDto.class, CurrentMonthDeliveryVo.class, HashSet.class, ArrayList.class);
                    page.setRecords(voList);
                }
            }
        }
        return page;
    }

    /**
     * 获取缓存中的所有明细行
     *
     * @param cacheKey 缓存key
     */
    @Override
    public List<CurrentMonthDeliveryDto> findCacheList(String cacheKey) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        if (!redisService.hasKey(redisCacheKey)) {
            return Lists.newArrayList();
        }
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);
        return (List<CurrentMonthDeliveryDto>) (List) objects;
    }

    /**
     * 2、新增一行接口，在缓存中行首插入一条数据
     *
     * @param cacheKey 缓存key
     * @param saveList 要保存的当前页数据
     */
    @Override
    public void addItemCache(String cacheKey, List<CurrentMonthDeliveryDto> saveList) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<CurrentMonthDeliveryDto> newList = Lists.newArrayList();
        Map<String, CurrentMonthDeliveryDto> saveItemMap;
        if (!CollectionUtils.isEmpty(saveList)) {
            saveItemMap = saveList.stream().collect(Collectors.toMap(CurrentMonthDeliveryDto::getId, Function.identity()));
        } else {
            saveItemMap = Maps.newHashMap();
        }
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);
        if (!CollectionUtils.isEmpty(objects)) {
            for (Object object : objects) {
                CurrentMonthDeliveryDto item = (CurrentMonthDeliveryDto) object;
                newList.add(saveItemMap.getOrDefault(item.getId(), item));
            }
        }

        CurrentMonthDeliveryDto newItem = new CurrentMonthDeliveryDto();
        newItem.setId(UUID.randomUUID().toString().replace("-", ""));
        newItem.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        newItem.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        newList.add(0, newItem);
        redisService.del(redisCacheKey);//先全部删掉再放进去吧
        redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());
    }

    /**
     * 3、复制行接口，保存当前页数据后，在缓存中行首复制选中数据并返回第一页数据
     *
     * @param cacheKey 缓存key
     * @param saveList 要保存的当前页数据
     */
    @Override
    public void copyItemListCache(String cacheKey, List<CurrentMonthDeliveryDto> saveList) {
        if (CollectionUtils.isEmpty(saveList)) {
            return;
        }
        Map<String, CurrentMonthDeliveryDto> saveItemMap = saveList.stream().collect(Collectors.toMap(CurrentMonthDeliveryDto::getId, Function.identity()));
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);

        List<CurrentMonthDeliveryDto> newList = Lists.newArrayList();
        List<CurrentMonthDeliveryDto> copyList = Lists.newArrayList();
        for (Object object : objects) {
            CurrentMonthDeliveryDto item = (CurrentMonthDeliveryDto) object;
            if (saveItemMap.containsKey(item.getId())) {
                CurrentMonthDeliveryDto newItem = saveItemMap.get(item.getId());
                if (BooleanEnum.TRUE.getNumStr().equals(newItem.getChecked())) {
                    //选中状态，代表该行数据要复制
                    CurrentMonthDeliveryDto copyItem = nebulaToolkitService.copyObjectByWhiteList(newItem, CurrentMonthDeliveryDto.class, HashSet.class, ArrayList.class);
                    copyItem.setId(UUID.randomUUID().toString().replace("-", ""));
                    copyItem.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                    copyItem.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                    copyList.add(copyItem);
                }
                newList.add(newItem);
            } else {
                newList.add(item);
            }
        }
        newList.addAll(0, copyList);
        redisService.del(redisCacheKey);//先全部删掉再放进去吧
        redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());
    }

    /**
     * 4、保存当前页数据到缓存并返回指定页数据接口
     *
     * @param cacheKey 缓存key
     * @param saveList
     */
    @Override
    public void saveCurrentPageCache(String cacheKey, List<CurrentMonthDeliveryDto> saveList) {
        if (CollectionUtils.isEmpty(saveList)) {
            return;
        }
        Map<String, CurrentMonthDeliveryDto> saveItemMap = saveList.stream().collect(Collectors.toMap(CurrentMonthDeliveryDto::getId, Function.identity()));
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);

        List<CurrentMonthDeliveryDto> newList = Lists.newArrayList();
        for (Object object : objects) {
            CurrentMonthDeliveryDto item = (CurrentMonthDeliveryDto) object;
            newList.add(saveItemMap.getOrDefault(item.getId(), item));
        }
        redisService.del(redisCacheKey);//删掉重新放
        if (!CollectionUtils.isEmpty(newList)) {
            redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());//设置过期时间1天
        }
    }

    /**
     * 7、多行删除并保存当前页数据并返回指定页数据接口
     *
     * @param cacheKey 缓存key
     * @param itemList
     */
    @Override
    public void deleteCacheList(String cacheKey, List<CurrentMonthDeliveryDto> itemList) {
        if (CollectionUtils.isEmpty(itemList)) {
            return;
        }
        Map<String, CurrentMonthDeliveryDto> saveItemMap = itemList.stream().collect(Collectors.toMap(CurrentMonthDeliveryDto::getId, Function.identity()));
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);

        List<CurrentMonthDeliveryDto> newList = Lists.newArrayList();
        for (Object object : objects) {
            CurrentMonthDeliveryDto item = (CurrentMonthDeliveryDto) object;
            if (saveItemMap.containsKey(item.getId())) {
                CurrentMonthDeliveryDto newItem = saveItemMap.get(item.getId());
                if (!BooleanEnum.TRUE.getNumStr().equals(newItem.getChecked())) {
                    //选中状态，代表该行数据要删除,非选中则该行数据保留并更新
                    newList.add(newItem);
                }
            } else {
                newList.add(item);
            }
        }
        redisService.del(redisCacheKey);//先全部删掉再放进去吧
        if (!CollectionUtils.isEmpty(newList)) {
            redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());
        }
    }

    /**
     * 6、清理缓存接口
     *
     * @param cacheKey 缓存key
     */
    @Override
    public void clearCache(String cacheKey) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        redisService.del(redisCacheKey);
    }

    /**
     * 获取总条数
     *
     * @param cacheKey
     * @return
     */
    @Override
    public Integer getTotal(String cacheKey) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        return redisService.lSize(redisCacheKey).intValue();
    }

    /**
     * 在缓存中加入多条数据
     *
     * @param cacheKey
     * @param itemList
     */
    @Override
    public void addListCache(String cacheKey, List<CurrentMonthDeliveryDto> itemList) {
        if (CollectionUtils.isEmpty(itemList)) {
            return;
        }
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<CurrentMonthDeliveryDto> newList = Lists.newArrayList();
        Map<String, CurrentMonthDeliveryDto> saveItemMap = itemList.stream().collect(Collectors.toMap(CurrentMonthDeliveryDto::getId, Function.identity()));
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);
        if (!CollectionUtils.isEmpty(objects)) {
            newList.addAll((List<CurrentMonthDeliveryDto>) (List) objects);
        }
        newList.addAll(itemList);
        redisService.del(redisCacheKey);
        redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());
    }

    /**
     * 自动计算费用
     *
     * @param dto
     */
    @Override
    public void calculationFee(CurrentMonthDeliveryDto dto, String platformName) {
        //预计销量（吨）：自动计算：预计销量（箱）*系数*组合数量/1000。
        dto.setEstimatedSalesTon(dto.getEstimatedSalesBox().multiply(dto.getRatio()).multiply(dto.getCombinationQuantity()).setScale(2, RoundingMode.HALF_UP).divide(new BigDecimal(1000), 6, BigDecimal.ROUND_HALF_UP));
        //预计销售额（元）：自动计算：组合数量*平台供货价*预计销量（箱）
        dto.setEstimatedAmount(dto.getCombinationQuantity().multiply(dto.getPlatformSupplyPrice()).multiply(dto.getEstimatedSalesBox()).setScale(2, RoundingMode.HALF_UP));
        //增值税（元）：自动计算：预计销售额/（1+税率）*税率
        dto.setVat(dto.getEstimatedAmount().divide(BigDecimal.ONE.add(dto.getTaxRate()), 6, BigDecimal.ROUND_HALF_UP).multiply(dto.getTaxRate()).setScale(2, RoundingMode.HALF_UP));
        //净收入（元）	预计折前销售额-增值税-折扣+折扣/（1+税率）*税率
        try {
            dto.setNetIncome((dto.getEstimatedAmount().subtract(dto.getVat()).subtract(dto.getDiscount()))
                    .add(dto.getDiscount()
                            .divide((BigDecimal.ONE.add(dto.getTaxRate())), 6, BigDecimal.ROUND_HALF_UP).multiply(dto.getTaxRate())).setScale(2, RoundingMode.HALF_UP));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            dto.setNetIncome(BigDecimal.ZERO);
        }
        //毛利（元）	净收入-成本价*组合数量*预计销量（箱）/（1+税率）
        try {
            dto.setGrossProfit(dto.getNetIncome().subtract(
                    (
                            dto.getCostPrice().multiply(dto.getEstimatedSalesBox()).multiply(dto.getCombinationQuantity())
                                    .divide((BigDecimal.ONE.add(dto.getTaxRate())), 2, RoundingMode.HALF_UP)
                    )).setScale(2, RoundingMode.HALF_UP));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            dto.setGrossProfit(BigDecimal.ZERO);
        }
        //经营利润（元）	净收入-净收入*0.5%-成本价*组合数量*预计销量（箱）/（1+税率）-销管报销-物流费用-（行政+人力）+其他收益
        try {
            dto.setOperateProfit(dto.getNetIncome().subtract(
                    dto.getNetIncome().multiply(new BigDecimal("0.005"))
            ).subtract(
                    dto.getCostPrice().multiply(dto.getEstimatedSalesBox()).multiply(dto.getCombinationQuantity())
                            .divide((BigDecimal.ONE.add(dto.getTaxRate())), 2, RoundingMode.HALF_UP)
            ).subtract(dto.getSaleReimburse()).subtract(dto.getLogistics()).subtract(
                    dto.getAdminHuman()
            ).setScale(2, RoundingMode.HALF_UP));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            dto.setOperateProfit(BigDecimal.ZERO);
        }
        //所得税（元）	经营利润*0.25
        try {
            dto.setIncomeTax(dto.getOperateProfit().multiply(new BigDecimal("0.25")).setScale(2, RoundingMode.HALF_UP));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            dto.setIncomeTax(BigDecimal.ZERO);
        }
        //净利润（元）	经营利润-所得税
        try {
            dto.setNetProfit(dto.getOperateProfit().subtract(dto.getIncomeTax()).setScale(2, RoundingMode.HALF_UP));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            dto.setNetProfit(BigDecimal.ZERO);
        }
        //净利率	净利润/净收入
        try {
            dto.setNetProfitRate(dto.getNetProfit().divide(dto.getNetIncome(), 2, RoundingMode.HALF_UP));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            dto.setNetProfitRate(BigDecimal.ZERO);
        }
    }

    /**
     * 获取redis缓存key
     *
     * @param cacheKey 前端传过来的key
     * @return redis的缓存key
     */
    private String getRedisCacheKey(String cacheKey) {
        return MONTH_DELIVERY_CACHE_KEY_PREFIX + cacheKey;
    }
}

