package com.biz.crm.tpm.business.promotion.plan.local.service.internal;


import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.tpm.business.promotion.plan.local.repository.CurrentMonthSaleRepository;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.CurrentMonthSaleDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.CurrentMonthSaleService;
import com.biz.crm.tpm.business.promotion.plan.sdk.vo.CurrentMonthSaleVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.beans.factory.annotation.Autowired;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.Pageable;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.biz.crm.tpm.business.promotion.plan.sdk.constant.PromotionPlanConstant.*;

/**
 * 当月销售实体类(CurrentMonthSaleEntity)表服务实现类
 *
 * @author yaoyongming
 * @since 2022-11-02 18:14:37
 */
@Slf4j
@Service("currentMonthSaleService")
public class CurrentMonthSaleServiceImpl implements CurrentMonthSaleService {

    @Autowired(required = false)
    private CurrentMonthSaleRepository currentMonthSaleRepository;

    @Autowired(required = false)
    private RedisService redisService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    /**
     * 分页查询数据
     *
     * @param pageable         分页对象
     * @param currentMonthSale 实体对象
     * @return
     */
    @Override
    public Page<CurrentMonthSaleVo> findByConditions(Pageable pageable, CurrentMonthSaleDto currentMonthSale) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(currentMonthSale)) {
            currentMonthSale = new CurrentMonthSaleDto();
        }
        return this.currentMonthSaleRepository.findByConditions(pageable, currentMonthSale);
    }

    /**
     * 1、获取明细列表分页接口
     *
     * @param pageable          分页参数
     * @param cacheKey          缓存key
     * @param promotionPlanCode 促销规划编码
     * @return
     */
    @Override
    public Page<CurrentMonthSaleVo> findCachePageList(Pageable pageable, String cacheKey, String promotionPlanCode) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        Page<CurrentMonthSaleVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        page.setTotal(0);
        page.setRecords(Lists.newArrayList());

        if (redisService.hasKey(redisCacheKey)) {
            //redis里面有的话直接从redis里面取
            Long total = redisService.lSize(redisCacheKey);
            page.setTotal(total);
            List<Object> objects = redisService.lRange(redisCacheKey, page.offset(), page.offset() + page.getSize() - 1);
            if (!CollectionUtils.isEmpty(objects)) {
                List<CurrentMonthSaleVo> voList = (List<CurrentMonthSaleVo>) nebulaToolkitService.copyCollectionByBlankList((List<CurrentMonthSaleDto>) (List) objects, CurrentMonthSaleDto.class, CurrentMonthSaleVo.class, HashSet.class, ArrayList.class);
                page.setRecords(voList);
            }
        } else {
            //redis里面没有
            if (StringUtils.isNotEmpty(promotionPlanCode)) {
                //从数据库里面查出来放到缓存里面
                List<CurrentMonthSaleDto> dtoList = currentMonthSaleRepository.findByPlanCode(promotionPlanCode);
                if (!CollectionUtils.isEmpty(dtoList)) {
                    redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, dtoList.toArray());
                }
                page.setTotal(dtoList.size());
                long start = page.offset();
                if (page.getTotal() > start) {
                    long end = page.offset() + page.getSize();
                    if (page.getTotal() < end) {
                        end = page.getTotal();
                    }
                    List<CurrentMonthSaleVo> voList = (List<CurrentMonthSaleVo>) nebulaToolkitService.copyCollectionByBlankList(dtoList.subList((int) page.offset(), (int) end), CurrentMonthSaleDto.class, CurrentMonthSaleVo.class, HashSet.class, ArrayList.class);
                    page.setRecords(voList);
                }
            }
        }
        return page;
    }

    /**
     * 获取缓存中的所有明细行
     *
     * @param cacheKey 缓存key
     */
    @Override
    public List<CurrentMonthSaleDto> findCacheList(String cacheKey) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        if (!redisService.hasKey(redisCacheKey)) {
            return Lists.newArrayList();
        }
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);
        return (List<CurrentMonthSaleDto>) (List) objects;
    }

    /**
     * 获取所有明细行
     *
     * @param code 促销规划编码
     */
    @Override
    public List<CurrentMonthSaleDto> findList(String code) {
        return this.currentMonthSaleRepository.findByPlanCode(code);
    }

    /**
     * 2、新增一行接口，在缓存中行首插入一条数据
     *
     * @param cacheKey 缓存key
     * @param saveList 要保存的当前页数据
     */
    @Override
    public void addItemCache(String cacheKey, List<CurrentMonthSaleDto> saveList) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<CurrentMonthSaleDto> newList = Lists.newArrayList();
        Map<String, CurrentMonthSaleDto> saveItemMap;
        if (!CollectionUtils.isEmpty(saveList)) {
            saveItemMap = saveList.stream().collect(Collectors.toMap(CurrentMonthSaleDto::getId, Function.identity()));
        } else {
            saveItemMap = Maps.newHashMap();
        }
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);
        if (!CollectionUtils.isEmpty(objects)) {
            for (Object object : objects) {
                CurrentMonthSaleDto item = (CurrentMonthSaleDto) object;
                newList.add(saveItemMap.getOrDefault(item.getId(), item));
            }
        }

        CurrentMonthSaleDto newItem = new CurrentMonthSaleDto();
        newItem.setId(UUID.randomUUID().toString().replace("-", ""));
        newItem.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        newItem.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        newList.add(0, newItem);
        redisService.del(redisCacheKey);//先全部删掉再放进去吧
        if (!CollectionUtils.isEmpty(newList)) {
            redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());
        }
    }

    /**
     * 3、复制行接口，保存当前页数据后，在缓存中行首复制选中数据并返回第一页数据
     *
     * @param cacheKey 缓存key
     * @param saveList 要保存的当前页数据
     */
    @Override
    public void copyItemListCache(String cacheKey, List<CurrentMonthSaleDto> saveList) {
        if (CollectionUtils.isEmpty(saveList)) {
            return;
        }
        Map<String, CurrentMonthSaleDto> saveItemMap = saveList.stream().collect(Collectors.toMap(CurrentMonthSaleDto::getId, Function.identity()));
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);

        List<CurrentMonthSaleDto> newList = Lists.newArrayList();
        List<CurrentMonthSaleDto> copyList = Lists.newArrayList();
        for (Object object : objects) {
            CurrentMonthSaleDto item = (CurrentMonthSaleDto) object;
            if (saveItemMap.containsKey(item.getId())) {
                CurrentMonthSaleDto newItem = saveItemMap.get(item.getId());
                if (BooleanEnum.TRUE.getNumStr().equals(newItem.getChecked())) {
                    //选中状态，代表该行数据要复制
                    CurrentMonthSaleDto copyItem = nebulaToolkitService.copyObjectByWhiteList(newItem, CurrentMonthSaleDto.class, HashSet.class, ArrayList.class);
                    copyItem.setId(UUID.randomUUID().toString().replace("-", ""));
                    copyItem.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                    copyItem.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                    copyList.add(copyItem);
                }
                newList.add(newItem);
            } else {
                newList.add(item);
            }
        }
        newList.addAll(0, copyList);
        redisService.del(redisCacheKey);//先全部删掉再放进去吧
        if (!CollectionUtils.isEmpty(newList)) {
            redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());
        }
    }

    /**
     * 4、保存当前页数据到缓存并返回指定页数据接口
     *
     * @param cacheKey 缓存key
     * @param saveList
     */
    @Override
    public void saveCurrentPageCache(String cacheKey, List<CurrentMonthSaleDto> saveList) {
        if (CollectionUtils.isEmpty(saveList)) {
            return;
        }
        Map<String, CurrentMonthSaleDto> saveItemMap = saveList.stream().collect(Collectors.toMap(CurrentMonthSaleDto::getId, Function.identity()));
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);

        List<CurrentMonthSaleDto> newList = Lists.newArrayList();
        for (Object object : objects) {
            CurrentMonthSaleDto item = (CurrentMonthSaleDto) object;
            newList.add(saveItemMap.getOrDefault(item.getId(), item));
        }
        redisService.del(redisCacheKey);//删掉重新放
        if (!CollectionUtils.isEmpty(newList)) {
            redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());//设置过期时间1天
        }
    }

    /**
     * 7、多行删除并保存当前页数据并返回指定页数据接口
     *
     * @param cacheKey 缓存key
     * @param itemList
     */
    @Override
    public void deleteCacheList(String cacheKey, List<CurrentMonthSaleDto> itemList) {
        if (CollectionUtils.isEmpty(itemList)) {
            return;
        }
        Map<String, CurrentMonthSaleDto> saveItemMap = itemList.stream().collect(Collectors.toMap(CurrentMonthSaleDto::getId, Function.identity()));
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);

        List<CurrentMonthSaleDto> newList = Lists.newArrayList();
        for (Object object : objects) {
            CurrentMonthSaleDto item = (CurrentMonthSaleDto) object;
            if (saveItemMap.containsKey(item.getId())) {
                CurrentMonthSaleDto newItem = saveItemMap.get(item.getId());
                if (!BooleanEnum.TRUE.getNumStr().equals(newItem.getChecked())) {
                    //选中状态，代表该行数据要删除,非选中则该行数据保留并更新
                    newList.add(newItem);
                }
            } else {
                newList.add(item);
            }
        }
        redisService.del(redisCacheKey);//先全部删掉再放进去吧
        if (!CollectionUtils.isEmpty(newList)) {
            redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());
        }
    }

    /**
     * 6、清理缓存接口
     *
     * @param cacheKey 缓存key
     */
    @Override
    public void clearCache(String cacheKey) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        redisService.del(redisCacheKey);
    }

    /**
     * 获取总条数
     *
     * @param cacheKey
     * @return
     */
    @Override
    public Integer getTotal(String cacheKey) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        return redisService.lSize(redisCacheKey).intValue();
    }

    /**
     * 在缓存中加入多条数据
     *
     * @param cacheKey
     * @param saveList
     */
    @Override
    public void addListCache(String cacheKey, List<CurrentMonthSaleDto> saveList) {
        if (CollectionUtils.isEmpty(saveList)) {
            return;
        }
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<CurrentMonthSaleDto> newList = Lists.newArrayList();
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);
        if (!CollectionUtils.isEmpty(objects)) {
            newList.addAll((List<CurrentMonthSaleDto>) (List) objects);
        }
        newList.addAll(saveList);
        redisService.del(redisCacheKey);
        redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());
    }

    /**
     * 自动计算费用
     *
     * @param dto
     */
    @Override
    public void calculationFee(CurrentMonthSaleDto dto, String businessModelCode) {
        //预计销量（吨）：自动计算：预计销量（箱）*系数*组合数量/1000。
        dto.setEstimatedSalesTon(dto.getEstimatedSalesBox().multiply(dto.getRatio()).multiply(dto.getCombinationQuantity()).setScale(2, RoundingMode.HALF_UP).divide(new BigDecimal(1000), 6, BigDecimal.ROUND_HALF_UP));
        //预计折后销售额（元）：自动计算：预计折前销售额-折扣
        if (dto.getDiscount() != null) {
            dto.setEstimatedAmountAfter(dto.getEstimatedAmountBefore().subtract(dto.getDiscount()).setScale(2, RoundingMode.HALF_UP));
        }
        //增值税（元）：自动计算：预计折前销售额/（1+税率）*税率
        dto.setVat(dto.getEstimatedAmountBefore().divide(BigDecimal.ONE.add(dto.getTaxRate()), 6, BigDecimal.ROUND_HALF_UP).multiply(dto.getTaxRate()).setScale(2, RoundingMode.HALF_UP));
        //定额费用合计（产品促销+大日期处理+新客专项+老客留存+仓间不均+高周转+平台毛保+销售返点）
        dto.setQuotaTotal(bdNull(dto.getProductPromotion())
                .add(bdNull(dto.getLargeDateProcess()))
                .add(bdNull(dto.getNewCustomer()))
                .add(bdNull(dto.getOldCustomer()))
                .add(bdNull(dto.getUneven()))
                .add(bdNull(dto.getHighTurnover()))
                .add(bdNull(dto.getPlatformGrossProtection()))
                .add(bdNull(dto.getSaleCommission())).setScale(2, RoundingMode.HALF_UP));
        //GMV(元）	活动底价*预计销量（箱）*组合数量
        dto.setGmv(dto.getActivityBasePrice().multiply(dto.getEstimatedSalesBox()).multiply(dto.getCombinationQuantity()).setScale(2, RoundingMode.HALF_UP));
        //净收入（元）	预计折前销售额-增值税-折扣+折扣/（1+税率）*税率
        try {
            dto.setNetIncome((dto.getEstimatedAmountBefore().subtract(dto.getVat()).subtract(dto.getDiscount()))
                    .add(dto.getDiscount()
                            .divide((BigDecimal.ONE.add(dto.getTaxRate())), 6, BigDecimal.ROUND_HALF_UP).multiply(dto.getTaxRate())).setScale(2, RoundingMode.HALF_UP));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            dto.setNetIncome(BigDecimal.ZERO);
        }
        //毛利（元）	净收入-成本价*组合数量*预计销量（箱）/（1+税率）
        try {
            dto.setGrossProfit(dto.getNetIncome().subtract(
                    (
                            dto.getCostPrice().multiply(dto.getEstimatedSalesBox()).multiply(dto.getCombinationQuantity())
                                    .divide((BigDecimal.ONE.add(dto.getTaxRate())), 2, RoundingMode.HALF_UP)
                    )).setScale(2, RoundingMode.HALF_UP));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            dto.setGrossProfit(BigDecimal.ZERO);
        }
        //经营利润（元）	净收入-净收入*0.5%-成本价*组合数量*预计销量（箱）/（1+税率）-销管报销-物流费用-（行政+人力）+其他收益
        try {
            dto.setOperateProfit(dto.getNetIncome().subtract(
                    dto.getNetIncome().multiply(new BigDecimal("0.005"))
            ).subtract(
                    dto.getCostPrice().multiply(dto.getEstimatedSalesBox()).multiply(dto.getCombinationQuantity())
                            .divide((BigDecimal.ONE.add(dto.getTaxRate())), 2, RoundingMode.HALF_UP)
            ).subtract(dto.getSaleReimburse()).subtract(dto.getLogistics()).subtract(
                    dto.getAdminHuman()
            ).setScale(2, RoundingMode.HALF_UP));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            dto.setOperateProfit(BigDecimal.ZERO);
        }
        //所得税（元）	经营利润*0.25
        try {
            dto.setIncomeTax(dto.getOperateProfit().multiply(new BigDecimal("0.25")).setScale(2, RoundingMode.HALF_UP));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            dto.setIncomeTax(BigDecimal.ZERO);
        }
        //净利润（元）	经营利润-所得税
        try {
            dto.setNetProfit(dto.getOperateProfit().subtract(dto.getIncomeTax()).setScale(2, RoundingMode.HALF_UP));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            dto.setNetProfit(BigDecimal.ZERO);
        }
        //净利率	净利润/净收入
        try {
            dto.setNetProfitRate(dto.getNetProfit().divide(dto.getNetIncome(), 2, RoundingMode.HALF_UP));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            dto.setNetProfitRate(BigDecimal.ZERO);
        }

        //分销计算
        if (FX_CODE.equals(businessModelCode)) {
            //分销基础返点（旬返）	供货价*组合系数*预计销量（箱）*旬返点数
            dto.setDistributionBaseTen(dto.getEstimatedAmountBefore().multiply(bdNull(dto.getDistributionBaseTenPoint())).divide(new BigDecimal("100"), 4, RoundingMode.HALF_UP));
            //分销基础返点（月返）	供货价*组合系数*预计销量（箱）*月返点数
            dto.setDistributionBaseMonth(dto.getEstimatedAmountBefore().multiply(bdNull(dto.getDistributionBaseMonthPoint())).divide(new BigDecimal("100"), 4, RoundingMode.HALF_UP));
            //月度目标达成返点（分销）	供货价*组合系数*预计销量（箱）*目标达成返点数
            dto.setMonthGoalAchievement(dto.getEstimatedAmountBefore().multiply(bdNull(dto.getMonthGoalAchievementPoint())).divide(new BigDecimal("100"), 4, RoundingMode.HALF_UP));
            //组合装（分销）	供货价*组合系数*预计销量（箱）*组合装点数
            dto.setPrePack(dto.getEstimatedAmountBefore().multiply(bdNull(dto.getPrePackPoint())).divide(new BigDecimal("100"), 4, RoundingMode.HALF_UP));
            //人员费用（分销）	供货价*组合系数*预计销量（箱）*人员点数
            dto.setStaffFee(dto.getEstimatedAmountBefore().multiply(bdNull(dto.getStaffFeePoint())).divide(new BigDecimal("100"), 4, RoundingMode.HALF_UP));
            //投放费用（分销）	供货价*组合系数*预计销量（箱）*投放点数
            dto.setPutFee(dto.getEstimatedAmountBefore().multiply(bdNull(dto.getPutFeePoint())).divide(new BigDecimal("100"), 4, RoundingMode.HALF_UP));
            //物流支持（分销）	供货价*组合系数*预计销量（箱）*物流点数
            dto.setLogisticsSupport(dto.getEstimatedAmountBefore().multiply(bdNull(dto.getLogisticsSupportPoint())).divide(new BigDecimal("100"), 4, RoundingMode.HALF_UP));
            //年度目标达成返点（分销）	供货价*组合数量*预计销量（箱）*{1-【其他所有费用的返点率相加】}*年返点数
            BigDecimal sum = bdNull(dto.getDistributionBaseTenPoint()).divide(new BigDecimal("100"), 4, RoundingMode.HALF_UP)
                    .add(bdNull(dto.getDistributionBaseMonthPoint()).divide(new BigDecimal("100"), 4, RoundingMode.HALF_UP))
                    .add(bdNull(dto.getMonthGoalAchievementPoint()).divide(new BigDecimal("100"), 4, RoundingMode.HALF_UP))
                    .add(bdNull(dto.getPrePackPoint()).divide(new BigDecimal("100"), 4, RoundingMode.HALF_UP))
                    .add(bdNull(dto.getStaffFeePoint()).divide(new BigDecimal("100"), 4, RoundingMode.HALF_UP))
                    .add(bdNull(dto.getPutFeePoint()).divide(new BigDecimal("100"), 4, RoundingMode.HALF_UP))
                    .add(bdNull(dto.getLogisticsSupportPoint()).divide(new BigDecimal("100"), 4, RoundingMode.HALF_UP));
            dto.setYearGoalAchievement(dto.getEstimatedAmountBefore()
//                    .multiply((new BigDecimal(1).subtract(sum)))
                    .multiply(bdNull(dto.getYearGoalAchievementPoint()))
                    .divide(new BigDecimal("100"), 2, RoundingMode.HALF_UP));
            //到手价政策费用 （供货价-到手价政策价）*组合数量*预计销量（箱）
            if (dto.getTakeHomePricePolicy() != null) {
                dto.setTakeHomePricePolicyFee(
                        (dto.getPlatformSupplyPrice().subtract(dto.getTakeHomePricePolicy()))
                                .multiply(dto.getCombinationQuantity())
                                .multiply(dto.getEstimatedSalesBox())
                );
            }
        }
    }

    /**
     * 获取redis缓存key
     *
     * @param cacheKey 前端传过来的key
     * @return redis的缓存key
     */
    private String getRedisCacheKey(String cacheKey) {
        return MONTH_SALE_CACHE_KEY_PREFIX + cacheKey;
    }

    private BigDecimal bdNull(BigDecimal b) {
        return b == null ? BigDecimal.ZERO : b;
    }
}

