package com.biz.crm.tpm.business.promotion.plan.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.tpm.business.business.policy.sdk.dto.BusinessPolicyDto;
import com.biz.crm.tpm.business.business.policy.sdk.service.BusinessPolicyService;
import com.biz.crm.tpm.business.month.budget.sdk.eunm.YesOrNoEnum;
import com.biz.crm.tpm.business.promotion.plan.local.entity.PromotionPlanEntity;
import com.biz.crm.tpm.business.promotion.plan.local.repository.GeneralExpensesRepository;
import com.biz.crm.tpm.business.promotion.plan.local.repository.PromotionPlanRepository;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.GeneralExpensesDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.GeneralExpensesService;
import com.biz.crm.tpm.business.promotion.plan.sdk.vo.GeneralExpensesVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.biz.crm.tpm.business.promotion.plan.sdk.constant.PromotionPlanConstant.CACHE_EXPIRE_TIME;
import static com.biz.crm.tpm.business.promotion.plan.sdk.constant.PromotionPlanConstant.GENERAL_EXPENSES_CACHE_KEY_PREFIX;

/**
 * 费用申请实体类(GeneralExpenses)表服务实现类
 *
 * @author yaoyongming
 * @since 2022-11-03 09:32:22
 */
@Service("generalExpensesService")
public class GeneralExpensesServiceImpl implements GeneralExpensesService {

    @Autowired(required = false)
    private GeneralExpensesRepository generalExpensesRepository;

    @Autowired(required = false)
    private RedisService redisService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private BusinessPolicyService businessPolicyService;

    @Autowired(required = false)
    private PromotionPlanRepository promotionPlanRepository;

    /**
     * 分页查询数据
     *
     * @param pageable        分页对象
     * @param generalExpenses 实体对象
     * @return
     */
    @Override
    public Page<GeneralExpensesVo> findByConditions(Pageable pageable, GeneralExpensesDto generalExpenses) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(generalExpenses)) {
            generalExpenses = new GeneralExpensesDto();
        }
        return this.generalExpensesRepository.findByConditions(pageable, generalExpenses);
    }

    /**
     * 1、获取明细列表分页接口
     *
     * @param pageable          分页参数
     * @param cacheKey          缓存key
     * @param promotionPlanCode 促销规划编码
     * @return
     */
    @Override
    public Page<GeneralExpensesVo> findCachePageList(Pageable pageable, String cacheKey, String promotionPlanCode) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        Page<GeneralExpensesVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        page.setTotal(0);
        page.setRecords(Lists.newArrayList());

        if (redisService.hasKey(redisCacheKey)) {
            //redis里面有的话直接从redis里面取
            Long total = redisService.lSize(redisCacheKey);
            page.setTotal(total);
            List<Object> objects = redisService.lRange(redisCacheKey, page.offset(), page.offset() + page.getSize() - 1);
            if (!CollectionUtils.isEmpty(objects)) {
                List<GeneralExpensesVo> voList = (List<GeneralExpensesVo>) nebulaToolkitService.copyCollectionByBlankList((List<GeneralExpensesDto>) (List) objects, GeneralExpensesDto.class, GeneralExpensesVo.class, HashSet.class, ArrayList.class);
                page.setRecords(voList);
            }
        } else {
            //redis里面没有
            if (StringUtils.isNotEmpty(promotionPlanCode)) {
                //从数据库里面查出来放到缓存里面
                List<GeneralExpensesDto> dtoList = generalExpensesRepository.findByPlanCode(promotionPlanCode);
                if (!CollectionUtils.isEmpty(dtoList)) {
                    redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, dtoList.toArray());
                }
                page.setTotal(dtoList.size());
                long start = page.offset();
                if (page.getTotal() > start) {
                    long end = page.offset() + page.getSize();
                    if (page.getTotal() < end) {
                        end = page.getTotal();
                    }
                    List<GeneralExpensesVo> voList = (List<GeneralExpensesVo>) nebulaToolkitService.copyCollectionByBlankList(dtoList.subList((int) page.offset(), (int) end), GeneralExpensesDto.class, GeneralExpensesVo.class, HashSet.class, ArrayList.class);
                    page.setRecords(voList);
                }
            }
        }
        return page;
    }

    /**
     * 获取缓存中的所有明细行
     *
     * @param cacheKey 缓存key
     */
    @Override
    public List<GeneralExpensesDto> findCacheList(String cacheKey) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        if (!redisService.hasKey(redisCacheKey)) {
            return Lists.newArrayList();
        }
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);
        return (List<GeneralExpensesDto>) (List) objects;
    }

    /**
     * 获取所有明细行
     *
     * @param code 促销规划编码
     */
    @Override
    public List<GeneralExpensesDto> findList(String code) {
        return this.generalExpensesRepository.findByPlanCode(code);
    }

    @Override
    public List<GeneralExpensesDto> findByDetailCodeList(List<String> codes) {
        if (CollectionUtils.isEmpty(codes)){
            return null;
        }
        List<GeneralExpensesDto> generalExpensesDtos = this.generalExpensesRepository.findByPlanDetailCodes(codes);
        if(!CollectionUtils.isEmpty(generalExpensesDtos)){
            List<String> promotionPlanCodes = generalExpensesDtos.stream().map(GeneralExpensesDto::getPromotionPlanCode).collect(Collectors.toList());
            List<PromotionPlanEntity> promotionPlanEntities = this.promotionPlanRepository.findByCodes(promotionPlanCodes);
            Map<String, PromotionPlanEntity> promotionPlanEntityMap = promotionPlanEntities.stream().collect(Collectors.toMap(PromotionPlanEntity::getPromotionPlanCode, Function.identity()));
            for (GeneralExpensesDto generalExpensesDto : generalExpensesDtos) {
                PromotionPlanEntity promotionPlanEntity = promotionPlanEntityMap.get(generalExpensesDto.getPromotionPlanCode());
                if(promotionPlanEntity != null) {
                    generalExpensesDto.setSalesOrgCode(promotionPlanEntity.getSalesOrgCode());
                    generalExpensesDto.setSalesOrgName(promotionPlanEntity.getSalesOrgName());
                }
            }
        }
        return generalExpensesDtos;
    }

    /**
     * 2、新增一行接口，在缓存中行首插入一条数据
     *
     * @param cacheKey 缓存key
     * @param saveList 要保存的当前页数据
     */
    @Override
    public void addItemCache(String cacheKey, List<GeneralExpensesDto> saveList) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<GeneralExpensesDto> newList = Lists.newArrayList();
        Map<String, GeneralExpensesDto> saveItemMap;
        if (!CollectionUtils.isEmpty(saveList)) {
            saveItemMap = saveList.stream().collect(Collectors.toMap(GeneralExpensesDto::getId, Function.identity()));
        } else {
            saveItemMap = Maps.newHashMap();
        }
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);
        if (!CollectionUtils.isEmpty(objects)) {
            for (Object object : objects) {
                GeneralExpensesDto item = (GeneralExpensesDto) object;
                newList.add(saveItemMap.getOrDefault(item.getId(), item));
            }
        }

        GeneralExpensesDto newItem = new GeneralExpensesDto();
        newItem.setId(UUID.randomUUID().toString().replace("-", ""));
        newItem.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        newItem.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        newList.add(0, newItem);
        redisService.del(redisCacheKey);//先全部删掉再放进去吧
        redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());
    }

    /**
     * 3、复制行接口，保存当前页数据后，在缓存中行首复制选中数据并返回第一页数据
     *
     * @param cacheKey 缓存key
     * @param saveList 要保存的当前页数据
     */
    @Override
    public void copyItemListCache(String cacheKey, List<GeneralExpensesDto> saveList) {
        if (CollectionUtils.isEmpty(saveList)) {
            return;
        }
        Map<String, GeneralExpensesDto> saveItemMap = saveList.stream().collect(Collectors.toMap(GeneralExpensesDto::getId, Function.identity()));
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);

        List<GeneralExpensesDto> newList = Lists.newArrayList();
        List<GeneralExpensesDto> copyList = Lists.newArrayList();
        for (Object object : objects) {
            GeneralExpensesDto item = (GeneralExpensesDto) object;
            if (saveItemMap.containsKey(item.getId())) {
                GeneralExpensesDto newItem = saveItemMap.get(item.getId());
                if (BooleanEnum.TRUE.getNumStr().equals(newItem.getChecked())) {
                    //选中状态，代表该行数据要复制
                    GeneralExpensesDto copyItem = nebulaToolkitService.copyObjectByWhiteList(newItem, GeneralExpensesDto.class, HashSet.class, ArrayList.class);
                    copyItem.setId(UUID.randomUUID().toString().replace("-", ""));
                    copyItem.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                    copyItem.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                    copyList.add(copyItem);
                }
                newList.add(newItem);
            } else {
                newList.add(item);
            }
        }
        newList.addAll(0, copyList);
        redisService.del(redisCacheKey);//先全部删掉再放进去吧
        redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());
    }

    /**
     * 4、保存当前页数据到缓存并返回指定页数据接口
     *
     * @param cacheKey 缓存key
     * @param saveList
     */
    @Override
    public void saveCurrentPageCache(String cacheKey, List<GeneralExpensesDto> saveList) {
        if (CollectionUtils.isEmpty(saveList)) {
            return;
        }
        Map<String, GeneralExpensesDto> saveItemMap = saveList.stream().collect(Collectors.toMap(GeneralExpensesDto::getId, Function.identity()));
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);

        List<GeneralExpensesDto> newList = Lists.newArrayList();
        for (Object object : objects) {
            GeneralExpensesDto item = (GeneralExpensesDto) object;
            newList.add(saveItemMap.getOrDefault(item.getId(), item));
        }
        redisService.del(redisCacheKey);//删掉重新放
        if (!CollectionUtils.isEmpty(newList)) {
            redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());//设置过期时间1天
        }
    }

    /**
     * 7、多行删除并保存当前页数据并返回指定页数据接口
     *
     * @param cacheKey 缓存key
     * @param itemList
     */
    @Override
    public void deleteCacheList(String cacheKey, List<GeneralExpensesDto> itemList) {
        if (CollectionUtils.isEmpty(itemList)) {
            return;
        }
        Map<String, GeneralExpensesDto> saveItemMap = itemList.stream().collect(Collectors.toMap(GeneralExpensesDto::getId, Function.identity()));
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);

        List<GeneralExpensesDto> newList = Lists.newArrayList();
        for (Object object : objects) {
            GeneralExpensesDto item = (GeneralExpensesDto) object;
            if (saveItemMap.containsKey(item.getId())) {
                GeneralExpensesDto newItem = saveItemMap.get(item.getId());
                if (!BooleanEnum.TRUE.getNumStr().equals(newItem.getChecked())) {
                    //选中状态，代表该行数据要删除,非选中则该行数据保留并更新
                    newList.add(newItem);
                }
            } else {
                newList.add(item);
            }
        }
        redisService.del(redisCacheKey);//先全部删掉再放进去吧
        if (!CollectionUtils.isEmpty(newList)) {
            redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());
        }
    }

    /**
     * 6、清理缓存接口
     *
     * @param cacheKey 缓存key
     */
    @Override
    public void clearCache(String cacheKey) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        redisService.del(redisCacheKey);
    }

    /**
     * 新增缓存
     *
     * @param cacheKey
     * @param saveList
     */
    @Override
    public void addItemListCache(String cacheKey, List<GeneralExpensesDto> saveList) {
        List<GeneralExpensesDto> newList = new ArrayList<>();
        String redisCacheKey = getRedisCacheKey(cacheKey);

        List<GeneralExpensesDto> cacheList = findCacheList(cacheKey);
        List<GeneralExpensesDto> manualList = cacheList.stream().filter(e -> !Boolean.TRUE.equals(e.getBeGenerate())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(manualList)) {
            newList.addAll(manualList);
            cacheList.removeAll(manualList);
        }
        List<String> oldListCode = cacheList.stream().map(e -> e.getActivityFormCode() + e.getCustomerCode() + Optional.ofNullable(e.getChannelCode()).orElse("")).collect(Collectors.toList());
        List<String> newListCode = saveList.stream().map(e -> e.getActivityFormCode() + e.getCustomerCode() + Optional.ofNullable(e.getChannelCode()).orElse("")).collect(Collectors.toList());
        //原有的活动形式-现有的活动形式，差集做删除
        List<GeneralExpensesDto> subList = cacheList.stream().filter(e -> !newListCode.contains(e.getActivityFormCode()+e.getCustomerCode() + Optional.ofNullable(e.getChannelCode()).orElse(""))).collect(Collectors.toList());
        cacheList.removeAll(subList);
        //相同的活动形式，只更新申请费用
        cacheList.forEach(e -> {
            for (GeneralExpensesDto dto : saveList) {
                if (e.getActivityFormCode().equals(dto.getActivityFormCode())
                        && e.getCustomerCode().equals(dto.getCustomerCode())
                        && Optional.ofNullable(e.getChannelCode()).orElse("").equals(Optional.ofNullable(dto.getChannelCode()).orElse(""))) {
                    e.setApplyAmount(dto.getApplyAmount());
                    break;
                }
            }
        });
        //现有的活动形式-原有的活动形式，差集做新增
        List<GeneralExpensesDto> addList = saveList.stream().filter(e -> !oldListCode.contains(e.getActivityFormCode()+e.getCustomerCode() + Optional.ofNullable(e.getChannelCode()).orElse(""))).collect(Collectors.toList());
        cacheList.addAll(addList);

        redisService.del(redisCacheKey);
        newList.addAll(cacheList);
        redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());
    }

    /**
     * 更新结案金额
     *
     * @param dtoList
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void updateAuditAmount(List<GeneralExpensesDto> dtoList) {
        generalExpensesRepository.updateAuditAmount(dtoList);

        List<String> expensesCodes = dtoList.stream().map(GeneralExpensesDto::getExpensesCode).collect(Collectors.toList());
        List<GeneralExpensesDto> generalExpensesDtoList = generalExpensesRepository.findByPlanDetailCodes(expensesCodes);
        //找出关联商务政策的map(商务政策，活动明细集合)
        Map<String, List<GeneralExpensesDto>> generalExpensesDtoMap = generalExpensesDtoList.stream().filter(o -> StringUtils.isNotEmpty(o.getCommercePolicyCode())).collect(Collectors.groupingBy(GeneralExpensesDto::getCommercePolicyCode));
        if(!CollectionUtils.isEmpty(generalExpensesDtoList)) {

            //map（活动明细，结案金额）
            Map<String, GeneralExpensesDto> generalExpensesDtoMap1 = dtoList.stream().collect(Collectors.toMap(GeneralExpensesDto::getExpensesCode, Function.identity()));
            List<BusinessPolicyDto> businessPolicyDtoList = new ArrayList<>();
            generalExpensesDtoMap.forEach((key,value)->{
                BusinessPolicyDto businessPolicyDto = new BusinessPolicyDto();
                businessPolicyDto.setBusinessPolicyCode(key);
                businessPolicyDto.setAlreadyAuditAmount(BigDecimal.ZERO);

                for (GeneralExpensesDto generalExpensesDto : value) {
                    GeneralExpensesDto generalExpensesDto1 = generalExpensesDtoMap1.get(generalExpensesDto.getExpensesCode());
                    businessPolicyDto.setAlreadyAuditAmount(businessPolicyDto.getAlreadyAuditAmount().add(generalExpensesDto1.getAlreadyAuditAmount()));
                    businessPolicyDto.setLastAuditAmount(businessPolicyDto.getAlreadyAuditAmount());
                    if(!YesOrNoEnum.YES.getCode().equals(generalExpensesDto1.getWholeAudit())) {
                        businessPolicyDto.setWholeAudit(YesOrNoEnum.NO.getCode());
                    }
                }
                if(!YesOrNoEnum.NO.getCode().equals(businessPolicyDto.getWholeAudit())) {
                    //查询其它此商务政策的促销规划是否完全结案
                    businessPolicyDto.setWholeAudit(generalExpensesRepository.findOtherDetailWholeAudit(key));
                }
                businessPolicyDtoList.add(businessPolicyDto);
            });
            if(!CollectionUtils.isEmpty(businessPolicyDtoList)) {
                businessPolicyService.updateBusinessPolicyByAuditPass(businessPolicyDtoList);
            }
        }
    }

    /**
     * 根据明细编码查询
     *
     * @param expensesCodeList
     * @return
     */
    @Override
    public List<GeneralExpensesVo> findByExpensesCode(List<String> expensesCodeList) {
        return generalExpensesRepository.findByExpensesCodeExt(expensesCodeList);
    }

    @Override
    public Page<GeneralExpensesVo> findForWithholding(Pageable pageable, String processStatus, String upAccountStatus) {
        Page<GeneralExpensesVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return generalExpensesRepository.findForWithholding(page, processStatus, upAccountStatus);
    }

    /**
     * 获取总条数
     *
     * @param cacheKey
     * @return
     */
    @Override
    public Integer getTotal(String cacheKey) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        return redisService.lSize(redisCacheKey).intValue();
    }

    /**
     * 在缓存中加入多条数据
     *
     * @param cacheKey
     * @param saveList
     */
    @Override
    public void addListCache(String cacheKey, List<GeneralExpensesDto> saveList) {
        if (CollectionUtils.isEmpty(saveList)) {
            return;
        }
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<GeneralExpensesDto> newList = Lists.newArrayList();
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);
        //需要设置为自动生成的明细id
        Set<String> ids = new HashSet<>();
        if (!CollectionUtils.isEmpty(objects)) {
            List<GeneralExpensesDto> original = (List<GeneralExpensesDto>) (List) objects;
            original.forEach(e -> {
                GeneralExpensesDto sameDto = saveList.stream().filter(save -> save.getId().equals(e.getId())).findFirst().orElse(null);
                if (sameDto != null) {
                    if (Boolean.TRUE.equals(e.getBeGenerate())) {
                        ids.add(e.getId());
                    }
                } else {
                    //保留原有的明细
                    newList.add(e);
                }
            });
        }
        saveList.forEach(e -> {
            if (ids.contains(e.getId())) {
                e.setBeGenerate(Boolean.TRUE);
            }
        });
        newList.addAll(saveList);
        redisService.del(redisCacheKey);
        redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());
    }

    @Override
    public BigDecimal getContractAmount(String promotionPlanCode) {

        return this.generalExpensesRepository.getContractAmount(promotionPlanCode);

    }

    /**
     * 根据商务政策查询
     *
     * @param businessPolicyCodeList
     * @return
     */
    @Override
    public List<GeneralExpensesVo> findByBusinessPolicyCode(List<String> businessPolicyCodeList) {
        return generalExpensesRepository.findByBusinessPolicyCode(businessPolicyCodeList);
    }

    @Override
    public boolean isOtherDetailWholeAudit(String businessPolicyCode, List<String> expensesCodes) {
        if(StringUtils.isEmpty(businessPolicyCode)){
            return false;
        }
        return generalExpensesRepository.isOtherDetailWholeAudit2(businessPolicyCode,expensesCodes);
    }

    /**
     * 获取redis缓存key
     *
     * @param cacheKey 前端传过来的key
     * @return redis的缓存key
     */
    private String getRedisCacheKey(String cacheKey) {
        return GENERAL_EXPENSES_CACHE_KEY_PREFIX + cacheKey;
    }
}

