package com.biz.crm.tpm.business.promotion.plan.local.service.internal;


import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.tpm.business.promotion.plan.local.repository.OtherExpensesRepository;
import com.biz.crm.tpm.business.promotion.plan.sdk.constant.PromotionPlanConstant;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.OtherExpensesDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.BudgetProfitLossService;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.OtherExpensesService;
import com.biz.crm.tpm.business.promotion.plan.sdk.vo.BudgetProfitLossVo;
import com.biz.crm.tpm.business.promotion.plan.sdk.vo.OtherExpensesVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.springframework.stereotype.Service;
import org.springframework.beans.factory.annotation.Autowired;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.Pageable;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.biz.crm.tpm.business.promotion.plan.sdk.constant.PromotionPlanConstant.*;

/**
 * 其他费用实体类(OtherExpensesEntity)表服务实现类
 *
 * @author yaoyongming
 * @since 2022-11-03 09:32:22
 */
@Service("otherExpensesService")
public class OtherExpensesServiceImpl implements OtherExpensesService {

    @Autowired(required = false)
    private OtherExpensesRepository otherExpensesRepository;

    @Autowired(required = false)
    private RedisService redisService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private BudgetProfitLossService budgetProfitLossService;

    /**
     * 分页查询数据
     *
     * @param pageable      分页对象
     * @param otherExpenses 实体对象
     * @return
     */
    @Override
    public Page<OtherExpensesVo> findByConditions(Pageable pageable, OtherExpensesDto otherExpenses) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(otherExpenses)) {
            otherExpenses = new OtherExpensesDto();
        }
        return this.otherExpensesRepository.findByConditions(pageable, otherExpenses);
    }

    /**
     * 1、获取明细列表分页接口
     *
     * @param pageable          分页参数
     * @param cacheKey          缓存key
     * @param promotionPlanCode 促销规划编码
     * @return
     */
    @Override
    public Page<OtherExpensesVo> findCachePageList(Pageable pageable, String cacheKey, String promotionPlanCode) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        Page<OtherExpensesVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        page.setTotal(0);
        page.setRecords(Lists.newArrayList());

        if (redisService.hasKey(redisCacheKey)) {
            //redis里面有的话直接从redis里面取
            Long total = redisService.lSize(redisCacheKey);
            page.setTotal(total);
            List<Object> objects = redisService.lRange(redisCacheKey, page.offset(), page.offset() + page.getSize() - 1);
            if (!CollectionUtils.isEmpty(objects)) {
                List<OtherExpensesVo> voList = (List<OtherExpensesVo>) nebulaToolkitService.copyCollectionByBlankList((List<OtherExpensesDto>) (List) objects, OtherExpensesDto.class, OtherExpensesVo.class, HashSet.class, ArrayList.class);
                page.setRecords(voList);
            }
        } else {
            //redis里面没有
            if (StringUtils.isNotEmpty(promotionPlanCode)) {
                //从数据库里面查出来放到缓存里面
                List<OtherExpensesDto> dtoList = otherExpensesRepository.findByPlanCode(promotionPlanCode);
                if (!CollectionUtils.isEmpty(dtoList)) {
                    redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, dtoList.toArray());
                }
                page.setTotal(dtoList.size());
                long start = page.offset();
                if (page.getTotal() > start) {
                    long end = page.offset() + page.getSize();
                    if (page.getTotal() < end) {
                        end = page.getTotal();
                    }
                    List<OtherExpensesVo> voList = (List<OtherExpensesVo>) nebulaToolkitService.copyCollectionByBlankList(dtoList.subList((int) page.offset(), (int) end), OtherExpensesDto.class, OtherExpensesVo.class, HashSet.class, ArrayList.class);
                    page.setRecords(voList);
                }
            }
        }
        return page;
    }

    /**
     * 获取缓存中的所有明细行
     *
     * @param cacheKey 缓存key
     */
    @Override
    public List<OtherExpensesDto> findCacheList(String cacheKey) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        if (!redisService.hasKey(redisCacheKey)) {
            return Lists.newArrayList();
        }
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);
        return (List<OtherExpensesDto>) (List) objects;
    }

    /**
     * 2、新增一行接口，在缓存中行首插入一条数据
     *
     * @param cacheKey 缓存key
     * @param saveList 要保存的当前页数据
     */
    @Override
    public void addItemCache(String cacheKey, String businessModelCode, String customerCode, String yearMonthLy, List<OtherExpensesDto> saveList) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<OtherExpensesDto> newList = Lists.newArrayList();
        Map<String, OtherExpensesDto> saveItemMap;
        if (!CollectionUtils.isEmpty(saveList)) {
            if (saveList.size() > 1) {
                throw new RuntimeException("其他费用录入只能有一条记录！");
            }
            saveItemMap = saveList.stream().collect(Collectors.toMap(OtherExpensesDto::getId, Function.identity()));
        } else {
            saveItemMap = Maps.newHashMap();
        }
        //如果业务模式=分销模式，用“分销业务部的部门编码”2006279“+活动规划月份（开始时间所属月）+对应账面”匹配预算损益表对应部门对应月份的【（运输装卸费用）+（销售费用--仓储费用）】
        //	其他业务模式：用促销规划的当前客户和活动规划月份（开始时间所属月）对应的预算月份以及账面匹配预算损益表的【（运输装卸费用）+（销售费用--仓储费用）】
        BudgetProfitLossVo dto = new BudgetProfitLossVo();
        if (PromotionPlanConstant.FX_CODE.equals(businessModelCode)) {
            dto.setDepartmentCode(PromotionPlanConstant.FXYWB_CODE);
        } else {
            dto.setCustomerCode(customerCode);
        }
        dto.setType("账面");
        List<BudgetProfitLossVo> profitLoss = budgetProfitLossService.findProfitLoss(dto, Arrays.asList(yearMonthLy));
        BigDecimal sum = BigDecimal.ZERO;
        if (!CollectionUtils.isEmpty(profitLoss)) {
            sum = profitLoss.stream().map(e -> bdNull(e.getTransportHandling()).add(bdNull(e.getSaleFeeInventory()))).reduce(BigDecimal.ZERO, BigDecimal::add);
        }
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);
        if (!CollectionUtils.isEmpty(objects)) {
            for (Object object : objects) {
                OtherExpensesDto item = (OtherExpensesDto) object;
                newList.add(saveItemMap.getOrDefault(item.getId(), item));
            }
        }

        OtherExpensesDto newItem = new OtherExpensesDto();
        newItem.setId(UUID.randomUUID().toString().replace("-", ""));
        newItem.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        newItem.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        newItem.setLogisticsAmountPlan(sum);
        newItem.setLogisticsAmountBudget(sum);
        newList.add(0, newItem);
        redisService.del(redisCacheKey);//先全部删掉再放进去吧
        redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());
    }

    /**
     * 3、复制行接口，保存当前页数据后，在缓存中行首复制选中数据并返回第一页数据
     *
     * @param cacheKey 缓存key
     * @param saveList 要保存的当前页数据
     */
    @Override
    public void copyItemListCache(String cacheKey, List<OtherExpensesDto> saveList) {
        if (CollectionUtils.isEmpty(saveList)) {
            return;
        }
        Map<String, OtherExpensesDto> saveItemMap = saveList.stream().collect(Collectors.toMap(OtherExpensesDto::getId, Function.identity()));
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);

        List<OtherExpensesDto> newList = Lists.newArrayList();
        List<OtherExpensesDto> copyList = Lists.newArrayList();
        for (Object object : objects) {
            OtherExpensesDto item = (OtherExpensesDto) object;
            if (saveItemMap.containsKey(item.getId())) {
                OtherExpensesDto newItem = saveItemMap.get(item.getId());
                if (BooleanEnum.TRUE.getNumStr().equals(newItem.getChecked())) {
                    //选中状态，代表该行数据要复制
                    OtherExpensesDto copyItem = nebulaToolkitService.copyObjectByWhiteList(newItem, OtherExpensesDto.class, HashSet.class, ArrayList.class);
                    copyItem.setId(UUID.randomUUID().toString().replace("-", ""));
                    copyItem.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                    copyItem.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                    copyList.add(copyItem);
                }
                newList.add(newItem);
            } else {
                newList.add(item);
            }
        }
        newList.addAll(0, copyList);
        redisService.del(redisCacheKey);//先全部删掉再放进去吧
        redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());
    }

    /**
     * 4、保存当前页数据到缓存并返回指定页数据接口
     *
     * @param cacheKey 缓存key
     * @param saveList
     */
    @Override
    public void saveCurrentPageCache(String cacheKey, List<OtherExpensesDto> saveList) {
        if (CollectionUtils.isEmpty(saveList)) {
            return;
        }
        Map<String, OtherExpensesDto> saveItemMap = saveList.stream().collect(Collectors.toMap(OtherExpensesDto::getId, Function.identity()));
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);

        List<OtherExpensesDto> newList = Lists.newArrayList();
        for (Object object : objects) {
            OtherExpensesDto item = (OtherExpensesDto) object;
            newList.add(saveItemMap.getOrDefault(item.getId(), item));
        }
        redisService.del(redisCacheKey);//删掉重新放
        if (!CollectionUtils.isEmpty(newList)) {
            redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());//设置过期时间1天
        }
    }

    /**
     * 7、多行删除并保存当前页数据并返回指定页数据接口
     *
     * @param cacheKey 缓存key
     * @param itemList
     */
    @Override
    public void deleteCacheList(String cacheKey, List<OtherExpensesDto> itemList) {
        if (CollectionUtils.isEmpty(itemList)) {
            return;
        }
        Map<String, OtherExpensesDto> saveItemMap = itemList.stream().collect(Collectors.toMap(OtherExpensesDto::getId, Function.identity()));
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);

        List<OtherExpensesDto> newList = Lists.newArrayList();
        for (Object object : objects) {
            OtherExpensesDto item = (OtherExpensesDto) object;
            if (saveItemMap.containsKey(item.getId())) {
                OtherExpensesDto newItem = saveItemMap.get(item.getId());
                if (!BooleanEnum.TRUE.getNumStr().equals(newItem.getChecked())) {
                    //选中状态，代表该行数据要删除,非选中则该行数据保留并更新
                    newList.add(newItem);
                }
            } else {
                newList.add(item);
            }
        }
        redisService.del(redisCacheKey);//先全部删掉再放进去吧
        if (!CollectionUtils.isEmpty(newList)) {
            redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());
        }
    }

    /**
     * 6、清理缓存接口
     *
     * @param cacheKey 缓存key
     */
    @Override
    public void clearCache(String cacheKey) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        redisService.del(redisCacheKey);
    }

    /**
     * 获取行政规划+人力成本汇总
     *
     * @param cacheKey
     * @param purchaseCode
     * @return
     */
    @Override
    public BigDecimal getAdminHumanTotal(String cacheKey, String purchaseCode) {
        BigDecimal amount = BigDecimal.ZERO;
        List<OtherExpensesDto> otherExpensesDtos = new ArrayList<>();
        if (StringUtils.isNotBlank(cacheKey)) {
            otherExpensesDtos = findCacheList(cacheKey);
        } else {
            if (StringUtils.isNotBlank(purchaseCode)) {
                otherExpensesDtos = otherExpensesRepository.findByPlanCode(purchaseCode);
            }
        }
        if (!CollectionUtils.isEmpty(otherExpensesDtos)) {
            amount = otherExpensesDtos.stream().map(e -> (e.getAdminAmountPlan() == null ? BigDecimal.ZERO : e.getAdminAmountPlan()).add(
                        e.getHumanAmountPlan() == null ? BigDecimal.ZERO : e.getHumanAmountPlan()
                )).reduce(BigDecimal.ZERO, BigDecimal::add);
        }
        return amount;
    }

    private BigDecimal bdNull(BigDecimal b) {
        return b == null ? BigDecimal.ZERO : b;
    }

    /**
     * 获取所有明细行
     *
     * @param code 促销规划编码
     */
    @Override
    public List<OtherExpensesDto> findList(String code) {
        return otherExpensesRepository.findByPlanCode(code);
    }

    /**
     * 获取redis缓存key
     *
     * @param cacheKey 前端传过来的key
     * @return redis的缓存key
     */
    private String getRedisCacheKey(String cacheKey) {
        return OTHER_EXPENSES_CACHE_KEY_PREFIX + cacheKey;
    }
}

