package com.biz.crm.tpm.business.promotion.plan.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.tpm.business.promotion.plan.local.repository.PurchaseSaleRepository;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.PurchaseSaleDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.PurchaseSaleService;
import com.biz.crm.tpm.business.promotion.plan.sdk.vo.PurchaseSaleVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.biz.crm.tpm.business.promotion.plan.sdk.constant.PromotionPlanConstant.CACHE_EXPIRE_TIME;
import static com.biz.crm.tpm.business.promotion.plan.sdk.constant.PromotionPlanConstant.PURCHASE_SALE_CACHE_KEY_PREFIX;

/**
 * 采销库存(PurchaseSale)表服务实现类
 *
 * @author yaoyongming
 * @since 2022-11-26 14:58:22
 */
@Service("purchaseSaleService")
public class PurchaseSaleServiceImpl implements PurchaseSaleService {

    @Autowired(required = false)
    private PurchaseSaleRepository purchaseSaleRepository;

    @Autowired(required = false)
    private RedisService redisService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    /**
     * 分页查询数据
     *
     * @param pageable     分页对象
     * @param purchaseSale 实体对象
     * @return
     */
    @Override
    public Page<PurchaseSaleVo> findByConditions(Pageable pageable, PurchaseSaleDto purchaseSale) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(purchaseSale)) {
            purchaseSale = new PurchaseSaleDto();
        }
        return this.purchaseSaleRepository.findByConditions(pageable, purchaseSale);
    }

    /**
     * 1、获取明细列表分页接口
     *
     * @param pageable          分页参数
     * @param cacheKey          缓存key
     * @param promotionPlanCode 促销规划编码
     * @return
     */
    @Override
    public Page<PurchaseSaleVo> findCachePageList(Pageable pageable, String cacheKey, String promotionPlanCode) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        Page<PurchaseSaleVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        page.setTotal(0);
        page.setRecords(Lists.newArrayList());

        if (redisService.hasKey(redisCacheKey)) {
            //redis里面有的话直接从redis里面取
            Long total = redisService.lSize(redisCacheKey);
            page.setTotal(total);
            List<Object> objects = redisService.lRange(redisCacheKey, page.offset(), page.offset() + page.getSize() - 1);
            if (!CollectionUtils.isEmpty(objects)) {
                List<PurchaseSaleVo> voList = (List<PurchaseSaleVo>) nebulaToolkitService.copyCollectionByBlankList((List<PurchaseSaleDto>) (List) objects, PurchaseSaleDto.class, PurchaseSaleVo.class, HashSet.class, ArrayList.class);
                page.setRecords(voList);
            }
        } else {
            //redis里面没有
            if (StringUtils.isNotEmpty(promotionPlanCode)) {
                //从数据库里面查出来放到缓存里面
                List<PurchaseSaleDto> dtoList = purchaseSaleRepository.findByPlanCode(promotionPlanCode);
                if (!CollectionUtils.isEmpty(dtoList)) {
                    redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, dtoList.toArray());
                }
                page.setTotal(dtoList.size());
                long start = page.offset();
                if (page.getTotal() > start) {
                    long end = page.offset() + page.getSize();
                    if (page.getTotal() < end) {
                        end = page.getTotal();
                    }
                    List<PurchaseSaleVo> voList = (List<PurchaseSaleVo>) nebulaToolkitService.copyCollectionByBlankList(dtoList.subList((int) page.offset(), (int) end), PurchaseSaleDto.class, PurchaseSaleVo.class, HashSet.class, ArrayList.class);
                    page.setRecords(voList);
                }
            }
        }
        return page;
    }

    /**
     * 获取缓存中的所有明细行
     *
     * @param cacheKey 缓存key
     */
    @Override
    public List<PurchaseSaleDto> findCacheList(String cacheKey) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        if (!redisService.hasKey(redisCacheKey)) {
            return Lists.newArrayList();
        }
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);
        return (List<PurchaseSaleDto>) (List) objects;
    }

    /**
     * 获取所有明细行
     *
     * @param code 促销规划编码
     */
    @Override
    public List<PurchaseSaleDto> findList(String code) {
        return this.purchaseSaleRepository.findByPlanCode(code);
    }

    /**
     * 2、新增一行接口，在缓存中行首插入一条数据
     *
     * @param cacheKey 缓存key
     * @param saveList 要保存的当前页数据
     */
    @Override
    public void addItemCache(String cacheKey, List<PurchaseSaleDto> saveList) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<PurchaseSaleDto> newList = Lists.newArrayList();
        Map<String, PurchaseSaleDto> saveItemMap;
        if (!CollectionUtils.isEmpty(saveList)) {
            saveItemMap = saveList.stream().collect(Collectors.toMap(PurchaseSaleDto::getId, Function.identity()));
        } else {
            saveItemMap = Maps.newHashMap();
        }
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);
        if (!CollectionUtils.isEmpty(objects)) {
            for (Object object : objects) {
                PurchaseSaleDto item = (PurchaseSaleDto) object;
                newList.add(saveItemMap.getOrDefault(item.getId(), item));
            }
        }

        PurchaseSaleDto newItem = new PurchaseSaleDto();
        newItem.setId(UUID.randomUUID().toString().replace("-", ""));
        newItem.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        newItem.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        newList.add(0, newItem);
        redisService.del(redisCacheKey);//先全部删掉再放进去吧
        redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());
    }

    /**
     * 3、复制行接口，保存当前页数据后，在缓存中行首复制选中数据并返回第一页数据
     *
     * @param cacheKey 缓存key
     * @param saveList 要保存的当前页数据
     */
    @Override
    public void copyItemListCache(String cacheKey, List<PurchaseSaleDto> saveList) {
        if (CollectionUtils.isEmpty(saveList)) {
            return;
        }
        Map<String, PurchaseSaleDto> saveItemMap = saveList.stream().collect(Collectors.toMap(PurchaseSaleDto::getId, Function.identity()));
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);

        List<PurchaseSaleDto> newList = Lists.newArrayList();
        List<PurchaseSaleDto> copyList = Lists.newArrayList();
        for (Object object : objects) {
            PurchaseSaleDto item = (PurchaseSaleDto) object;
            if (saveItemMap.containsKey(item.getId())) {
                PurchaseSaleDto newItem = saveItemMap.get(item.getId());
                if (BooleanEnum.TRUE.getNumStr().equals(newItem.getChecked())) {
                    //选中状态，代表该行数据要复制
                    PurchaseSaleDto copyItem = nebulaToolkitService.copyObjectByWhiteList(newItem, PurchaseSaleDto.class, HashSet.class, ArrayList.class);
                    copyItem.setId(UUID.randomUUID().toString().replace("-", ""));
                    copyItem.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                    copyItem.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                    copyList.add(copyItem);
                }
                newList.add(newItem);
            } else {
                newList.add(item);
            }
        }
        newList.addAll(0, copyList);
        redisService.del(redisCacheKey);//先全部删掉再放进去吧
        redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());
    }

    /**
     * 4、保存当前页数据到缓存并返回指定页数据接口
     *
     * @param cacheKey 缓存key
     * @param saveList
     */
    @Override
    public void saveCurrentPageCache(String cacheKey, List<PurchaseSaleDto> saveList) {
        if (CollectionUtils.isEmpty(saveList)) {
            return;
        }
        Map<String, PurchaseSaleDto> saveItemMap = saveList.stream().collect(Collectors.toMap(PurchaseSaleDto::getId, Function.identity()));
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);

        List<PurchaseSaleDto> newList = Lists.newArrayList();
        for (Object object : objects) {
            PurchaseSaleDto item = (PurchaseSaleDto) object;
            newList.add(saveItemMap.getOrDefault(item.getId(), item));
        }
        redisService.del(redisCacheKey);//删掉重新放
        if (!CollectionUtils.isEmpty(newList)) {
            redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());//设置过期时间1天
        }
    }

    /**
     * 7、多行删除并保存当前页数据并返回指定页数据接口
     *
     * @param cacheKey 缓存key
     * @param itemList
     */
    @Override
    public void deleteCacheList(String cacheKey, List<PurchaseSaleDto> itemList) {
        if (CollectionUtils.isEmpty(itemList)) {
            return;
        }
        Map<String, PurchaseSaleDto> saveItemMap = itemList.stream().collect(Collectors.toMap(PurchaseSaleDto::getId, Function.identity()));
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);

        List<PurchaseSaleDto> newList = Lists.newArrayList();
        for (Object object : objects) {
            PurchaseSaleDto item = (PurchaseSaleDto) object;
            if (saveItemMap.containsKey(item.getId())) {
                PurchaseSaleDto newItem = saveItemMap.get(item.getId());
                if (!BooleanEnum.TRUE.getNumStr().equals(newItem.getChecked())) {
                    //选中状态，代表该行数据要删除,非选中则该行数据保留并更新
                    newList.add(newItem);
                }
            } else {
                newList.add(item);
            }
        }
        redisService.del(redisCacheKey);//先全部删掉再放进去吧
        if (!CollectionUtils.isEmpty(newList)) {
            redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());
        }
    }

    /**
     * 6、清理缓存接口
     *
     * @param cacheKey 缓存key
     */
    @Override
    public void clearCache(String cacheKey) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        redisService.del(redisCacheKey);
    }

    /**
     * 获取总条数
     *
     * @param cacheKey
     * @return
     */
    @Override
    public Integer getTotal(String cacheKey) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        return redisService.lSize(redisCacheKey).intValue();
    }

    /**
     * 在缓存中加入多条数据
     *
     * @param cacheKey
     * @param saveList
     */
    @Override
    public void addListCache(String cacheKey, List<PurchaseSaleDto> saveList) {
        if (CollectionUtils.isEmpty(saveList)) {
            return;
        }
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<PurchaseSaleDto> newList = Lists.newArrayList();
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);
        if (!CollectionUtils.isEmpty(objects)) {
            newList.addAll((List<PurchaseSaleDto>) (List) objects);
        }
        newList.addAll(saveList);
        redisService.del(redisCacheKey);
        redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, newList.toArray());
    }

    /**
     * 在缓存中更新所有数据
     *
     * @param cacheKey
     * @param itemList
     */
    @Override
    public void updateListCache(String cacheKey, List<PurchaseSaleDto> itemList) {
        if (CollectionUtils.isEmpty(itemList)) {
            return;
        }
        String redisCacheKey = getRedisCacheKey(cacheKey);
        redisService.del(redisCacheKey);
        redisService.lPushAll(redisCacheKey, CACHE_EXPIRE_TIME, itemList.toArray());
    }

    /**
     * 获取redis缓存key
     *
     * @param cacheKey 前端传过来的key
     * @return redis的缓存key
     */
    private String getRedisCacheKey(String cacheKey) {
        return PURCHASE_SALE_CACHE_KEY_PREFIX + cacheKey;
    }
}

