package com.biz.crm.tpm.business.promotion.plan.local.service.process;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mdm.business.org.sdk.service.OrgVoService;
import com.biz.crm.mdm.business.org.sdk.vo.OrgVo;
import com.biz.crm.tpm.business.promotion.plan.sdk.constant.PromotionPlanConstant;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.ActualProfitLossService;
import com.biz.crm.tpm.business.promotion.plan.sdk.vo.ActualProfitLossImportsVo;
import com.biz.crm.tpm.business.promotion.plan.sdk.vo.ActualProfitLossVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 损益分析表导入
 *
 * @author: yaoyongming
 * @date: 2023/2/16 10:26
 */
@Slf4j
@Component
public class ActualProfitLossImportsProcess implements ImportProcess<ActualProfitLossImportsVo> {

    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private DictDataVoService dictDataVoService;

    @Autowired(required = false)
    private OrgVoService orgVoService;

    @Autowired(required = false)
    private ActualProfitLossService actualProfitLossService;


    /**
     * 单次缓存的数量
     *
     * @return
     */
    @Override
    public Integer getBatchCount() {
        return 10000;
    }

    /**
     * 数据处理
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, ActualProfitLossImportsVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        try {
            log.info("开始导入......");
            Validate.notEmpty(data, "导入数据不能为空！");

            log.info("开始导入校验......");
            List<ActualProfitLossVo> importList = this.validate(data, params);
            log.info("导入校验结束......");

            log.info("开始导入保存......");
            actualProfitLossService.saveBatch(importList);
            log.info("结束导入保存......");

        } catch (Exception e) {
            log.error(e.getMessage(), e);
            e.printStackTrace();
            throw e;
        }
        return null;
    }


    private List<ActualProfitLossVo> validate(LinkedHashMap<Integer, ActualProfitLossImportsVo> data, Map<String, Object> params) {
        List<ActualProfitLossImportsVo> importList = new ArrayList<>(data.values());

        log.info("开始非空校验......");
        //非空校验
        for (int i = 0; i < importList.size(); i++) {
            Validate.notBlank(importList.get(i).getBusinessFormatName(), "业态，不能为空！错误数据位于第【"+ (i+2) +"】行");
            Validate.notBlank(importList.get(i).getBusinessUnitName(), "业务单元，不能为空！错误数据位于第【"+ (i+2) +"】行");
            Validate.notBlank(importList.get(i).getYearMonthly(), "年月，不能为空！错误数据位于第【"+ (i+2) +"】行");
            Validate.notBlank(importList.get(i).getDepartmentCode(), "部门编码，不能为空！错误数据位于第【"+ (i+2) +"】行");
            Validate.notBlank(importList.get(i).getCustomerCode(), "客户编码，不能为空！错误数据位于第【"+ (i+2) +"】行");
            Validate.notBlank(importList.get(i).getType(), "前台/账面，不能为空！错误数据位于第【"+ (i+2) +"】行");
        }
        log.info("结束非空校验......");

        log.info("开始查询关联数据......");
        // 查询数据字典
        Map<String, List<DictDataVo>> dictDataMap = dictDataVoService.findByDictTypeCodeList(Arrays.asList(PromotionPlanConstant.MDM_BUSINESS_FORMAT, PromotionPlanConstant.MDM_BUSINESS_UNIT));

        log.info("开始关联数据封装......");
        //关联数据封装
        Set<String> customerCodeSet = new HashSet<>();
        Set<String> departmentCodeSet = new HashSet<>();
        for (ActualProfitLossImportsVo importVo : importList) {
            if (StringUtils.isNotBlank(importVo.getCustomerCode())) {
                customerCodeSet.add(importVo.getCustomerCode());
            }
            if (StringUtils.isNotBlank(importVo.getDepartmentCode())) {
                departmentCodeSet.add(importVo.getDepartmentCode());
            }
        }

        Map<String, CustomerVo> customerVoMap = null;
        Map<String, OrgVo> orgVoMap = null;

                //关联数据查询
        List<CustomerVo> customerVos = customerVoService.findByCustomerCodes(new ArrayList<>(customerCodeSet));
        if (CollectionUtils.isNotEmpty(customerVos)) {
            customerVoMap = customerVos.stream().collect(Collectors.toMap(CustomerVo::getCustomerCode, Function.identity()));
        }
        List<OrgVo> orgVos = orgVoService.findByOrgCodes(new ArrayList<>(departmentCodeSet));
        if (CollectionUtils.isNotEmpty(orgVos)) {
            orgVoMap = orgVos.stream().collect(Collectors.toMap(OrgVo::getOrgCode, Function.identity()));
        }
        log.info("结束关联数据封装......");

        Collection<ActualProfitLossVo> dtoList = nebulaToolkitService.copyCollectionByWhiteList(importList, ActualProfitLossImportsVo.class, ActualProfitLossVo.class, LinkedHashSet.class, ArrayList.class);


        log.info("开始校验关联数据校验......");
        int index = 2;
        for (ActualProfitLossVo dto : dtoList) {
            DictDataVo businessFormat = dictDataMap.get(PromotionPlanConstant.MDM_BUSINESS_FORMAT).stream().filter(e -> e.getDictValue().equals(dto.getBusinessFormatName())).findFirst().orElse(null);
            if (businessFormat != null) {
                dto.setBusinessFormatCode(businessFormat.getDictCode());
            } else {
                throw new IllegalArgumentException("业态【" + dto.getBusinessFormatName() + "】错误，请检查！");
            }
            DictDataVo businessUnit = dictDataMap.get(PromotionPlanConstant.MDM_BUSINESS_UNIT).stream().filter(e -> e.getDictValue().equals(dto.getBusinessUnitName())).findFirst().orElse(null);
            if (businessUnit != null) {
                dto.setBusinessUnitCode(businessUnit.getDictCode());
            } else {
                throw new IllegalArgumentException("业务单元【" + dto.getBusinessUnitName() + "】错误，请检查！");
            }
            if (StringUtils.isNotBlank(dto.getCustomerCode())) {
                if (customerVoMap != null && customerVoMap.containsKey(dto.getCustomerCode())) {
                    CustomerVo customerVo = customerVoMap.get(dto.getCustomerCode());
                    dto.setCustomerName(customerVo.getCustomerName());
                } else {
                    throw new IllegalArgumentException("客户编码【" + dto.getCustomerCode() + "】错误，请检查！");
                }
            }
            if (orgVoMap != null && orgVoMap.containsKey(dto.getDepartmentCode())) {
                OrgVo orgVo = orgVoMap.get(dto.getDepartmentCode());
                dto.setDepartmentName(orgVo.getOrgName());
            } else {
                throw new IllegalArgumentException("部门编码【" + dto.getDepartmentCode() + "】错误，请检查！");
            }

            dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            dto.setTenantCode(TenantUtils.getTenantCode());
        }
        log.info("结束校验关联数据校验......");

        return new ArrayList<>(dtoList);
    }

    /**
     * 获取数据实体
     *
     * @return
     */
    @Override
    public Class<ActualProfitLossImportsVo> findCrmExcelVoClass() {
        return ActualProfitLossImportsVo.class;
    }

    /**
     * 获取业务对应的模板编码，全局唯一
     *
     * @return
     */
    @Override
    public String getTemplateCode() {
        return "TPM_ACTUAL_PROFIT_LOSS_IMPORT";
    }

    /**
     * 获取业务对应的模板描述
     *
     * @return
     */
    @Override
    public String getTemplateName() {
        return "损益分析表导入";
    }
}
