package com.biz.crm.tpm.business.promotion.plan.local.service.process;

import cn.hutool.core.date.DateUtil;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductPlanningDetailsVo;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.dto.MonthDeliveryPromotionPlanAmountDto;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.dto.MonthDeliveryPromotionPlanAmountDto;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.service.ActivityApplyRulesService;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.vo.ActivityApplyRulesPromotionPlanAmountVo;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.CurrentMonthDeliveryDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.CurrentMonthDeliveryService;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.PromotionPlanService;
import com.biz.crm.tpm.business.promotion.plan.sdk.vo.CurrentMonthDeliveryImportsVo;
import com.biz.crm.tpm.business.promotion.plan.sdk.vo.CurrentMonthSaleDistributionImportsVo;
import com.biz.crm.tpm.business.sales.volume.channel.feign.feign.TpmSalesVolumeChannelServiceFeign;
import com.biz.crm.tpm.business.sales.volume.channel.sdk.dto.TpmSalesVolumeChannelDto;
import com.biz.crm.tpm.business.sales.volume.channel.sdk.vo.TpmSalesVolumeChannelVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 促销规划-当月送货
 *
 * @author: yaoyongming
 * @date: 2023/1/11 9:23
 */
@Slf4j
@Component
public class CurrentMonthDeliveryImportsProcess implements ImportProcess<CurrentMonthDeliveryImportsVo> {

    @Autowired(required = false)
    PromotionPlanService promotionPlanService;

    @Autowired(required = false)
    private TpmSalesVolumeChannelServiceFeign tpmSalesVolumeChannelServiceFeign;

    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private ProductVoService productVoService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private ActivityApplyRulesService activityApplyRulesService;

    @Autowired(required = false)
    private CurrentMonthDeliveryService currentMonthDeliveryService;

    /**
     * 单次缓存的数量
     *
     * @return
     */
    @Override
    public Integer getBatchCount() {
        return 10000;
    }

    /**
     * 是否开启先校验后导入的模式 默认false: false执行旧逻辑 true执行新逻辑(先校验再保存) <br/>
     * 新逻辑需同时实现接口 tryVerify tryConfirm（默认方法调用execute）
     *
     * @return
     */
    @Override
    public boolean importBeforeValidationFlag() {
        return Boolean.TRUE;
    }

    /**
     * 数据校验
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> tryVerify(LinkedHashMap<Integer, CurrentMonthDeliveryImportsVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Map<Integer, String> errMap = new HashMap<>();
        Validate.notNull(params.get("platformCode"), "平台编码，参数未传入");
        Validate.notNull(params.get("platformName"), "平台名称，参数未传入");
        Validate.notNull(params.get("salesOrgCode"), "组织编码，参数未传入");
        Validate.notNull(params.get("businessModelCode"), "业务模式，参数未传入");
        Validate.notNull(params.get("channelCode"), "渠道编码，参数未传入");
        Validate.notNull(params.get("businessFormatCode"), "业态编码，参数未传入");
        for (Map.Entry<Integer, CurrentMonthDeliveryImportsVo> row : data.entrySet()) {
            int rowNum = row.getKey();
            CurrentMonthDeliveryImportsVo vo = row.getValue();
            this.validateIsTrue(vo.getStartDate() != null, "开始时间，不能为空！");
            this.validateIsTrue(vo.getEndDate() != null, "结束时间，不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getCustomerCode()), "客户编码，不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getProductCode()), "产品编码，不能为空！");
            this.validateIsTrue(vo.getCombinationQuantity() != null, "组合数量，不能为空！");
            this.validateIsTrue(vo.getEstimatedSalesBox() != null, "预计销量（箱），不能为空！");
            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(rowNum, errInfo);
            }
        }
        return errMap;
    }

    /**
     * 数据处理
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, CurrentMonthDeliveryImportsVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Map<Integer, String> errMap = new HashMap<>();
        try {
            log.info("开始导入......");
            Validate.notEmpty(data, "导入数据不能为空！");

            Validate.notNull(params.get("cacheKey"), "缓存键，参数未传入");
            String cacheKey = (String) params.get("cacheKey");

            log.info("开始导入校验......");
            List<CurrentMonthDeliveryDto> importList = this.validate(data, params, errMap);
            log.info("导入校验结束......");

            if (errMap.isEmpty()) {
                log.info("开始导入缓存......");
                currentMonthDeliveryService.addListCache(cacheKey, importList);
                log.info("结束导入缓存......");
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            e.printStackTrace();
            throw e;
        }
        return errMap;
    }

    private List<CurrentMonthDeliveryDto> validate(LinkedHashMap<Integer, CurrentMonthDeliveryImportsVo> data, Map<String, Object> params, Map<Integer, String> errMap) {
        String platformCode = (String) params.get("platformCode");
        String platformName = (String) params.get("platformName");
        String salesOrgCode = (String) params.get("salesOrgCode");
        String businessModelCode = (String) params.get("businessModelCode");
        String channelCode = (String) params.get("channelCode");
        String businessFormatCode = (String) params.get("businessFormatCode");

        log.info("开始查询关联数据......");
        //查询渠道
        TpmSalesVolumeChannelDto salesVolumeChannelDto = new TpmSalesVolumeChannelDto();
        salesVolumeChannelDto.setPlatformCode(platformCode);
        Result<Page<TpmSalesVolumeChannelVo>> result = tpmSalesVolumeChannelServiceFeign.findByConditions(PageRequest.of(1, 50), salesVolumeChannelDto);
        List<TpmSalesVolumeChannelVo> channelList = result.getResult().getRecords();
        log.info("结束查询关联数据......");

        log.info("开始关联数据封装......");
        //关联数据封装
        Set<String> customerCodeSet = data.values().stream().map(CurrentMonthDeliveryImportsVo::getCustomerCode).filter(Objects::nonNull).collect(Collectors.toSet());
        Set<String> productCodeSet = data.values().stream().map(CurrentMonthDeliveryImportsVo::getProductCode).filter(Objects::nonNull).collect(Collectors.toSet());
        data.forEach((k, v) -> v.setIndex(k));

        Map<String, List<CustomerVo>> customerVoMap = new HashMap<>();
        Map<String, ProductVo> productVoMap = new HashMap<>();
        Map<String, TpmSalesVolumeChannelVo> channelVoMap = new HashMap<>();

        //关联数据查询
        if (!CollectionUtils.isEmpty(channelList)) {
            channelVoMap = channelList.stream().collect(Collectors.toMap(TpmSalesVolumeChannelVo::getSalesVolumeChannel, item -> item));
        }
        List<CustomerVo> customerVos = customerVoService.findByCustomerMdgCodes(new ArrayList<>(customerCodeSet));
        if (!CollectionUtils.isEmpty(customerVos)) {
            customerVoMap = customerVos.stream().collect(Collectors.groupingBy(e -> e.getErpCode()));
        }
        List<ProductVo> productVos = productVoService.findDetailsByIdsOrProductCodes(null, new ArrayList<>(productCodeSet));
        if (!CollectionUtils.isEmpty(productVos)) {
            productVoMap = productVos.stream().collect(Collectors.toMap(ProductVo::getProductCode, Function.identity()));
        }
        log.info("结束关联数据封装......");

        Collection<CurrentMonthDeliveryDto> dtoList = nebulaToolkitService.copyCollectionByWhiteList(data.values(), CurrentMonthDeliveryImportsVo.class, CurrentMonthDeliveryDto.class, LinkedHashSet.class, ArrayList.class);


        log.info("开始校验关联数据校验......");
        for (CurrentMonthDeliveryDto dto : dtoList) {
            try {
                if (customerVoMap.containsKey(dto.getCustomerCode())) {
                    List<CustomerVo> customerVoList = customerVoMap.get(dto.getCustomerCode());
                    CustomerVo customerVo = customerVoList.stream().filter(e -> channelCode.equals(e.getCustomerChannelCode()) && salesOrgCode.equals(e.getSalesInstitutionCode()) && businessFormatCode.equals(e.getBusinessFormatCode()))
                            .findFirst().orElse(null);
                    if (customerVo != null) {
                        dto.setCustomerName(customerVo.getCustomerName());
                        dto.setCustomerCode(customerVo.getCustomerCode());
                        dto.setErpCode(customerVo.getErpCode());
                    } else {
                        this.validateIsTrue(false, "客户编码【" + dto.getCustomerCode() + "】错误，请检查！");
                    }
                } else {
                    this.validateIsTrue(false, "客户编码【" + dto.getCustomerCode() + "】错误，请检查！");
                }
                if (productVoMap.containsKey(dto.getProductCode())) {
                    ProductVo productVo = productVoMap.get(dto.getProductCode());
                    dto.setProductName(productVo.getProductName());
                    dto.setProductBrandCode(productVo.getProductBrandCode());
                    dto.setProductBrandName(productVo.getProductBrandName());
                    dto.setTaxRate(productVo.getRate());
                    dto.setRatio(productVo.getNetWeight() == null ? BigDecimal.ZERO : productVo.getNetWeight());
                } else {
                    this.validateIsTrue(false, "产品编码【" + dto.getProductCode() + "】错误，请检查！");
                }
                if (StringUtils.isNotBlank(dto.getChannelName())) {
                    if (channelVoMap.containsKey(dto.getChannelName())) {
                        dto.setChannelCode(channelVoMap.get(dto.getChannelName()).getSalesVolumeChannelCode());
                    } else {
                        this.validateIsTrue(false, "电商渠道【" + dto.getChannelName() + "】错误，请检查！");
                    }
                }
                log.info("dto.getProductCode()=" + dto.getProductCode() + "、getErpCode="+ dto.getErpCode() + "、businessModelCode="+ businessModelCode + "、getStartDate="+ DateUtil.format(dto.getStartDate(), "yyyy-MM-dd"));
                ProductPlanningDetailsVo feeVo = null;
                try {
                    feeVo = productVoService.findByProductPlanningDetails(dto.getProductCode(), dto.getErpCode(), businessModelCode, DateUtil.format(dto.getStartDate(), "yyyy-MM-dd"));
                } catch (Exception e) {
                    validateIsTrue(false,"价格管理查询数据不不存在");
                }
                if (Objects.nonNull(feeVo)) {
                    dto.setPlatformSupplyPrice(feeVo.getPlatformSupplyPrice());
                    dto.setStandardRetailPrice(feeVo.getStandardRetailPrice());
                    dto.setRedLinePrice(feeVo.getRedLinePrice());
                    dto.setCostPrice(feeVo.getCostPrice());
                }

                this.validateIsTrue(dto.getPlatformSupplyPrice() != null, "产品【"+ dto.getProductCode() +"】的标准零售价，不能为空！");
                this.validateIsTrue(dto.getStandardRetailPrice() != null, "产品【"+ dto.getProductCode() +"】平台供货价，不能为空！");
                this.validateIsTrue(dto.getCostPrice() != null, "产品【"+ dto.getProductCode() +"】成本价，不能为空！");

                String errInfo = this.validateGetErrorInfo();
                if (errInfo != null) {
                    errMap.put(dto.getIndex(), errInfo);
                    continue;
                }

                log.info("开始自动计算......");
                //自动计算
                log.info("开始内部自动计算......");
                currentMonthDeliveryService.calculationFee(dto, platformName);
                log.info("结束内部自动计算......");
                log.info("开始申请规则接口自动计算......");
                MonthDeliveryPromotionPlanAmountDto amountDto = nebulaToolkitService.copyObjectByWhiteList(dto, MonthDeliveryPromotionPlanAmountDto.class, LinkedHashSet.class, ArrayList.class);
                amountDto.setPlatformCode(platformCode);
                amountDto.setSalesOrgCode(salesOrgCode);
                ActivityApplyRulesPromotionPlanAmountVo amountVo = activityApplyRulesService.calAmountMonthDelivery(amountDto);
                log.info("结束申请规则接口自动计算......");
                log.info("结束自动计算......");

                BeanUtils.copyProperties(amountVo, dto);
                dto.setId(UUID.randomUUID().toString().replace("-", ""));
                dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                dto.setTenantCode(TenantUtils.getTenantCode());
                dto.setEndDate(DateUtil.parse(DateUtil.format(dto.getEndDate(), "yyyy-MM-dd") + " 23:59:59", "yyyy-MM-dd HH:mm:ss"));
            } catch (Exception e) {
                log.error(e.getMessage(), e);
                e.printStackTrace();
            }

        }
        log.info("结束校验关联数据校验......");

        return new ArrayList<>(dtoList);
    }

    /**
     * 获取数据实体
     *
     * @return
     */
    @Override
    public Class<CurrentMonthDeliveryImportsVo> findCrmExcelVoClass() {
        return CurrentMonthDeliveryImportsVo.class;
    }

    /**
     * 获取业务对应的模板编码，全局唯一
     *
     * @return
     */
    @Override
    public String getTemplateCode() {
        return "TPM_PP_CURRENT_MONTH_DELIVERY_IMPORT";
    }

    /**
     * 获取业务对应的模板描述
     *
     * @return
     */
    @Override
    public String getTemplateName() {
        return "促销规划-当月送货导入";
    }
}
