package com.biz.crm.tpm.business.promotion.plan.local.service.process;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductPlanningDetailsVo;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.dto.MonthSalePromotionPlanAmountDto;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.service.ActivityApplyRulesService;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.vo.ActivityApplyRulesPromotionPlanAmountVo;
import com.biz.crm.tpm.business.promotion.plan.sdk.constant.PromotionPlanConstant;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.CurrentMonthSaleDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.GeneralExpensesDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.CurrentMonthSaleService;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.GeneralExpensesService;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.PromotionPlanService;
import com.biz.crm.tpm.business.promotion.plan.sdk.vo.CurrentMonthSaleImportsVo;
import com.biz.crm.tpm.business.sales.volume.channel.feign.feign.TpmSalesVolumeChannelServiceFeign;
import com.biz.crm.tpm.business.sales.volume.channel.sdk.dto.TpmSalesVolumeChannelDto;
import com.biz.crm.tpm.business.sales.volume.channel.sdk.vo.TpmSalesVolumeChannelVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 促销规划-当月销售
 *
 * @author: yaoyongming
 * @date: 2023/1/11 9:26
 */
@Slf4j
@Component
public class CurrentMonthSaleImportsProcess implements ImportProcess<CurrentMonthSaleImportsVo> {

    @Autowired(required = false)
    PromotionPlanService promotionPlanService;

    @Autowired(required = false)
    private DictDataVoService dictDataVoService;

    @Autowired(required = false)
    private TpmSalesVolumeChannelServiceFeign tpmSalesVolumeChannelServiceFeign;
    
    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private ProductVoService productVoService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private ActivityApplyRulesService activityApplyRulesService;

    @Autowired(required = false)
    private CurrentMonthSaleService currentMonthSaleService;

    @Autowired(required = false)
    private GeneralExpensesService generalExpensesService;

    /**
     * 单次缓存的数量
     *
     * @return
     */
    @Override
    public Integer getBatchCount() {
        return 10000;
    }

    /**
     * 是否开启先校验后导入的模式 默认false: false执行旧逻辑 true执行新逻辑(先校验再保存) <br/>
     * 新逻辑需同时实现接口 tryVerify tryConfirm（默认方法调用execute）
     *
     * @return
     */
    @Override
    public boolean importBeforeValidationFlag() {
        return Boolean.TRUE;
    }

    /**
     * 数据校验
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> tryVerify(LinkedHashMap<Integer, CurrentMonthSaleImportsVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Map<Integer, String> errMap = new HashMap<>();
        Validate.notNull(params.get("platformCode"), "平台编码，参数未传入");
        Validate.notNull(params.get("platformName"), "平台名称，参数未传入");
        Validate.notNull(params.get("salesOrgCode"), "组织编码，参数未传入");
        Validate.notNull(params.get("businessModelCode"), "业务模式，参数未传入");
        Validate.notNull(params.get("channelCode"), "渠道编码，参数未传入");
        Validate.notNull(params.get("businessFormatCode"), "业态编码，参数未传入");
        for (Map.Entry<Integer, CurrentMonthSaleImportsVo> row : data.entrySet()) {
            int rowNum = row.getKey();
            CurrentMonthSaleImportsVo vo = row.getValue();
            this.validateIsTrue(vo.getStartDate() != null, "开始时间，不能为空！");
            this.validateIsTrue(vo.getEndDate() != null, "结束时间，不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getCustomerCode()), "客户编码，不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getProductCode()), "产品编码，不能为空！");
            this.validateIsTrue(vo.getActivityBasePrice() != null, "活动底价，不能为空！");
            this.validateIsTrue(vo.getCombinationQuantity() != null, "组合数量，不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getPromotionTypeCode()), "促销类型，不能为空！");
            this.validateIsTrue(vo.getEstimatedSalesBox() != null, "预计销量（箱），不能为空！");
            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(rowNum, errInfo);
            }
        }
        return errMap;
    }

    /**
     * 数据处理
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, CurrentMonthSaleImportsVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Map<Integer, String> errMap = new HashMap<>();
        try {
            log.info("开始导入......");
            Validate.notEmpty(data, "导入数据不能为空！");

            Validate.notNull(params.get("cacheKey"), "缓存键，参数未传入");
            String cacheKey = (String) params.get("cacheKey");

            log.info("开始导入校验......");
            List<CurrentMonthSaleDto> importList = this.validate(data, params, errMap);
            log.info("导入校验结束......");

            if (errMap.isEmpty()) {
                log.info("开始导入缓存......");
                currentMonthSaleService.addListCache(cacheKey, importList);
                log.info("结束导入缓存......");
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            e.printStackTrace();
            throw e;
        }

        return errMap;
    }

    private List<CurrentMonthSaleDto> validate(LinkedHashMap<Integer, CurrentMonthSaleImportsVo> data, Map<String, Object> params, Map<Integer, String> errMap) {
        String platformCode = (String) params.get("platformCode");
        String platformName = (String) params.get("platformName");
        String salesOrgCode = (String) params.get("salesOrgCode");
        String businessModelCode = (String) params.get("businessModelCode");
        String channelCode = (String) params.get("channelCode");
        String businessFormatCode = (String) params.get("businessFormatCode");
        String cacheKeyGeneral = params.get("cacheKeyGeneral") != null ? (String) params.get("cacheKeyGeneral") : null;
        String cacheKey = (String) params.get("cacheKey");


        log.info("开始查询关联数据......");
        // 查询数据字典
        List<DictDataVo> dictDataVos = dictDataVoService.findByDictTypeCode(PromotionPlanConstant.TPM_PROMOTION_TYPE);
        //查询渠道
        TpmSalesVolumeChannelDto salesVolumeChannelDto = new TpmSalesVolumeChannelDto();
        salesVolumeChannelDto.setPlatformCode(platformCode);
        Result<Page<TpmSalesVolumeChannelVo>> result = tpmSalesVolumeChannelServiceFeign.findByConditions(PageRequest.of(1, 50), salesVolumeChannelDto);
        List<TpmSalesVolumeChannelVo> channelList = result.getResult().getRecords();
        //费用申请
        List<GeneralExpensesDto> expensesDtoList = StringUtils.isNotBlank(cacheKeyGeneral) ? generalExpensesService.findCacheList(cacheKeyGeneral) : new ArrayList<>();
        //当月销售
        List<CurrentMonthSaleDto> salesList = currentMonthSaleService.findCacheList(cacheKey);
        log.info("结束查询关联数据......");


        log.info("开始关联数据封装......");
        //关联数据封装
        Set<String> customerCodeSet = data.values().stream().map(CurrentMonthSaleImportsVo::getCustomerCode).filter(Objects::nonNull).collect(Collectors.toSet());
        Set<String> productCodeSet = data.values().stream().map(CurrentMonthSaleImportsVo::getProductCode).filter(Objects::nonNull).collect(Collectors.toSet());
        data.forEach((k, v) -> v.setIndex(k));

        Map<String, List<CustomerVo>> customerVoMap = new HashMap<>();
        Map<String, ProductVo> productVoMap = new HashMap<>();
        Map<String, TpmSalesVolumeChannelVo> channelVoMap = new HashMap<>();

        //关联数据查询
        List<CustomerVo> customerVos = customerVoService.findByCustomerMdgCodes(new ArrayList<>(customerCodeSet));
        if (!CollectionUtils.isEmpty(customerVos)) {
            customerVoMap = customerVos.stream().collect(Collectors.groupingBy(e -> e.getErpCode()));
        } else {
            throw new IllegalArgumentException("未找到任意客户，请检查！");
        }
        List<ProductVo> productVos = productVoService.findDetailsByIdsOrProductCodes(null, new ArrayList<>(productCodeSet));
        if (!CollectionUtils.isEmpty(productVos)) {
            productVoMap = productVos.stream().collect(Collectors.toMap(ProductVo::getProductCode, Function.identity()));
        } else {
            throw new IllegalArgumentException("未找到任意产品，请检查！");
        }
        log.info("结束关联数据封装......");

        Collection<CurrentMonthSaleDto> dtoList = nebulaToolkitService.copyCollectionByWhiteList(data.values(), CurrentMonthSaleImportsVo.class, CurrentMonthSaleDto.class, LinkedHashSet.class, ArrayList.class);

        log.info("开始校验关联数据校验......");
        //关联数据校验
        for (CurrentMonthSaleDto dto : dtoList) {
            try {
                if (customerVoMap.containsKey(dto.getCustomerCode())) {
                    List<CustomerVo> customerVoList = customerVoMap.get(dto.getCustomerCode());
                    CustomerVo customerVo = customerVoList.stream().filter(e -> channelCode.equals(e.getCustomerChannelCode()) && salesOrgCode.equals(e.getSalesInstitutionCode()) && businessFormatCode.equals(e.getBusinessFormatCode()))
                            .findFirst().orElse(null);
                    if (customerVo != null) {
                        dto.setCustomerName(customerVo.getCustomerName());
                        dto.setCustomerCode(customerVo.getCustomerCode());
                        dto.setErpCode(customerVo.getErpCode());
                    } else {
                        this.validateIsTrue(false, "客户编码【" + dto.getCustomerCode() + "】错误，请检查！");
                    }
                } else {
                    this.validateIsTrue(false, "客户编码【" + dto.getCustomerCode() + "】错误，请检查！");
                }
                if (productVoMap.containsKey(dto.getProductCode())) {
                    ProductVo productVo = productVoMap.get(dto.getProductCode());
                    dto.setProductName(productVo.getProductName());
                    dto.setProductBrandCode(productVo.getProductBrandCode());
                    dto.setProductBrandName(productVo.getProductBrandName());
                    dto.setTaxRate(productVo.getRate());
                    dto.setRatio(productVo.getNetWeight() == null ? BigDecimal.ZERO : productVo.getNetWeight());
                } else {
                    this.validateIsTrue(false, "产品编码【" + dto.getProductCode() + "】错误，请检查！");
                }
                DictDataVo promotionType = dictDataVos.stream().filter(e -> e.getDictValue().equals(dto.getPromotionTypeCode())).findFirst().orElse(null);
                if (promotionType != null) {
                    dto.setPromotionTypeCode(promotionType.getDictCode());
                    dto.setPromotionTypeName(promotionType.getDictValue());
                } else {
                    this.validateIsTrue(false, "促销类型【" + dto.getPromotionTypeCode() + "】错误，请检查！");
                }
                if (StringUtils.isNotBlank(dto.getChannelName())) {
                    if (channelVoMap.containsKey(dto.getChannelName())) {
                        dto.setChannelCode(channelVoMap.get(dto.getChannelName()).getSalesVolumeChannelCode());
                    } else {
                        this.validateIsTrue(false, "电商渠道【" + dto.getChannelName() + "】错误，请检查！");
                    }
                }
                log.info("dto.getProductCode()=" + dto.getProductCode() + "、getErpCode="+ dto.getErpCode() + "、businessModelCode="+ businessModelCode + "、getStartDate="+ DateUtil.format(dto.getStartDate(), "yyyy-MM-dd"));
                ProductPlanningDetailsVo feeVo = null;
                try {
                    feeVo = productVoService.findByProductPlanningDetails(dto.getProductCode(), dto.getErpCode(), businessModelCode, DateUtil.format(dto.getStartDate(), "yyyy-MM-dd"));
                } catch (Exception e) {
                    this.validateIsTrue(false, "价格管理查询数据不不存在");
                }
                if (Objects.nonNull(feeVo)) {
                    dto.setPlatformSupplyPrice(feeVo.getPlatformSupplyPrice());
                    dto.setStandardRetailPrice(feeVo.getStandardRetailPrice());
                    dto.setRedLinePrice(feeVo.getRedLinePrice());
                    dto.setCostPrice(feeVo.getCostPrice());
                }

                this.validateIsTrue(dto.getStandardRetailPrice() != null, "产品【"+ dto.getProductCode() +"】的标准零售价，不能为空！");
                this.validateIsTrue(dto.getPlatformSupplyPrice() != null, "产品【"+ dto.getProductCode() +"】平台供货价，不能为空！");
                this.validateIsTrue(dto.getCostPrice() != null, "产品【"+ dto.getProductCode() +"】成本价，不能为空！");

                String errInfo = this.validateGetErrorInfo();
                if (errInfo != null) {
                    errMap.put(dto.getIndex(), errInfo);
                    continue;
                }

                log.info("开始自动计算......");
                //自动计算
                log.info("开始内部自动计算......");
                //预计折前销售额（元）：自动计算：组合数量*平台供货价*预计销量（箱）
                dto.setEstimatedAmountBefore(dto.getCombinationQuantity().multiply(dto.getPlatformSupplyPrice()).multiply(dto.getEstimatedSalesBox()).setScale(2, RoundingMode.HALF_UP));
                //京东采购返（元）	"IF平台=“京东商城”才计算该字段，逻辑为：平台供货价*预计销量（箱）*组合数量*0.06"
                if ("京东商城".equals(platformName)) {
                    dto.setJd(dto.getEstimatedAmountBefore().multiply(new BigDecimal("0.06")).setScale(2, RoundingMode.HALF_UP));
                }
                currentMonthSaleService.calculationFee(dto, businessModelCode);
                log.info("结束内部自动计算......");
                log.info("结束自动计算......");


                dto.setId(UUID.randomUUID().toString().replace("-", ""));
                dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                dto.setTenantCode(TenantUtils.getTenantCode());
                dto.setEndDate(cn.hutool.core.date.DateUtil.parse(cn.hutool.core.date.DateUtil.format(dto.getEndDate(), "yyyy-MM-dd") + " 23:59:59", "yyyy-MM-dd HH:mm:ss"));
            } catch (Exception e) {
                log.error(e.getMessage(), e);
                e.printStackTrace();
            }

        }
        salesList.addAll(dtoList);
        for (CurrentMonthSaleDto dto : dtoList) {
            log.info("开始申请规则接口自动计算......");
            MonthSalePromotionPlanAmountDto amountDto = nebulaToolkitService.copyObjectByWhiteList(dto, MonthSalePromotionPlanAmountDto.class, LinkedHashSet.class, ArrayList.class);
            amountDto.setPlatformCode(platformCode);
            amountDto.setSalesOrgCode(salesOrgCode);
            amountDto.setExpensesDtoList(expensesDtoList);
            amountDto.setCurrentMonthSaleDtoList(salesList);
            ActivityApplyRulesPromotionPlanAmountVo amountVo = activityApplyRulesService.calAmountMonthSale(amountDto);
            BeanUtils.copyProperties(amountVo, dto);
            resetRulesAmount(dto, amountVo, dto.getPromotionTypeName());
            log.info("结束申请规则接口自动计算......");
        }
        log.info("结束校验关联数据校验......");

        return new ArrayList<>(dtoList);
    }

    /**
     * 重置规则金额
     *
     * @param dto
     */
    private void resetRulesAmount(CurrentMonthSaleDto dto, ActivityApplyRulesPromotionPlanAmountVo amountVo, String promotionTypeName) {
        dto.setProductPromotion(null);
        dto.setLargeDateProcess(null);
        dto.setNewCustomer(null);
        dto.setOldCustomer(null);
        dto.setTiktokMaster(null);
        dto.setUneven(null);
        dto.setHighTurnover(null);

        if (StringUtils.isNotBlank(promotionTypeName)) {
            switch (promotionTypeName) {
                case "产品促销":
                    dto.setProductPromotion(amountVo.getProductPromotion());
                    break;
                case "大日期处理":
                    dto.setLargeDateProcess(amountVo.getLargeDateProcess());
                    break;
                case "新客专享":
                    dto.setNewCustomer(amountVo.getNewCustomer());
                    break;
                case "老客留存":
                    dto.setOldCustomer(amountVo.getOldCustomer());
                    break;
                case "达人佣金（抖音）":
                    dto.setProductPromotion(amountVo.getProductPromotion());
                    dto.setTiktokMaster(amountVo.getTiktokMaster());
                    break;
                case "仓间不均":
                    dto.setUneven(amountVo.getUneven());
                    break;
                case "高周转":
                    dto.setHighTurnover(amountVo.getHighTurnover());
                    break;
                default:
                    throw new IllegalArgumentException("未知的产品促销名称，请检查");

            }
        }
    }

    /**
     * 获取数据实体
     *
     * @return
     */
    @Override
    public Class<CurrentMonthSaleImportsVo> findCrmExcelVoClass() {
        return CurrentMonthSaleImportsVo.class;
    }

    /**
     * 获取业务对应的模板编码，全局唯一
     *
     * @return
     */
    @Override
    public String getTemplateCode() {
        return "TPM_PP_CURRENT_MONTH_SALE_IMPORT";
    }

    /**
     * 获取业务对应的模板描述
     *
     * @return
     */
    @Override
    public String getTemplateName() {
        return "促销规划-当月销售导入";
    }

    /**
     * 获取业务编码(一个业务场景多模板情况,不设置默认取模板编码)
     *
     * @return
     */
    @Override
    public String getBusinessCode() {
        return "TPM_PP_CURRENT_MONTH_SALE_IMPORT";
    }

    /**
     * 业务名称(一个业务场景多模板情况,不设置默认取模板名称)
     *
     * @return
     */
    @Override
    public String getBusinessName() {
        return "促销规划-当月销售导入";
    }
}
