package com.biz.crm.tpm.business.promotion.plan.local.service.process;

import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.common.ie.sdk.excel.process.ExportProcess;
import com.biz.crm.common.ie.sdk.vo.ExportTaskProcessVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mn.common.base.constant.CommonConstant;
import com.biz.crm.tpm.business.promotion.plan.sdk.constant.PromotionPlanConstant;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.GeneralExpensesService;
import com.biz.crm.tpm.business.promotion.plan.sdk.vo.GeneralExpensesExportsVo;
import com.biz.crm.tpm.business.promotion.plan.sdk.vo.GeneralExpensesVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import java.util.*;

/**
 * 费用申请导出
 *
 * @author: yaoyongming
 * @date: 2023/2/1 14:14
 */
@Slf4j
@Component
public class GeneralExpensesExportsProcess implements ExportProcess<GeneralExpensesExportsVo> {

    @Autowired(required = false)
    private GeneralExpensesService generalExpensesService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private DictDataVoService dictDataVoService;


    @Override
    public Integer getPageSize() {
        return CommonConstant.IE_EXPORT_PAGE_SIZE;
    }


    /**
     * 获取当前任务需要导出数据总记录数(用于拆分子任务)
     *
     * @param params 自定义参数
     * @return
     */
    @Override
    public Integer getTotal(Map<String, Object> params) {
        log.info("=========获取费用申请导出总记录数");
        Validate.notNull(params.get("cacheKey"), "缓存键，参数未传入");
        log.info("=========cacheKey：" + params.get("cacheKey"));
        Integer total = generalExpensesService.getTotal((String) params.get("cacheKey"));
        Validate.isTrue(total < CommonConstant.IE_EXPORT_MAX_TOTAL, "导出时，" +
                "单次最大导出[" + CommonConstant.IE_EXPORT_MAX_TOTAL + "]条,请输入更多查询条件!!");
        log.info("=========total：" + total);
        return total;
    }

    /**
     * 数据处理（最终处理）
     *
     * @param vo     导出任务处理器实体信息
     * @param params 扩展参数
     * @return
     */
    @Override
    public JSONArray getData(ExportTaskProcessVo vo, Map<String, Object> params) {
        log.info("=========开始费用申请导出");
        Validate.notNull(params.get("cacheKey"), "缓存键，参数未传入");
        String cacheKey = (String) params.get("cacheKey");
        Pageable pageable = PageRequest.of(vo.getPageNo() + 1, this.getPageSize());
        Page<GeneralExpensesVo> cachePage = generalExpensesService.findCachePageList(pageable, cacheKey, params.get("promotionPlanCode") == null ? null : (String) params.get("promotionPlanCode"));
        log.info("=========获取费用申请导出总记录数：" + cachePage.getSize());
        if (cachePage.getTotal() == 0) {
            return new JSONArray();
        }
        List<GeneralExpensesVo> records = cachePage.getRecords();
        records.forEach(e -> {
            e.setStartDateStr(DateUtil.format(e.getStartDate(), "yyyy-MM-dd"));
            e.setEndDateStr(DateUtil.format(e.getEndDate(), "yyyy-MM-dd HH:mm:ss"));
        });
        Collection<GeneralExpensesExportsVo> exportsVos = nebulaToolkitService.copyCollectionByWhiteList(records, GeneralExpensesVo.class, GeneralExpensesExportsVo.class, LinkedHashSet.class, ArrayList.class);
        //调整
        adjustData(exportsVos);
        log.info("=========结束费用申请导出");
        return JSON.parseArray(JSON.toJSONString(exportsVos));
    }

    /**
     * 调整数据
     *
     * @param data
     */
    private void adjustData(Collection<GeneralExpensesExportsVo> data) {
        //  转换数据字典值
        Map<String, List<DictDataVo>> mapDict = this.dictDataVoService.findByDictTypeCodeList(Lists.newArrayList(PromotionPlanConstant.TPM_PROMOTION_TYPE, PromotionPlanConstant.TPM_AUDIT_TYPE));
        for (GeneralExpensesExportsVo vo : data) {
            vo.setPromotionTypeCode(this.findDictValue(mapDict, PromotionPlanConstant.TPM_PROMOTION_TYPE, String.valueOf(vo.getPromotionTypeCode())));
            vo.setAuditConditionType(this.findDictValue(mapDict, PromotionPlanConstant.TPM_AUDIT_TYPE, String.valueOf(vo.getAuditConditionType())));
        }
    }

    /**
     * 获取字典值
     *
     * @param mapDict
     * @param dictTypeCode
     * @param code
     * @return
     */
    private String findDictValue(Map<String, List<DictDataVo>> mapDict, String dictTypeCode, String code) {
        if (Objects.isNull(mapDict) || StringUtils.isBlank(dictTypeCode) || StringUtils.isBlank(code)) {
            return StringUtils.EMPTY;
        }
        final List<DictDataVo> vos = mapDict.get(dictTypeCode);
        if (org.apache.commons.collections.CollectionUtils.isEmpty(vos)) {
            return StringUtils.EMPTY;
        }
        final Optional<String> first =
                vos.stream()
                        .filter(a -> a.getDictCode().equals(code))
                        .map(DictDataVo::getDictValue)
                        .findFirst();
        return first.orElse(StringUtils.EMPTY);
    }

    /**
     * 导出业务编码
     *
     * @return
     */
    @Override
    public String getBusinessCode() {
        return "TPM_PP_GENERAL_EXPENSES_EXPORT";
    }

    /**
     * 导出业务名称
     *
     * @return
     */
    @Override
    public String getBusinessName() {
        return "促销规划-费用申请导出";
    }

    /**
     * 获取对象转换实体
     *
     * @return
     */
    @Override
    public Class<GeneralExpensesExportsVo> findCrmExcelVoClass() {
        return GeneralExpensesExportsVo.class;
    }
}
