package com.biz.crm.tpm.business.promotion.plan.local.service.process;

import cn.hutool.core.date.DateUtil;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.tpm.business.activity.form.sdk.service.ActivityFormService;
import com.biz.crm.tpm.business.activity.form.sdk.vo.ActivityFormVo;
import com.biz.crm.tpm.business.activity.type.sdk.service.ActivityTypeService;
import com.biz.crm.tpm.business.activity.type.sdk.vo.ActivityTypeVo;
import com.biz.crm.tpm.business.month.budget.sdk.service.MonthBudgetService;
import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetBusinessPolicyQueryVo;
import com.biz.crm.tpm.business.promotion.plan.sdk.constant.PromotionPlanConstant;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.GeneralExpensesDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.GeneralExpensesService;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.PromotionPlanService;
import com.biz.crm.tpm.business.promotion.plan.sdk.vo.GeneralExpensesImportsVo;
import com.biz.crm.tpm.business.sales.volume.channel.feign.feign.TpmSalesVolumeChannelServiceFeign;
import com.biz.crm.tpm.business.sales.volume.channel.sdk.dto.TpmSalesVolumeChannelDto;
import com.biz.crm.tpm.business.sales.volume.channel.sdk.vo.TpmSalesVolumeChannelVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 促销规划-费用申请
 *
 * @author: yaoyongming
 * @date: 2023/1/11 9:26
 */
@Slf4j
@Component
public class GeneralExpensesImportsProcess implements ImportProcess<GeneralExpensesImportsVo> {
    @Autowired(required = false)
    PromotionPlanService promotionPlanService;

    @Autowired(required = false)
    private TpmSalesVolumeChannelServiceFeign tpmSalesVolumeChannelServiceFeign;

    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private ProductVoService productVoService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private GeneralExpensesService generalExpensesService;

    @Autowired(required = false)
    private MonthBudgetService monthBudgetService;

    @Autowired(required = false)
    private DictDataVoService dictDataVoService;

    @Autowired(required = false)
    private ActivityTypeService activityTypeService;

    @Autowired(required = false)
    private ActivityFormService activityFormService;


    /**
     * 单次缓存的数量
     *
     * @return
     */
    @Override
    public Integer getBatchCount() {
        return 10000;
    }

    /**
     * 是否开启先校验后导入的模式 默认false: false执行旧逻辑 true执行新逻辑(先校验再保存) <br/>
     * 新逻辑需同时实现接口 tryVerify tryConfirm（默认方法调用execute）
     *
     * @return
     */
    @Override
    public boolean importBeforeValidationFlag() {
        return Boolean.TRUE;
    }

    /**
     * 数据校验
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> tryVerify(LinkedHashMap<Integer, GeneralExpensesImportsVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Map<Integer, String> errMap = new HashMap<>();
        Validate.notEmpty(data, "导入数据不能为空！");

        Validate.notNull(params.get("cacheKey"), "缓存键，参数未传入");
        Validate.notNull(params.get("platformCode"), "平台编码，参数未传入");
        Validate.notNull(params.get("platformName"), "平台名称，参数未传入");
        Validate.notNull(params.get("salesOrgCode"), "组织编码，参数未传入");
        Validate.notNull(params.get("businessModelCode"), "业务模式，参数未传入");
        Validate.notNull(params.get("channelCode"), "渠道编码，参数未传入");
        Validate.notNull(params.get("businessFormatCode"), "业态编码，参数未传入");
        for (Map.Entry<Integer, GeneralExpensesImportsVo> row : data.entrySet()) {
            int rowNum = row.getKey();
            GeneralExpensesImportsVo vo = row.getValue();
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getActivityTypeCode()), "活动类型编码，不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getActivityFormCode()), "活动形式编码，不能为空！");
            this.validateIsTrue(vo.getStartDate() != null, "开始时间，不能为空！");
            this.validateIsTrue(vo.getEndDate() != null, "结束时间，不能为空！");
            this.validateIsTrue(vo.getCustomerCode() != null, "客户编码，不能为空！");
            this.validateIsTrue(vo.getApplyAmount() != null, "申请金额，不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getAuditWayName()), "结案方式，不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getAuditConditionType()), "核销条件，不能为空！");
            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(rowNum, errInfo);
            }
        }
        return errMap;
    }

    /**
     * 数据处理
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, GeneralExpensesImportsVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Map<Integer, String> errMap = new HashMap<>();
        try {
            log.info("开始导入......");
            String cacheKey = (String) params.get("cacheKey");

            log.info("开始导入校验......");
            List<GeneralExpensesDto> importList = this.validate(data, params, errMap);
            log.info("导入校验结束......");

            if (errMap.isEmpty()) {
                log.info("开始导入缓存......");
                generalExpensesService.addListCache(cacheKey, importList);
                log.info("结束导入缓存......");
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            e.printStackTrace();
            throw e;
        }
        return errMap;
    }


    private List<GeneralExpensesDto> validate(LinkedHashMap<Integer, GeneralExpensesImportsVo> data, Map<String, Object> params, Map<Integer, String> errMap) {
        String platformCode = (String) params.get("platformCode");
        String salesOrgCode = (String) params.get("salesOrgCode");
        String channelCode = (String) params.get("channelCode");
        String businessFormatCode = (String) params.get("businessFormatCode");

        log.info("开始查询关联数据......");
        // 查询数据字典
        Map<String, List<DictDataVo>> dictDataMap = dictDataVoService.findByDictTypeCodeList(Arrays.asList(PromotionPlanConstant.TPM_PROMOTION_TYPE, PromotionPlanConstant.TPM_AUDIT_TYPE, PromotionPlanConstant.TPM_AUDIT_FORM));
        //查询渠道
        TpmSalesVolumeChannelDto salesVolumeChannelDto = new TpmSalesVolumeChannelDto();
        salesVolumeChannelDto.setPlatformCode(platformCode);
        Result<Page<TpmSalesVolumeChannelVo>> result = tpmSalesVolumeChannelServiceFeign.findByConditions(PageRequest.of(1, 500), salesVolumeChannelDto);
        List<TpmSalesVolumeChannelVo> channelList = result.getResult().getRecords();
        log.info("结束查询关联数据......");

        log.info("开始关联数据封装......");
        //关联数据封装
        Set<String> customerCodeSet = new HashSet<>();
        Set<String> productCodeSet = new HashSet<>();
        Set<String> monthBudgetCodes = new HashSet<>();
        Set<String> activityTypeCodes = new HashSet<>();
        Set<String> activityFormCodes = new HashSet<>();
        for (GeneralExpensesImportsVo importVo : data.values()) {
            if (StringUtils.isNotBlank(importVo.getCustomerCode())) {
                customerCodeSet.add(importVo.getCustomerCode());
            }
            if (StringUtils.isNotBlank(importVo.getProductCode())) {
                productCodeSet.add(importVo.getProductCode());
            }
            if (StringUtils.isNotBlank(importVo.getMonthBudgetCode())) {
                monthBudgetCodes.add(importVo.getMonthBudgetCode());
            }
            if (StringUtils.isNotBlank(importVo.getActivityTypeCode())) {
                activityTypeCodes.add(importVo.getActivityTypeCode());
            }
            if (StringUtils.isNotBlank(importVo.getActivityFormCode())) {
                activityFormCodes.add(importVo.getActivityFormCode());
            }
        }
        data.forEach((k, v) -> v.setIndex(k));

        Map<String, List<CustomerVo>> customerVoMap = new HashMap<>();
        Map<String, ProductVo> productVoMap = new HashMap<>();
        Map<String, TpmSalesVolumeChannelVo> channelVoMap = new HashMap<>();
        Map<String, MonthBudgetBusinessPolicyQueryVo> monthBudgetMap = new HashMap<>();
        Map<String, ActivityTypeVo> activityTypeMap = new HashMap<>();
        Map<String, ActivityFormVo> activityFormMap = new HashMap<>();

        //关联数据查询
        if (!org.springframework.util.CollectionUtils.isEmpty(channelList)) {
            channelVoMap = channelList.stream().collect(Collectors.toMap(TpmSalesVolumeChannelVo::getSalesVolumeChannel, item -> item));
        }
        List<CustomerVo> customerVos = customerVoService.findByCustomerMdgCodes(new ArrayList<>(customerCodeSet));
        if (!org.springframework.util.CollectionUtils.isEmpty(customerVos)) {
            customerVoMap = customerVos.stream().collect(Collectors.groupingBy(e -> e.getErpCode()));
        }
        if (CollectionUtils.isNotEmpty(productCodeSet)) {
            List<ProductVo> productVos = productVoService.findDetailsByIdsOrProductCodes(null, new ArrayList<>(productCodeSet));
            if (!CollectionUtils.isEmpty(productVos)) {
                productVoMap = productVos.stream().collect(Collectors.toMap(ProductVo::getProductCode, Function.identity()));
            }
        }
        if (CollectionUtils.isNotEmpty(monthBudgetCodes)) {
            List<MonthBudgetBusinessPolicyQueryVo> monthBudgetList = monthBudgetService.findBusinessPolicyByCodes(new ArrayList<>(monthBudgetCodes));
            if (!CollectionUtils.isEmpty(monthBudgetList)) {
                monthBudgetMap = monthBudgetList.stream().collect(Collectors.toMap(MonthBudgetBusinessPolicyQueryVo::getMonthBudgetCode, Function.identity()));
            }
        }
        List<ActivityTypeVo> activityTypeList = activityTypeService.findByCodes(new ArrayList<>(activityTypeCodes));
        if (!CollectionUtils.isEmpty(activityTypeList)) {
            activityTypeMap = activityTypeList.stream().collect(Collectors.toMap(ActivityTypeVo::getActivityTypeCode, Function.identity()));
        }
        List<ActivityFormVo> activityFormList = activityFormService.findActivityFormByCode(activityFormCodes);
        if (!CollectionUtils.isEmpty(activityFormList)) {
            activityFormMap = activityFormList.stream().collect(Collectors.toMap(ActivityFormVo::getActivityFormCode, Function.identity()));
        }
        log.info("结束关联数据封装......");

        Collection<GeneralExpensesDto> dtoList = nebulaToolkitService.copyCollectionByWhiteList(data.values(), GeneralExpensesImportsVo.class, GeneralExpensesDto.class, LinkedHashSet.class, ArrayList.class);


        log.info("开始校验关联数据校验......");
        for (GeneralExpensesDto dto : dtoList) {
            try {
                if (customerVoMap.containsKey(dto.getCustomerCode())) {
                    List<CustomerVo> customerVoList = customerVoMap.get(dto.getCustomerCode());
                    CustomerVo customerVo = customerVoList.stream().filter(e -> channelCode.equals(e.getCustomerChannelCode()) && salesOrgCode.equals(e.getSalesInstitutionCode()) && businessFormatCode.equals(e.getBusinessFormatCode()))
                            .findFirst().orElse(null);
                    if (customerVo != null) {
                        dto.setCustomerName(customerVo.getCustomerName());
                        dto.setCustomerCode(customerVo.getCustomerCode());
                        dto.setErpCode(customerVo.getErpCode());
                    } else {
                        this.validateIsTrue(false, "客户编码【" + dto.getCustomerCode() + "】错误，请检查！");
                    }
                } else {
                    this.validateIsTrue(false, "客户编码【" + dto.getCustomerCode() + "】错误，请检查！");
                }
                if (StringUtils.isNotBlank(dto.getProductCode())) {
                    if (productVoMap.containsKey(dto.getProductCode())) {
                        ProductVo productVo = productVoMap.get(dto.getProductCode());
                        dto.setProductName(productVo.getProductName());
                        dto.setProductBrandCode(productVo.getProductBrandCode());
                        dto.setProductBrandName(productVo.getProductBrandName());
                    } else {
                        this.validateIsTrue(false, "产品编码【" + dto.getCustomerCode() + "】错误，请检查！");
                    }
                }
                if (StringUtils.isNotBlank(dto.getChannelName())) {
                    if (channelVoMap.containsKey(dto.getChannelName())) {
                        dto.setChannelCode(channelVoMap.get(dto.getChannelName()).getSalesVolumeChannelCode());
                    } else {
                        this.validateIsTrue(false, "电商渠道【" + dto.getChannelName() + "】错误，请检查！");
                    }
                }
                if (StringUtils.isNotBlank(dto.getMonthBudgetCode())) {
                    if (monthBudgetMap.containsKey(dto.getMonthBudgetCode())) {
                        MonthBudgetBusinessPolicyQueryVo monthBudget = monthBudgetMap.get(dto.getMonthBudgetCode());
                        dto.setBudgetItemCode1(monthBudget.getFirstBudgetItemCode());
                        dto.setBudgetItemCode2(monthBudget.getSecondBudgetItemCode());
                        dto.setBudgetItemCode3(monthBudget.getBudgetItemCode());
                        dto.setBudgetItemName1(monthBudget.getFirstBudgetItemName());
                        dto.setBudgetItemName2(monthBudget.getSecondBudgetItemName());
                        dto.setBudgetItemName3(monthBudget.getBudgetItemName());
                    } else {
                        this.validateIsTrue(false, "关联预算【" + dto.getMonthBudgetCode() + "】错误，请检查！");
                    }
                }
                if (activityTypeMap.containsKey(dto.getActivityTypeCode())) {
                    dto.setActivityTypeName(activityTypeMap.get(dto.getActivityTypeCode()).getActivityTypeName());
                } else {
                    this.validateIsTrue(false, "活动类型【" + dto.getActivityTypeCode() + "】错误，请检查！");
                }
                if (activityFormMap.containsKey(dto.getActivityFormCode())) {
                    dto.setActivityFormName(activityFormMap.get(dto.getActivityFormCode()).getActivityFormName());
                } else {
                    this.validateIsTrue(false, "活动形式【" + dto.getActivityFormCode() + "】错误，请检查！");
                }
                if (StringUtils.isNotBlank(dto.getPromotionTypeCode())) {
                    DictDataVo promotionType = dictDataMap.get(PromotionPlanConstant.TPM_PROMOTION_TYPE).stream().filter(e -> e.getDictValue().equals(dto.getPromotionTypeCode())).findFirst().orElse(null);
                    if (promotionType != null) {
                        dto.setPromotionTypeCode(promotionType.getDictCode());
                    } else {
                        this.validateIsTrue(false, "促销类型【" + dto.getPromotionTypeCode() + "】错误，请检查！");
                    }
                }
                DictDataVo auditWay = dictDataMap.get(PromotionPlanConstant.TPM_AUDIT_FORM).stream().filter(e -> e.getDictValue().equals(dto.getAuditWayName())).findFirst().orElse(null);
                if (auditWay != null) {
                    dto.setAuditWayCode(auditWay.getDictCode());
                } else {
                    this.validateIsTrue(false, "结案方式【" + dto.getPromotionTypeCode() + "】错误，请检查！");
                }
                DictDataVo auditType = dictDataMap.get(PromotionPlanConstant.TPM_AUDIT_TYPE).stream().filter(e -> e.getDictValue().equals(dto.getAuditConditionType())).findFirst().orElse(null);
                if (auditType != null) {
                    dto.setAuditConditionType(auditType.getDictCode());
                } else {
                    this.validateIsTrue(false, "核销条件【" + dto.getPromotionTypeCode() + "】错误，请检查！");
                }

                String errInfo = this.validateGetErrorInfo();
                if (errInfo != null) {
                    errMap.put(dto.getIndex(), errInfo);
                    continue;
                }

                if (StringUtils.isBlank(dto.getId())) {
                    dto.setId(UUID.randomUUID().toString().replace("-", ""));
                }
                dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                dto.setEndDate(DateUtil.parse(DateUtil.format(dto.getEndDate(), "yyyy-MM-dd") + " 23:59:59", "yyyy-MM-dd HH:mm:ss"));
                dto.setYearMonthLy(DateUtil.format(dto.getStartDate(), "yyyy-MM"));
            } catch (Exception e) {
                log.error(e.getMessage(), e);
                e.printStackTrace();
            }
        }
        log.info("结束校验关联数据校验......");

        return new ArrayList<>(dtoList);
    }

    /**
     * 获取数据实体
     *
     * @return
     */
    @Override
    public Class<GeneralExpensesImportsVo> findCrmExcelVoClass() {
        return GeneralExpensesImportsVo.class;
    }

    /**
     * 获取业务对应的模板编码，全局唯一
     *
     * @return
     */
    @Override
    public String getTemplateCode() {
        return "TPM_PP_GENERAL_EXPENSES_IMPORT";
    }

    /**
     * 获取业务对应的模板描述
     *
     * @return
     */
    @Override
    public String getTemplateName() {
        return "促销规划-费用申请导入";
    }
}
