package com.biz.crm.tpm.business.promotion.plan.local.service.process;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.channel.sdk.service.CustomerChannelVoService;
import com.biz.crm.mdm.business.customer.channel.sdk.vo.CustomerChannelVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.biz.crm.tpm.business.activity.form.sdk.service.ActivityFormService;
import com.biz.crm.tpm.business.activity.form.sdk.vo.ActivityFormVo;
import com.biz.crm.tpm.business.promotion.plan.local.entity.PromotionFeeApportion;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.PromotionFeeApportionDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.PromotionFeeApportionService;
import com.biz.crm.tpm.business.promotion.plan.sdk.vo.PromotionFeeApportionImportsVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 促销费用分摊导入
 *
 * @author: yaoyongming
 * @date: 2023/8/29 16:38
 */
@Slf4j
@Component
public class PromotionFeeApportionImportsProcess implements ImportProcess<PromotionFeeApportionImportsVo> {

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private CustomerChannelVoService customerChannelVoService;

    @Autowired(required = false)
    private DictDataVoService dictDataVoService;

    @Autowired(required = false)
    private SalesOrgVoService salesOrgVoService;

    @Autowired(required = false)
    private ActivityFormService activityFormService;

    @Autowired(required = false)
    private PromotionFeeApportionService promotionFeeApportionService;

    private static final String MDM_BUSINESS_FORMAT = "mdm_business_format";

    /**
     * 单次缓存的数量
     *
     * @return
     */
    @Override
    public Integer getBatchCount() {
        return Integer.MAX_VALUE;
    }

    /**
     * 是否开启先校验后导入的模式 默认false: false执行旧逻辑 true执行新逻辑(先校验再保存) <br/>
     * 新逻辑需同时实现接口 tryVerify tryConfirm（默认方法调用execute）
     *
     * @return
     */
    @Override
    public boolean importBeforeValidationFlag() {
        return Boolean.TRUE;
    }

    /**
     * 数据处理
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, PromotionFeeApportionImportsVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Map<Integer, String> errMap = new HashMap<>();
        Map<String, List<CustomerVo>> customerVoMap = new HashMap<>();
        Map<String, List<SalesOrgVo>> orgVoMap = new HashMap<>();
        Map<String, CustomerChannelVo> channelVoMap = new HashMap<>();
        Map<String, ActivityFormVo> activityFormMap = new HashMap<>();

        Set<String> customerCodeSet = new HashSet<>();
        Set<String> channelCodeSet = new HashSet<>();
        Set<String> orgCodeSet = new HashSet<>();
        Set<String> uniqueKeySet = new HashSet<>();
        Set<String> activityFormCodes = new HashSet<>();

        // 查询数据字典
        Map<String, List<DictDataVo>> dictDataMap = dictDataVoService.findByDictTypeCodeList(Arrays.asList(MDM_BUSINESS_FORMAT));
        for (Integer k : data.keySet()) {
            PromotionFeeApportionImportsVo vo = data.get(k);

            //业态
            DictDataVo businessFormat = dictDataMap.get(MDM_BUSINESS_FORMAT).stream().filter(e -> e.getDictValue().equals(vo.getBusinessFormatCode())).findFirst().orElse(null);
            if (businessFormat != null) {
                vo.setBusinessFormatCode(businessFormat.getDictCode());
            } else {
                this.validateIsTrue(false, "业态【"+ vo.getBusinessFormatCode() +"】错误!");
            }

            channelCodeSet.add(vo.getChannelCode());
            orgCodeSet.add(vo.getSalesInstitutionCode());
            customerCodeSet.add(vo.getCustomerCode());
            activityFormCodes.add(vo.getActivityFormCode());

            String uniqueKey = businessFormat.getDictCode() +
                    vo.getChannelCode() +
                    vo.getSalesInstitutionCode() +
                    vo.getCustomerCode() +
                    vo.getYearMonthLy() +
                    vo.getActivityFormCode();
            this.validateIsTrue(!uniqueKeySet.contains(uniqueKey), "导入文件中存在重复数据");
            uniqueKeySet.add(uniqueKey);

            data.get(k).setUniqueKey(uniqueKey);
            data.get(k).setBusinessFormatCode(businessFormat.getDictCode());
        }

        //查询渠道
        List<CustomerChannelVo> channelList = customerChannelVoService.findByCodes(new ArrayList<>(channelCodeSet));
        if (!CollectionUtils.isEmpty(channelList)) {
            channelVoMap = channelList.stream().collect(Collectors.toMap(e -> e.getCustomerChannelCode(), Function.identity()));
        }
        //销售机构
        List<SalesOrgVo> orgVos = salesOrgVoService.findByErpCodeList(new ArrayList<>(orgCodeSet));
        if (!CollectionUtils.isEmpty(orgVos)) {
            orgVoMap = orgVos.stream().collect(Collectors.groupingBy(e -> e.getErpCode()));
        }
        // 客户
        List<CustomerVo> customerVos = customerVoService.findByCustomerMdgCodes(new ArrayList<>(customerCodeSet));
        if (!CollectionUtils.isEmpty(customerVos)) {
            customerVoMap = customerVos.stream().collect(Collectors.groupingBy(e -> e.getErpCode()));
        }
        //活动形式
        List<ActivityFormVo> activityFormList = activityFormService.findActivityFormByCode(activityFormCodes);
        if (!CollectionUtils.isEmpty(activityFormList)) {
            activityFormMap = activityFormList.stream().collect(Collectors.toMap(ActivityFormVo::getActivityFormCode, Function.identity()));
        }

        for (Integer k : data.keySet()) {
            PromotionFeeApportionImportsVo vo = data.get(k);

            if (activityFormMap.containsKey(vo.getActivityFormCode())) {
                data.get(k).setActivityFormName(activityFormMap.get(vo.getActivityFormCode()).getActivityFormName());
            } else {
                this.validateIsTrue(false, "活动形式编码【"+ vo.getActivityFormCode() +"】错误！");
            }
            if (channelVoMap.containsKey(vo.getChannelCode())) {
                data.get(k).setChannelName(channelVoMap.get(vo.getChannelCode()).getCustomerChannelName());
            } else {
                this.validateIsTrue(false, "渠道编码【"+ vo.getChannelCode() +"】错误！");
            }
            SalesOrgVo salesOrgVo = null;
            if (orgVoMap.containsKey(vo.getSalesInstitutionCode())) {
                List<SalesOrgVo> orgList = orgVoMap.get(vo.getSalesInstitutionCode());
                salesOrgVo = orgList.stream().filter(e -> vo.getChannelCode().equals(e.getChannelCode()) && vo.getBusinessFormatCode().equals(e.getBusinessFormatCode()))
                        .findFirst().orElse(null);
                if (salesOrgVo == null) {
                    this.validateIsTrue(false, "销售机构编码【" + vo.getSalesInstitutionCode() + "】未找到对应销售机构，请检查！");
                } else {
                    data.get(k).setSalesInstitutionCode(salesOrgVo.getSalesOrgCode());
                    data.get(k).setSalesInstitutionErpCode(salesOrgVo.getErpCode());
                    data.get(k).setSalesInstitutionName(salesOrgVo.getSalesOrgName());
                }
            } else {
                this.validateIsTrue(false, "销售机构编码【"+ vo.getSalesInstitutionCode() +"】错误！");
            }
            if (customerVoMap.containsKey(vo.getCustomerCode())) {
                SalesOrgVo finalSalesOrgVo = salesOrgVo;
                List<CustomerVo> customerList = customerVoMap.get(vo.getCustomerCode());
                if (finalSalesOrgVo != null) {
                    CustomerVo customerVo = customerList.stream().filter(e -> vo.getChannelCode().equals(e.getCustomerChannelCode()) && finalSalesOrgVo.getErpCode().equals(e.getSalesInstitutionErpCode()) && vo.getBusinessFormatCode().equals(e.getBusinessFormatCode()))
                            .findFirst().orElse(null);
                    if (customerVo == null) {
                        this.validateIsTrue(false, "客户编码【" + vo.getCustomerCode() + "】未找到对应客户，请检查！");
                    } else {
                        data.get(k).setCustomerCode(customerVo.getCustomerCode());
                        data.get(k).setCustomerErpCode(customerVo.getErpCode());
                        data.get(k).setCustomerName(customerVo.getCustomerName());
                        data.get(k).setPlatformCode(customerVo.getEstorePlatform());
                    }
                }
            } else {
                this.validateIsTrue(false, "客户编码【"+ vo.getCustomerCode() +"】错误！");
            }

            data.get(k).setTenantCode(TenantUtils.getTenantCode());
            data.get(k).setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            data.get(k).setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            data.get(k).setFeeApportion(vo.getFeeApportionTotal().multiply(vo.getRatio()).setScale(2, BigDecimal.ROUND_HALF_UP));

            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(k, errInfo);
            }
        }

        if (errMap.isEmpty()) {
            Collection<PromotionFeeApportionImportsVo> values = data.values();
            promotionFeeApportionService.importSave(new ArrayList<>(nebulaToolkitService.copyCollectionByWhiteList(values, PromotionFeeApportionImportsVo.class, PromotionFeeApportionDto.class, LinkedHashSet.class, ArrayList.class)));
        }

        return errMap;
    }

    /**
     * 数据校验
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> tryVerify(LinkedHashMap<Integer, PromotionFeeApportionImportsVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Map<Integer, String> errMap = new HashMap<>();
        log.info("===========================开始数据校验");
        for (Map.Entry<Integer, PromotionFeeApportionImportsVo> row : data.entrySet()) {
            int rowNum = row.getKey();
            PromotionFeeApportionImportsVo vo = row.getValue();
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getBusinessFormatCode()), "业态必填!");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getChannelCode()), "渠道编码必填!");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getSalesInstitutionCode()), "销售机构编码必填!");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getYearMonthLy()), "年月必填!");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getCustomerCode()), "客户编码必填!");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getActivityFormCode()), "活动形式编码必填!");
            this.validateIsTrue(vo.getRatio() != null, "分摊比例必填!");
            this.validateIsTrue(vo.getFeeApportionTotal() != null, "促销费用分摊总额(元)必填!");
            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(rowNum, errInfo);
            }
        }
        log.info("===========================数据校验信息：" + JSONObject.toJSONString(errMap));
        return errMap;
    }

    /**
     * 获取数据实体
     *
     * @return
     */
    @Override
    public Class<PromotionFeeApportionImportsVo> findCrmExcelVoClass() {
        return PromotionFeeApportionImportsVo.class;
    }

    /**
     * 获取业务对应的模板编码，全局唯一
     *
     * @return
     */
    @Override
    public String getTemplateCode() {
        return "PROMOTION_FEE_APPORTION_IMPORT";
    }

    /**
     * 获取业务对应的模板描述
     *
     * @return
     */
    @Override
    public String getTemplateName() {
        return "TPM-促销费用分摊导入";
    }
}
