package com.biz.crm.tpm.business.promotion.plan.local.service.process;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductPlanningDetailsVo;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.dto.PurchaseSalePromotionPlanAmountDto;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.service.ActivityApplyRulesService;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.vo.ActivityApplyRulesPromotionPlanAmountVo;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.PurchaseSaleDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.PurchaseSaleService;
import com.biz.crm.tpm.business.promotion.plan.sdk.vo.PurchaseSaleImportsVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import liquibase.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 促销规划-采销库存
 *
 * @author: yaoyongming
 * @date: 2023/1/11 9:26
 */
@Slf4j
@Component
public class PurchaseSaleImportsProcess implements ImportProcess<PurchaseSaleImportsVo> {

    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private ProductVoService productVoService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private ActivityApplyRulesService activityApplyRulesService;

    @Autowired(required = false)
    private PurchaseSaleService purchaseSaleService;

    /**
     * 单次缓存的数量
     *
     * @return
     */
    @Override
    public Integer getBatchCount() {
        return 10000;
    }

    /**
     * 是否开启先校验后导入的模式 默认false: false执行旧逻辑 true执行新逻辑(先校验再保存) <br/>
     * 新逻辑需同时实现接口 tryVerify tryConfirm（默认方法调用execute）
     *
     * @return
     */
    @Override
    public boolean importBeforeValidationFlag() {
        return Boolean.TRUE;
    }

    /**
     * 数据校验
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> tryVerify(LinkedHashMap<Integer, PurchaseSaleImportsVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Map<Integer, String> errMap = new HashMap<>();
        Validate.notNull(params.get("platformCode"), "平台编码，参数未传入");
        Validate.notNull(params.get("platformName"), "平台名称，参数未传入");
        Validate.notNull(params.get("salesOrgCode"), "组织编码，参数未传入");
        Validate.notNull(params.get("businessModelCode"), "业务模式，参数未传入");
        Validate.notNull(params.get("customerCode"), "客户，参数未传入");
        Validate.notNull(params.get("startDate"), "开始时间，参数未传入");
        for (Map.Entry<Integer, PurchaseSaleImportsVo> row : data.entrySet()) {
            int rowNum = row.getKey();
            PurchaseSaleImportsVo vo = row.getValue();
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getProductCode()), "产品编码，不能为空！");
            this.validateIsTrue(vo.getMonthInventoryEnd() != null, "本月期末库存（箱），不能为空！");
            this.validateIsTrue(vo.getPromotionPrice() != null, "本月库存预估价（元），不能为空！");
            this.validateIsTrue( StringUtil.isNotEmpty(vo.getNextMonthInventoryAmountStr()), "次月库存预估售价（元），不能为空！");
            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(rowNum, errInfo);
            }
        }
        return errMap;
    }

    /**
     * 数据处理
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, PurchaseSaleImportsVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Map<Integer, String> errMap = new HashMap<>();
        try {
            log.info("开始导入......");
            Validate.notEmpty(data, "导入数据不能为空！");

            Validate.notNull(params.get("cacheKey"), "缓存键，参数未传入");
            String cacheKey = (String) params.get("cacheKey");

            log.info("开始导入校验......");
            this.validate(data, params, errMap);
            log.info("导入校验结束......错误信息[{}]条", errMap.size());
            if (errMap.isEmpty()) {
                log.info("开始导入缓存......");
                List<PurchaseSaleDto> dtoList = (List<PurchaseSaleDto>) nebulaToolkitService.copyCollectionByWhiteList(data.values(), PurchaseSaleImportsVo.class, PurchaseSaleDto.class, LinkedHashSet.class, ArrayList.class);
                purchaseSaleService.addListCache(cacheKey, dtoList);
                log.info("结束导入缓存......");
            }
        } catch (Exception e) {
            log.error("促销规划-采销库存导入异常");
            errMap.put(1, errMap.getOrDefault(1, "") + " | " + e.getMessage());
            log.error("", e);
        }
        return errMap;
    }

    private void validate(LinkedHashMap<Integer, PurchaseSaleImportsVo> data, Map<String, Object> params, Map<Integer, String> errMap) {
        String platformCode = (String) params.get("platformCode");
        String platformName = (String) params.get("platformName");
        String salesOrgCode = (String) params.get("salesOrgCode");
        String businessModelCode = (String) params.get("businessModelCode");
        String customerCode = (String) params.get("customerCode");
        Date startDate = DateUtil.parse((String) params.get("startDate"));
        String startDateStr = DateUtil.format(startDate, "yyyy-MM-dd");
        String cacheKeySale = (String) params.get("cacheKeySale");
        String cacheKeyGeneral = params.get("cacheKeyGeneral") == null ? null : (String) params.get("cacheKeyGeneral");

        log.info("开始关联数据封装......");
        //关联数据封装
        Set<String> productCodeSet = data.values().stream()
                .filter(k -> StringUtil.isNotEmpty(k.getProductCode()))
                .map(PurchaseSaleImportsVo::getProductCode)
                .filter(Objects::nonNull).collect(Collectors.toSet());

        Map<String, CustomerVo> customerVoMap = new HashMap<>();
        Map<String, ProductVo> productVoMap = new HashMap<>();

        //关联数据查询
        List<CustomerVo> customerVos = customerVoService.findByCustomerCodes(Collections.singletonList(customerCode));
        if (!CollectionUtils.isEmpty(customerVos)) {
            customerVoMap.putAll(customerVos.stream().collect(Collectors.toMap(CustomerVo::getCustomerCode, Function.identity(), (n, o) -> n)));
        }
        List<ProductVo> productVos = productVoService.findDetailsByIdsOrProductCodes(null, new ArrayList<>(productCodeSet));
        if (!CollectionUtils.isEmpty(productVos)) {
            productVoMap.putAll(productVos.stream().collect(Collectors.toMap(ProductVo::getProductCode, Function.identity(), (n, o) -> n)));
            ;
        }
        log.info("结束关联数据封装......");


        log.info("开始校验关联数据校验......");
        //关联数据校验
        data.forEach((index, dto) -> {
            try {
                if (customerVoMap.containsKey(customerCode)) {
                    CustomerVo customerVo = customerVoMap.get(customerCode);
                    dto.setErpCode(customerVo.getErpCode());
                } else {
                    this.validateIsTrue(false, "客户编码【" + customerCode + "】错误，请检查！");
                }
                if (productVoMap.containsKey(dto.getProductCode())) {
                    ProductVo productVo = productVoMap.get(dto.getProductCode());
                    dto.setProductName(productVo.getProductName());
                    dto.setProductBrandCode(productVo.getProductBrandCode());
                    dto.setProductBrandName(productVo.getProductBrandName());
                } else {
                    this.validateIsTrue(false, "产品编码【" + dto.getProductCode() + "】错误，请检查！");
                }
                log.info("dto.getProductCode()=" + dto.getProductCode() + "、getErpCode=" + dto.getErpCode() + "、businessModelCode=" + businessModelCode + "、getStartDate=" + startDate);
                ProductPlanningDetailsVo feeVo = null;
                try {
                    feeVo = productVoService.findByProductPlanningDetails(dto.getProductCode(), dto.getErpCode(), businessModelCode, startDateStr);
                } catch (Exception e) {
                    validateIsTrue(false, "价格管理查询数据不存在!");
                }
                if (StringUtil.isNotEmpty(dto.getNextMonthEndInventoryStr())) {
                    try {
                        dto.setNextMonthEndInventory(new BigDecimal(dto.getNextMonthEndInventoryStr()));
                    } catch (Exception e) {
                        validateIsTrue(false, "次月期末库存（箱）格式错误!");
                    }
                }
                if (StringUtil.isNotEmpty(dto.getNextMonthInventoryAmountStr())) {
                    try {
                        dto.setNextMonthInventoryAmount(new BigDecimal(dto.getNextMonthInventoryAmountStr()));
                    } catch (Exception e) {
                        validateIsTrue(false, "次月库存预估售价（元）格式错误!");
                    }
                }

                if (Objects.nonNull(feeVo)) {
                    dto.setSupplyPrice(feeVo.getPlatformSupplyPrice());
                    dto.setStandardRetailPrice(feeVo.getStandardRetailPrice());
                    dto.setRedLinePrice(feeVo.getRedLinePrice());
                    dto.setCostPrice(feeVo.getCostPrice());
                }

                this.validateIsTrue(dto.getSupplyPrice() != null, "产品【" + dto.getProductCode() + "】平台供货价，不能为空！");

                String errInfo = this.validateGetErrorInfo();
                if (errInfo != null) {
                    errMap.put(index, errInfo);
                    return;
                }

                log.info("开始自动计算......");
                log.info("开始申请规则接口自动计算......");
                PurchaseSalePromotionPlanAmountDto amountDto = nebulaToolkitService.copyObjectByWhiteList(dto, PurchaseSalePromotionPlanAmountDto.class, LinkedHashSet.class, ArrayList.class);
                amountDto.setPlatformCode(platformCode);
                amountDto.setSalesOrgCode(salesOrgCode);
                amountDto.setCustomerCode(customerCode);
                amountDto.setMonthSaleCacheKey(cacheKeySale);
                amountDto.setGeneralExpensesCacheKey(cacheKeyGeneral);
                ActivityApplyRulesPromotionPlanAmountVo amountVo = activityApplyRulesService.calAmountPurchaseSale(amountDto);
                BeanUtils.copyProperties(amountVo, dto);
                resetRulesAmount(dto, amountVo);
                log.info("结束申请规则接口自动计算......");
                log.info("结束自动计算......");

                dto.setId(UUID.randomUUID().toString().replace("-", ""));
                dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                dto.setTenantCode(TenantUtils.getTenantCode());
            } catch (Exception e) {
                log.error("=====>   促销规划-采销库存 异常数据[{}]   <=====", JSON.toJSONString(dto));
                log.error("", e);
            }
        });
        log.info("结束校验关联数据校验......");
    }

    /**
     * 重置规则金额
     *
     * @param dto
     */
    private void resetRulesAmount(PurchaseSaleImportsVo dto, ActivityApplyRulesPromotionPlanAmountVo amountVo) {
        if (dto.getFeePoolDifference() == null) {
            dto.setFeePoolDifference(amountVo.getFeePoolDifference());
        }
        if (dto.getFeePoolSaleCommission() == null) {
            dto.setFeePoolSaleCommission(amountVo.getFeePoolSaleCommission());
        }
        if (dto.getFeePoolPurchaseRebate() == null) {
            dto.setFeePoolPurchaseRebate(amountVo.getFeePoolPurchaseRebate());
        }
        if (dto.getFeePoolPut() == null) {
            dto.setFeePoolPut(amountVo.getFeePoolPut());
        }
        if (dto.getFeePoolGrossProtection() == null) {
            dto.setFeePoolGrossProtection(amountVo.getFeePoolGrossProtection());
        }
    }

    /**
     * 获取数据实体
     *
     * @return
     */
    @Override
    public Class<PurchaseSaleImportsVo> findCrmExcelVoClass() {
        return PurchaseSaleImportsVo.class;
    }

    /**
     * 获取业务对应的模板编码，全局唯一
     *
     * @return
     */
    @Override
    public String getTemplateCode() {
        return "TPM_PP_PURCHASE_SALE_IMPORT";
    }

    /**
     * 获取业务对应的模板描述
     *
     * @return
     */
    @Override
    public String getTemplateName() {
        return "促销规划-采销库存导入";
    }
}
