package com.biz.crm.tpm.business.reconciliation.doc.list.local.service.internal;

import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.reconciliation.doc.list.local.service.ReconciliationDocBuildService;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.constant.ReconciliationConstant;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.service.ReconciliationDocListService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * 构建对账单账 service 实现
 *
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2023-07-14 15:58
 */
@Service
@Slf4j
public class ReconciliationDocBuildServiceImpl implements ReconciliationDocBuildService {

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Autowired(required = false)
    private ReconciliationDocListService reconciliationDocListService;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    /**
     * 手动触发创建对账单
     *
     * @param userIdentity
     * @return void
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023-07-14 16:37
     */
    @Override
    @Async
    public void handleBuildReconciliation(Date date, AbstractCrmUserIdentity userIdentity) {
        loginUserService.refreshAuthentication(userIdentity);
        String lockKey = this.getBuildRedisLockKey(date);
        boolean lock = false;
        try {
            lock = redisLockService.tryLock(lockKey, TimeUnit.HOURS, 4);
            Assert.isTrue(lock, "自动生成的对账单加锁失败");
            this.autoBuildReconciliation(date);
            log.info("=====>     自动生成的对账单成功     <=====");
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage());
        } finally {
            if (lock) {
                redisLockService.unlock(lockKey);
            }
        }

    }

    /**
     * 能力中心生成对账单，不走异步
     *
     * @param userIdentity
     * @return void
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023-07-14 16:37
     */
    @Override
    public List<String> handleBuildReconciliationForOut(Date date, AbstractCrmUserIdentity userIdentity) {
        loginUserService.refreshAuthentication(userIdentity);
        String lockKey = this.getBuildRedisLockKey(date);
        boolean lock = false;
        try {
            lock = redisLockService.tryLock(lockKey, TimeUnit.HOURS, 4);
            Assert.isTrue(lock, "自动生成的对账单加锁失败");
            List<String> list = this.autoBuildReconciliationForOut(date);
            log.info("=====>     自动生成的对账单成功     <=====");
            return list;
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage());
        } finally {
            if (lock) {
                redisLockService.unlock(lockKey);
            }
        }

    }

    private List<String> autoBuildReconciliationForOut(Date date) {
        return reconciliationDocListService.autoBuildReconciliationForOut(date);
    }

    /**
     * 自动触发创建对账单
     *
     * @param
     * @param date
     * @return void
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023-07-14 16:05
     */
    @Override
    public void autoBuildReconciliation(Date date) {
        reconciliationDocListService.autoBuildReconciliation(date);
    }

    /**
     * 自动触发补偿创建失败的对账单
     *
     * @param date
     * @return void
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023-07-14 19:57
     */
    @Override
    public void autoBuildFailReconciliation(Date date) {
        reconciliationDocListService.autoBuildFailReconciliation(date);
    }

    /**
     * 创建对账单锁定的KEY
     *
     * @param date 为空时,默认当天
     * @return java.lang.String
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023-07-14 16:30
     */
    @Override
    public String getBuildRedisLockKey(Date date) {
        if (Objects.isNull(date)) {
            date = new Date();
        }
        String lockKey = DateUtil.format(date, DateUtil.DEFAULT_YEAR_MONTH_DAY);
        return ReconciliationConstant.REDIS_LOCK + lockKey;
    }

    /**
     * 创建对账单失败补偿锁定的KEY
     *
     * @param date 为空时,默认当天
     * @return java.lang.String
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023-07-14 16:30
     */
    @Override
    public String getBuildFailRedisLockKey(Date date) {

        if (Objects.isNull(date)) {
            date = new Date();
        }
        String lockKey = DateUtil.format(date, DateUtil.DEFAULT_YEAR_MONTH_DAY);
        return ReconciliationConstant.REDIS_FAIL_LOCK + lockKey;

    }
}
