package com.biz.crm.tpm.business.reconciliation.doc.list.local.service.internal;

import cn.hutool.core.date.DatePattern;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.local.entity.UuidEntity;
import com.biz.crm.business.common.sdk.vo.FileVo;
import com.biz.crm.mn.third.system.electronic.signature.center.sdk.vo.CreateContractVo;
import com.biz.crm.mn.third.system.electronic.signature.center.sdk.vo.ElectronicSignatureVo;
import com.biz.crm.mn.third.system.electronic.signature.center.sdk.vo.NewFileVo;
import com.biz.crm.tpm.business.account.reconciliation.rule.sdk.service.TpmAccountReconciliationRuleService;
import com.biz.crm.tpm.business.account.reconciliation.rule.sdk.vo.TpmAccountReconciliationRuleDateConfigRespVo;
import com.biz.crm.tpm.business.account.reconciliation.rule.sdk.vo.TpmAccountReconciliationRuleRespVo;
import com.biz.crm.tpm.business.reconciliation.doc.list.local.entity.ReconciliationDocListEntity;
import com.biz.crm.tpm.business.reconciliation.doc.list.local.repository.ReconciliationDocListRepository;
import com.biz.crm.tpm.business.reconciliation.doc.list.local.service.BaseEcrmBuildService;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.constant.ReconciliationConstant;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.dto.ReconciliationDocListDto;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.dto.ReconciliationDocListTaskDto;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.dto.ReconciliationJobLogDto;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.dto.log.ReconciliationDocListLogEventDto;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.enums.ReconciliationFileTypeEnum;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.enums.ReconciliationStatusEnum;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.event.ReconciliationDocListLogEventListener;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.service.ReconciliationCorrelationVoService;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.service.ReconciliationDocListFileVoService;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.service.ReconciliationDocListService;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.service.ReconciliationJobLogVoService;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.vo.ReconciliationCorrelationVo;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.vo.ReconciliationDocListFileVo;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.vo.ReconciliationDocListVo;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.vo.ReconciliationJobLogVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.bizunited.nebula.venus.sdk.service.file.FileHandleService;
import com.bizunited.nebula.venus.sdk.vo.OrdinaryFileVo;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.flowable.common.engine.impl.util.CollectionUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * 对账单据列表(ReconciliationDocListEntity)服务实现类
 *
 * @author xiaoyoujun
 * @date 2022年11月17日 14:19
 */
@Service
@Slf4j
public class ReconciliationDocListServiceImpl implements ReconciliationDocListService {

    @Autowired(required = false)
    private ReconciliationDocListRepository reconciliationDocListRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private BaseElectronicSignature baseElectronicSignature;

    /**
     * TpmAccountReconciliationRule 列表
     */
    @Autowired(required = false)
    private TpmAccountReconciliationRuleService accountReconciliationRuleService;

    /**
     * 任务日志服务
     */
    @Autowired(required = false)
    private ReconciliationJobLogVoService reconciliationJobLogVoService;

    /**
     * redis 服务
     */
    @Autowired(required = false)
    private ReconciliationDocListTaskService reconciliationDocListTaskService;

    /**
     * 对账单关联数据服务
     */
    @Autowired(required = false)
    private ReconciliationCorrelationVoService correlationVoService;


    @Autowired(required = false)
    private BaseEcrmBuildService baseEcrmBuildService;

    @Autowired(required = false)
    private ReconciliationDocListFileVoService reconciliationDocListFileVoService;

    @Autowired
    private FileHandleService fileHandleService;

    @Override
    public Page<ReconciliationDocListVo> findByReconciliations(Pageable pageable,
                                                               ReconciliationDocListDto reconciliationDocListDto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(reconciliationDocListDto)) {
            reconciliationDocListDto = new ReconciliationDocListDto();
        }
        return this.reconciliationDocListRepository.findByReconciliations(pageable, reconciliationDocListDto);
    }

    /**
     * 重新生成对账单
     *
     * @param ids id
     */
    @Override
    public void regenerate(List<String> ids) {
        List<ReconciliationDocListVo> reconciliationDocListVos = validateIds(ids);

        reconciliationDocListVos.forEach(x -> {
            if (ReconciliationStatusEnum.CONFIRMED.getCode().equals(x.getStatementStatus())
                    || ReconciliationStatusEnum.COMPLETE.getCode().equals(x.getStatementStatus())) {

                throw new IllegalArgumentException("对账单【" + x.getStatementCode() + "】，为待客户确认或对账完成状态，不可更新，请检查后重新提交");
            }
        });


        // 执行重新生成pdf逻辑
        reconciliationDocListVos.forEach(this::regeneratePdf);

    }

    /**
     * 重新生成pdf（针对客户进行事物包裹，防止大事物出现）
     *
     * @param doc 对账单vo
     */
    @Transactional(rollbackFor = Exception.class)
    public void regeneratePdf(ReconciliationDocListVo doc) {
        TpmAccountReconciliationRuleDateConfigRespVo ruleDateConfigRespVo =
            new TpmAccountReconciliationRuleDateConfigRespVo();
        //构建请求参数
        ruleDateConfigRespVo.setStartDate(DateUtil.formatDateTime(doc.getStartDate()));
        ruleDateConfigRespVo.setEndDate(DateUtil.formatDateTime(doc.getEndDate()));
        ruleDateConfigRespVo.setAccountReconciliationRuleCode(doc.getAccountReconciliationRuleCode());

        TpmAccountReconciliationRuleRespVo ruleRespVo = new TpmAccountReconciliationRuleRespVo();
        ruleRespVo.setBusinessUnitCode(doc.getBusinessUnitCode());
        ruleRespVo.setBusinessFormatCode(doc.getBusinessFormatCode());
        ruleRespVo.setAccountReconciliationRuleCode(doc.getAccountReconciliationRuleCode());

        //修改之前的job日志为失败
        ReconciliationJobLogDto logDto = ReconciliationJobLogDto.builder()
            .accountReconciliationRuleCode(doc.getAccountReconciliationRuleCode())
            .businessUnitCode(doc.getBusinessUnitCode())
            .businessFormatCode(doc.getBusinessFormatCode())
            .startDate(doc.getStartDate())
            .endDate(doc.getEndDate())
            .build();
        ReconciliationJobLogVo jobLogVo = this.reconciliationJobLogVoService.findUniqueJobInfo(logDto);
        if (Objects.nonNull(jobLogVo)) {
            jobLogVo.setIsSuccess(false);
            this.reconciliationJobLogVoService.update(jobLogVo);
        }

        ReconciliationDocListTaskDto taskDto = null;
        //执行任务
        if (jobLogVo == null) {
            //对账单的计算结束时间记录错误，导致查询不到log中数据
            //解决当前重新生成对账单未在ReconciliationJobLog查询到数据导致报错问题，设置计算时间为当前时间
            taskDto = ReconciliationDocListTaskDto.builder()
                .accountReconciliationType(doc.getAccountReconciliationType())
                .accountReconciliationRuleCode(doc.getAccountReconciliationRuleCode())
                .calculationTime(DateUtil.date())
                .businessFormatCode(doc.getBusinessFormatCode())
                .businessUnitCode(doc.getBusinessUnitCode())
                .endDate(doc.getEndDate())
                .startDate(doc.getStartDate())
                .customerCode(doc.getCustomerCode())
                .build();
        } else {
            taskDto = ReconciliationDocListTaskDto.builder()
                .accountReconciliationType(doc.getAccountReconciliationType())
                .accountReconciliationRuleCode(jobLogVo.getAccountReconciliationRuleCode())
                .calculationTime(jobLogVo.getCalculationTime())
                .businessFormatCode(jobLogVo.getBusinessFormatCode())
                .businessUnitCode(jobLogVo.getBusinessUnitCode())
                .endDate(jobLogVo.getEndDate())
                .startDate(jobLogVo.getStartDate())
                .customerCode(doc.getCustomerCode())
                .build();
        }

        List<ReconciliationDocListVo> docListVos = reconciliationDocListTaskService.executeTask(taskDto);
        createBatch(docListVos);

        //删除之前的对账单
        this.reconciliationDocListRepository.removeById(doc.getId());
    }

    /**
     * 更新对账单
     *
     * @param reconciliationDocListDto 实体对象
     * @return {@link ReconciliationDocListVo}
     */
    @Override
    public ReconciliationDocListVo update(ReconciliationDocListDto reconciliationDocListDto) {
        ReconciliationDocListVo reconciliationDocListVo = findById(reconciliationDocListDto.getId());
        Validate.notNull(reconciliationDocListVo, "更新数据不存在，请检查！");
        if (!reconciliationDocListDto.getStatementStatus().equals(ReconciliationStatusEnum.WITHDRAW.getCode())) {
            throw new IllegalArgumentException("仅更新对账撤回状态!");
        }
        reconciliationDocListDto.setTenantCode(TenantUtils.getTenantCode());
        reconciliationDocListDto.setStatementStatus(ReconciliationStatusEnum.WAIT.getCode());
        ReconciliationDocListEntity reconciliationDocListEntity = this.nebulaToolkitService.copyObjectByWhiteList(
            reconciliationDocListDto, ReconciliationDocListEntity.class, LinkedHashSet.class, ArrayList.class);
        this.reconciliationDocListRepository.save(reconciliationDocListEntity);
        // 日志新增
        ReconciliationDocListDto dto = this.nebulaToolkitService.copyObjectByWhiteList(reconciliationDocListVo,
            ReconciliationDocListDto.class, LinkedHashSet.class, ArrayList.class);
        ReconciliationDocListLogEventDto logEventDto = new ReconciliationDocListLogEventDto();
        logEventDto.setOriginal(dto);
        logEventDto.setNewest(reconciliationDocListDto);
        SerializableBiConsumer<ReconciliationDocListLogEventListener, ReconciliationDocListLogEventDto> onCreate =
            ReconciliationDocListLogEventListener::onCreate;
        this.nebulaNetEventClient.publish(logEventDto, ReconciliationDocListLogEventListener.class, onCreate);
        return reconciliationDocListVo;
    }

    /**
     * 批量创建对账单据列表
     *
     * @param reconciliationDocListVos 对账单据列表
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<String> createBatch(List<ReconciliationDocListVo> reconciliationDocListVos) {
        if (CollectionUtils.isEmpty(reconciliationDocListVos)) {
            return new ArrayList<>();
        }

        Date reconciliationMonth = reconciliationDocListVos.stream().map(ReconciliationDocListVo::getReconciliationMonth).filter(Objects::nonNull).findFirst().orElse(null);
        String accountReconciliationType = reconciliationDocListVos.stream().map(ReconciliationDocListVo::getAccountReconciliationType).filter(Objects::nonNull).findFirst().orElse(null);
        String businessFormatCode = reconciliationDocListVos.stream().map(ReconciliationDocListVo::getBusinessFormatCode).filter(Objects::nonNull).findFirst().orElse(null);
        String businessUnitCode = reconciliationDocListVos.stream().map(ReconciliationDocListVo::getBusinessUnitCode).filter(Objects::nonNull).findFirst().orElse(null);
        if (!Objects.isNull(reconciliationMonth)
                && StringUtils.isNotBlank(accountReconciliationType)
                && StringUtils.isNotBlank(businessFormatCode)
                && StringUtils.isNotBlank(businessUnitCode)) {

            //查询同维度下，状态为 待客户确认和对账完成 的对账单，有则不更新或新增
            ReconciliationDocListDto dto = new ReconciliationDocListDto();
            dto.setReconciliationMonth(reconciliationMonth);
            dto.setAccountReconciliationType(accountReconciliationType);
            dto.setBusinessFormatCode(businessFormatCode);
            dto.setBusinessUnitCode(businessUnitCode);
            dto.setTenantCode(TenantUtils.getTenantCode());
            List<String> customerJointInstitutionErpCodes = this.findNotUpdateReconciliationDocList(dto);
            if (!CollectionUtils.isEmpty(customerJointInstitutionErpCodes)) {
                //剔除 状态为 待客户确认和对账完成 的对账单
                reconciliationDocListVos = reconciliationDocListVos.stream()
                        .filter(r -> StringUtils.isNotBlank(r.getCustomerErpCode()) && StringUtils.isNotBlank(r.getSalesInstitutionCode()))
                        .filter(r -> !(customerJointInstitutionErpCodes.contains(r.getCustomerErpCode() + "-" + r.getSalesInstitutionCode()))).collect(Collectors.toList());

            }
            List<String> collect = reconciliationDocListVos.stream().map(o -> o.getCustomerErpCode() + "-" + o.getSalesInstitutionCode()).distinct().collect(Collectors.toList());
            dto.setCustomerJointInstitutionErpCode(collect);
            //删除旧数据
            this.deleteByCondition(dto);
        }


        // 批量创建对账单
        ArrayList<ReconciliationDocListEntity> reconciliationDocListEntities =
            Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(reconciliationDocListVos,
                ReconciliationDocListVo.class, ReconciliationDocListEntity.class, HashSet.class, ArrayList.class));
        this.reconciliationDocListRepository.saveBatch(reconciliationDocListEntities);

        // 批量创建关联表
        List<ReconciliationCorrelationVo> correlationVos =
            reconciliationDocListVos.stream().filter(Objects::nonNull).map(ReconciliationDocListVo::getCorrelations)
                .filter(Objects::nonNull).flatMap(Collection::stream).collect(Collectors.toList());
        this.correlationVoService.createBatch(correlationVos);

        // 批量创建文件
        Set<ReconciliationDocListFileVo> files =
            reconciliationDocListVos.stream().filter(Objects::nonNull).map(ReconciliationDocListVo::getFiles)
                .filter(Objects::nonNull).flatMap(Collection::stream).collect(Collectors.toSet());
        reconciliationDocListFileVoService.createBatch(files);
        return reconciliationDocListEntities.stream().map(ReconciliationDocListEntity::getId).collect(Collectors.toList());
    }

    /**
     * 更据维度，批量逻辑删除
     *
     * @param dto
     */
    private void deleteByCondition(ReconciliationDocListDto dto) {
        if (Objects.isNull(dto)) {
            return;
        }
        if (Objects.isNull(dto.getReconciliationMonth())
                || StringUtils.isBlank(dto.getAccountReconciliationType())
                || StringUtils.isBlank(dto.getBusinessFormatCode())
                || StringUtils.isBlank(dto.getBusinessUnitCode())) {
            return;
        }
        dto.setTenantCode(TenantUtils.getTenantCode());
        this.reconciliationDocListRepository.deleteByCondition(dto);
    }

    private List<String> findNotUpdateReconciliationDocList(ReconciliationDocListDto dto) {
        if (Objects.isNull(dto) || StringUtils.isBlank(dto.getAccountReconciliationType())) {
            return Lists.newArrayList();
        }
        if (Objects.isNull(dto.getReconciliationMonth())) {
            dto.setReconciliationMonth(new Date());
        }
        dto.setTenantCode(TenantUtils.getTenantCode());
        List<String> list = reconciliationDocListRepository.findNotUpdateReconciliationDocList(dto);
        if (CollectionUtils.isEmpty(list)) {
            return Lists.newArrayList();
        }
        return list;
    }

    /**
     * 通过主键查询单条数据
     *
     * @param id 主键
     * @return {@link ReconciliationDocListVo}
     */
    @Override
    public ReconciliationDocListVo findById(String id) {
        Validate.notNull(id, "主键id不能为空");

        ReconciliationDocListEntity reconciliationDocListEntity = this.reconciliationDocListRepository.getById(id);
        if (reconciliationDocListEntity == null) {
            return null;
        }
        return nebulaToolkitService.copyObjectByWhiteList(reconciliationDocListEntity, ReconciliationDocListVo.class,
            LinkedHashSet.class, ArrayList.class);

    }

    /**
     * 根据合同编码查询
     *
     * @param contractId 合同编码
     * @return {@link ReconciliationDocListVo}
     */
    @Override
    public ReconciliationDocListVo findByContractId(String contractId) {
        Validate.notNull(contractId, "合同编码不能为空");

        ReconciliationDocListEntity reconciliationDocListEntity =
            this.reconciliationDocListRepository.findByContractId(contractId);
        if (reconciliationDocListEntity == null) {
            return null;
        }
        return nebulaToolkitService.copyObjectByWhiteList(reconciliationDocListEntity, ReconciliationDocListVo.class,
            LinkedHashSet.class, ArrayList.class);

    }

    /**
     * 验证id
     *
     * @param ids id
     * @return {@link List}<{@link ReconciliationDocListVo}>
     */
    private List<ReconciliationDocListVo> validateIds(List<String> ids) {
        Validate.notNull(ids, "主键id不能为空");
        List<ReconciliationDocListEntity> docListEntities = this.reconciliationDocListRepository.findByIds(ids);
        Validate.notEmpty(docListEntities, "对应id查询不到实例对象,请检查id是否正确");
        Collection<ReconciliationDocListVo> reconciliationDocListVos =
            nebulaToolkitService.copyCollectionByWhiteList(docListEntities, ReconciliationDocListEntity.class,
                ReconciliationDocListVo.class, HashSet.class, ArrayList.class);
        return Lists.newArrayList(reconciliationDocListVos);
    }

    /**
     * 推送电子签章
     *
     * @param ids 数据主键列表
     */
    @Override
    public void pushAccountCheckBill(List<String> ids) {
        List<ReconciliationDocListVo> reconciliationDocListVos = validateIds(ids);

        List<ReconciliationDocListVo> dateList = reconciliationDocListVos.stream()
            //获取不是 待客户确认状态的数据集  只有待客户确认可以撤回
            .filter(x -> !ReconciliationStatusEnum.WAIT.getCode().equals(x.getStatementStatus()))
            .collect(Collectors.toList());
        Validate.isTrue(CollectionUtils.isEmpty(dateList), "数据集中存在非待推送状态的数据，只有待推送可以推送电子签章");


        reconciliationDocListVos.forEach(reconciliationDocListVo -> {
            Validate.notNull(reconciliationDocListVo.getReceiverTelephone(), "接受人手机号不能为空");
            Validate.notNull(reconciliationDocListVo.getRecipientName(), "接受人姓名不能为空");
            // 执行推送逻辑
            if (StrUtil.isEmpty(reconciliationDocListVo.getContractId())
                && StrUtil.isEmpty(reconciliationDocListVo.getDocumentsId())) {
                ElectronicSignatureVo<CreateContractVo> longElectronicSignatureVo =
                    baseElectronicSignature.pushElectronicSignature(reconciliationDocListVo);
                Optional<ElectronicSignatureVo<CreateContractVo>> electronicSignatureVo =
                    Optional.ofNullable(longElectronicSignatureVo);
                if (electronicSignatureVo.isPresent()) {
                    ElectronicSignatureVo<CreateContractVo> voElectronicSignatureVo = electronicSignatureVo.get();
                    if (Objects.isNull(voElectronicSignatureVo.getResult())) {
                        return;
                    }
                    reconciliationDocListVo
                        .setContractId(String.valueOf(voElectronicSignatureVo.getResult().getContractId()));
                    reconciliationDocListVo
                        .setDocumentsId(String.valueOf(voElectronicSignatureVo.getResult().getDocumentId()));
                }
            }
            reconciliationDocListVo.setStatementStatus(ReconciliationStatusEnum.CONFIRMED.getCode());

            // 推送ecrm todo 改成 Rocketmq
            baseEcrmBuildService.pushECrm(reconciliationDocListVo);
            baseEcrmBuildService.syncECrm(reconciliationDocListVo, null);

            // 修改库
            ReconciliationDocListEntity docListEntity = nebulaToolkitService.copyObjectByWhiteList(
                reconciliationDocListVo, ReconciliationDocListEntity.class, HashSet.class, ArrayList.class);
            this.reconciliationDocListRepository.updateById(docListEntity);
        });

    }


    /**
     * 作废电子签章
     *
     * @param ids 数据主键列表
     */
    @Override
    public void cancellationContract(List<String> ids) {
        List<ReconciliationDocListVo> reconciliationDocListVos = validateIds(ids);

        List<ReconciliationDocListVo> dateList = reconciliationDocListVos.stream()
            //获取不是 待客户确认状态的数据集  只有待客户确认可以撤回
            .filter(x -> !ReconciliationStatusEnum.COMPLETE.getCode().equals(x.getStatementStatus()))
            .collect(Collectors.toList());
        Validate.isTrue(CollectionUtils.isEmpty(dateList), "数据集中存在非对账完成状态的数据，只有对账完成可以作废");

        reconciliationDocListVos.forEach(reconciliationDocListVo -> {
            // 如果已经是作废状态就不再执行
            if (ReconciliationStatusEnum.SCRAP.getCode().equals(reconciliationDocListVo.getStatementStatus())) {
                return;
            }
            Validate.notBlank(reconciliationDocListVo.getContractId(), "合同ID不能为空");
            Validate.notBlank(reconciliationDocListVo.getDocumentsId(), "文档ID不能为空");

            // 调用第三方接口
            ElectronicSignatureVo signatureVo = baseElectronicSignature.cancelElectronicSignature(reconciliationDocListVo);

            if (Objects.nonNull(signatureVo) && signatureVo.isSuccess()) {
                reconciliationDocListVo.setStatementStatus(ReconciliationStatusEnum.SCRAP.getCode());

                // 推送ecrm todo 改成 Rocket事物mq
                baseEcrmBuildService.syncECrm(reconciliationDocListVo, null);

                // 修改库
                this.reconciliationDocListRepository.lambdaUpdate()
                    .eq(UuidEntity::getId, reconciliationDocListVo.getId())
                    .set(ReconciliationDocListEntity::getStatementStatus, ReconciliationStatusEnum.SCRAP.getCode())
                    .set(ReconciliationDocListEntity::getContractId, null)
                    .set(ReconciliationDocListEntity::getDocumentsId, null)
                    .update();
            }
        });
    }


    /**
     * 撤回电子签章
     *
     * @param ids 数据主键列表
     */
    @Override
    public void withdrawalContract(List<String> ids) {
        // 这里查看一条
        List<ReconciliationDocListVo> reconciliationDocListVos = validateIds(ids);

        List<ReconciliationDocListVo> dateList = reconciliationDocListVos.stream()
            //获取不是 待客户确认状态的数据集  只有待客户确认可以撤回
            .filter(x -> !ReconciliationStatusEnum.CONFIRMED.getCode().equals(x.getStatementStatus()))
            .collect(Collectors.toList());
        Validate.isTrue(CollectionUtils.isEmpty(dateList), "数据集中存在非待客户确认状态的数据，只有待客户确认可以撤回");

        reconciliationDocListVos.forEach(reconciliationDocListVo -> {
            if (ReconciliationStatusEnum.WITHDRAW.getCode().equals(reconciliationDocListVo.getStatementStatus())) {
                return;
            }

            Validate.notBlank(reconciliationDocListVo.getContractId(), "合同ID不能为空");

            ElectronicSignatureVo signatureVo = baseElectronicSignature.withdrawalElectronicSignature(reconciliationDocListVo);
            if (Objects.nonNull(signatureVo) && signatureVo.isSuccess()) {
                reconciliationDocListVo.setStatementStatus(ReconciliationStatusEnum.WITHDRAW.getCode());

                // 推送ecrm todo 改成 Rocket事物mq
                baseEcrmBuildService.syncECrm(reconciliationDocListVo, null);

                // 修改库
                this.reconciliationDocListRepository.lambdaUpdate()
                    .eq(UuidEntity::getId, reconciliationDocListVo.getId())
                    .set(ReconciliationDocListEntity::getStatementStatus, ReconciliationStatusEnum.WITHDRAW.getCode())
                    .set(ReconciliationDocListEntity::getContractId, null)
                    .set(ReconciliationDocListEntity::getDocumentsId, null)
                    .update();
            }
        });
    }


    /**
     * 查看电子签章
     *
     * @param ids 数据主键列表
     * @return {@link List}<{@link ReconciliationDocListFileVo}>
     */
    @Override
    public List<ReconciliationDocListFileVo> browseContract(List<String> ids) {
        List<ReconciliationDocListVo> reconciliationDocListVos = validateIds(ids);

        List<ReconciliationDocListFileVo> result = new ArrayList<>(ids.size());
        reconciliationDocListVos.forEach(reconciliationDocListVo -> {
            ReconciliationDocListFileVo reconciliationDocListFileVo =
                reconciliationDocListFileVoService.findDetailByStatementCodeAndFileType(
                    reconciliationDocListVo.getStatementCode(), ReconciliationFileTypeEnum.MENG_NIU.getCode());
            if (reconciliationDocListFileVo == null) {
                reconciliationDocListFileVo =
                    reconciliationDocListFileVoService.findDetailByStatementCodeAndFileType(
                        reconciliationDocListVo.getStatementCode(), ReconciliationFileTypeEnum.BO_ZHI.getCode());
            }
            result.add(reconciliationDocListFileVo);
        });
        return result;
    }


    /**
     * 下载电子签章
     *
     * @param ids 数据主键列表
     * @return {@link List}<{@link ReconciliationDocListFileVo}>
     */
    @Override
    public List<ReconciliationDocListFileVo> downloadContract(List<String> ids) {
        List<ReconciliationDocListVo> reconciliationDocListVos = validateIds(ids);

        List<ReconciliationDocListFileVo> result = new ArrayList<>(ids.size());
        reconciliationDocListVos.forEach(reconciliationDocListVo -> {
            // 如果可以查到蒙牛的文件 就不用再去重新下载一次
            ReconciliationDocListFileVo reconciliationDocListFileVo =
                this.reconciliationDocListFileVoService.findDetailByStatementCodeAndFileType(
                    reconciliationDocListVo.getStatementCode(), ReconciliationFileTypeEnum.MENG_NIU.getCode());
            if (reconciliationDocListFileVo != null) {
                result.add(reconciliationDocListFileVo);
                return;
            }

            // 调用第三方接口下载文件
            ElectronicSignatureVo<FileVo> signatureVo = baseElectronicSignature.downloadElectronicSignature(reconciliationDocListVo);
            Optional<FileVo> fileVo = Optional.ofNullable(signatureVo).map(ElectronicSignatureVo::getResult);
            fileVo.ifPresent(file -> {
                ReconciliationDocListFileVo docListFileVo = new ReconciliationDocListFileVo();
                docListFileVo.setStatementCode(reconciliationDocListVo.getStatementCode());
                docListFileVo.setFileType(ReconciliationFileTypeEnum.MENG_NIU.getCode());
                docListFileVo.setFileCode(file.getFileCode());
                docListFileVo.setTenantCode(TenantUtils.getTenantCode());
                docListFileVo.setOriginalFileName(file.getOriginalFileName());
                this.reconciliationDocListFileVoService.create(docListFileVo);
                result.add(docListFileVo);
            });
        });

        return result;
    }

    @Override
    public byte[] findFileZipByIds(List<String> ids) throws IOException {
        log.info("进入批量下载电子签章方法");
        if (CollectionUtils.isEmpty(ids)) {
            return new byte[0];
        }
        Validate.isTrue(ids.size() <= 200, "最多下载200个文件!");
        List<ReconciliationDocListVo> reconciliationDocListVos = validateIds(ids);
        List<ReconciliationDocListFileVo> result = new ArrayList<>(ids.size());
        reconciliationDocListVos.forEach(reconciliationDocListVo -> {
            // 如果可以查到蒙牛的文件 就不用再去重新下载一次
            ReconciliationDocListFileVo reconciliationDocListFileVo =
                this.reconciliationDocListFileVoService.findDetailByStatementCodeAndFileType(
                    reconciliationDocListVo.getStatementCode(), ReconciliationFileTypeEnum.BO_ZHI.getCode());
            if (reconciliationDocListFileVo != null) {
                //if (StringUtils.isNotBlank(reconciliationDocListFileVo.getRelativeLocal())) {
                    result.add(reconciliationDocListFileVo);
                    return;
                //}
            }

            // 调用第三方接口下载文件
            ElectronicSignatureVo<NewFileVo> signatureVo = baseElectronicSignature.downloadElectronicSignatureNew(reconciliationDocListVo);
            if (null != signatureVo) {
                Optional<NewFileVo> fileVo = Optional.ofNullable(signatureVo).map(ElectronicSignatureVo::getResult);
                fileVo.ifPresent(file -> {
                    ReconciliationDocListFileVo docListFileVo = new ReconciliationDocListFileVo();
                    docListFileVo.setStatementCode(reconciliationDocListVo.getStatementCode());
                    docListFileVo.setFileType(ReconciliationFileTypeEnum.MENG_NIU.getCode());
                    docListFileVo.setFileCode(file.getFileCode());
                    docListFileVo.setTenantCode(TenantUtils.getTenantCode());
                    docListFileVo.setRelativeLocal(file.getRelativeLocal());
                    docListFileVo.setOriginalFileName(file.getOriginalFileName());
                    this.reconciliationDocListFileVoService.create(docListFileVo);
                    result.add(docListFileVo);
                });
            }
        });
        log.info("查询到的电子签章文件,{}", JSON.toJSONString(result));
        Map<String, ReconciliationDocListVo> params = reconciliationDocListVos.stream().collect(Collectors.toMap(ReconciliationDocListVo::getStatementCode, s -> s));
        if (!CollectionUtils.isEmpty(result)) {
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            ZipOutputStream zipOut = new ZipOutputStream(bos);
            for (ReconciliationDocListFileVo file : result) {
                ReconciliationDocListVo reconciliationDocListVo = params.get(file.getStatementCode());
                if (reconciliationDocListVo != null) {
                    OrdinaryFileVo venus = fileHandleService.findById(file.getFileCode());
                    if(Objects.isNull(venus)) continue;

                    log.info("查询venus文件结果为:{}",JSON.toJSONString(venus));
                    byte[] fileBytes = fileHandleService.findContentByFilePathAndFileRename(venus.getRelativeLocal(), venus.getFileName());
                    //log.info("查询venus文件字节长度结果为:{}",);
                    String fileName = StringUtils.join(reconciliationDocListVo.getCustomerName(), com.biz.crm.mn.common.base.util.DateUtil.format(reconciliationDocListVo.getReconciliationMonth(),"yyyy-MM"), reconciliationDocListVo.getAccountReconciliationType());
                    zipOut.putNextEntry(new ZipEntry(fileName + "." + venus.getPrefix()));
                    zipOut.write(fileBytes, 0, fileBytes.length);
                    zipOut.closeEntry();
                }
            }
            zipOut.flush();
            zipOut.close();
            return bos.toByteArray();
        }
        return new byte[0];
    }

    /**
     * 催签电子签章
     *
     * @param ids 数据主键列表
     */
    @Override
    public void contractExpediting(List<String> ids) {
        List<ReconciliationDocListVo> reconciliationDocListVos = validateIds(ids);

        reconciliationDocListVos.forEach(reconciliationDocListVo -> {
            Validate.isTrue(ReconciliationStatusEnum.CONFIRMED.getCode().equals(reconciliationDocListVo.getStatementStatus()),"状态为待客户确认的时候，可以点击催签");
            // 调用第三方接口
            ElectronicSignatureVo electronicSignatureVo = baseElectronicSignature.expeditingElectronicSignature(reconciliationDocListVo);

            if (Objects.isNull(electronicSignatureVo) || !electronicSignatureVo.isSuccess()) {
                throw new IllegalArgumentException("催签失败");
            }
        });
    }


    private String amountCharTransfer(String sapAmount) {
        if (StringUtils.isNotBlank(sapAmount)) {
            if (sapAmount.contains(ReconciliationConstant.SHORT_LINE)) {
                return ReconciliationConstant.SHORT_LINE + sapAmount.trim().replace(ReconciliationConstant.SHORT_LINE, StringUtils.EMPTY);
            } else {
                return sapAmount.trim();
            }
        }
        return BigDecimal.ZERO.toPlainString();
    }

    /**
     * 自动生产对账单
     *
     * @param
     * @param date
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023-07-14 16:25
     */
    @Override
    public void autoBuildReconciliation(Date date) {
        if (Objects.isNull(date)) {
            date = new Date();
        }
        Pageable page = PageRequest.of(1, 20);
        Page<TpmAccountReconciliationRuleRespVo> pageResult = null;
        do {
            pageResult = this.accountReconciliationRuleService.findCurrentEffectiveRule(page);
            log.info("=====>    对账规则 page[{}/{}]    <=====", pageResult.getCurrent(), pageResult.getPages());
            buildReconciliation(date, pageResult.getRecords());
            page = page.next();
            //每页20条,最多循环10000次,防止死循环
        } while (pageResult.hasNext()
                && page.getPageNumber() < ReconciliationConstant.LOOP_TIMES_MAX);
    }

    /**
     * 自动生产对账单
     *
     * @param
     * @param date
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023-07-14 16:25
     */
    @Override
    public List<String> autoBuildReconciliationForOut(Date date) {
        List<String> idList = Lists.newArrayList();
        if (Objects.isNull(date)) {
            date = new Date();
        }
        Pageable page = PageRequest.of(1, 20);
        Page<TpmAccountReconciliationRuleRespVo> pageResult = null;
        do {
            pageResult = this.accountReconciliationRuleService.findCurrentEffectiveRule(page);
            log.info("=====>    对账规则 page[{}/{}]    <=====", pageResult.getCurrent(), pageResult.getPages());
            buildReconciliation(date, pageResult.getRecords());
            page = page.next();
            //每页20条,最多循环10000次,防止死循环
        } while (pageResult.hasNext()
                && page.getPageNumber() < ReconciliationConstant.LOOP_TIMES_MAX);
        return idList;
    }

    @Override
    public Page<ReconciliationDocListVo> findPageForOut(Pageable pageable, ReconciliationDocListDto dto) {
        return reconciliationDocListRepository.findPageForOut(pageable, dto);
    }

    private List<String> buildReconciliationForOut(Date date, List<TpmAccountReconciliationRuleRespVo> ruleRespVos) {
        List<String> idList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(ruleRespVos)) {
            return Lists.newArrayList();
        }

        if (Objects.isNull(date)) {
            date = new Date();
        }

        Map<String, List<TpmAccountReconciliationRuleRespVo>> reconciliationRuleMap = ruleRespVos.stream()
            .filter(k -> StringUtils.isNotBlank(k.getAccountReconciliationRuleCode()))
            .collect(Collectors.groupingBy(TpmAccountReconciliationRuleRespVo::getAccountReconciliationRuleCode));

        // 1. 获取所有时间范围
        Date finalDate = date;
        List<TpmAccountReconciliationRuleDateConfigRespVo> ruleDateConfigRespVos = ruleRespVos.stream()
            .map(TpmAccountReconciliationRuleRespVo::getDateConfigList).flatMap(Collection::stream)
            .collect(Collectors.collectingAndThen(Collectors.toList(),
                a -> a.stream()
                    //获取需要当天执行的对账单规则
                    .filter(x -> StrUtil.isNotEmpty(x.getCalculateDate())
                        && DateUtil.isSameDay(finalDate, DateUtil.parseDateTime(x.getCalculateDate())))
                    .collect(Collectors.toList())));

        if (!CollectionUtils.isEmpty(ruleDateConfigRespVos)) {
            // 2. 遍历对账规则时间配置
            ruleDateConfigRespVos.forEach(dataConfig -> {

                // 3. 获取对应的对账规则
                List<TpmAccountReconciliationRuleRespVo> rules =
                    reconciliationRuleMap.get(dataConfig.getAccountReconciliationRuleCode());
                if (!CollectionUtils.isEmpty(rules)) {

                    //执行任务
                    rules.forEach(rule -> {
                        ReconciliationDocListTaskDto taskDto = ReconciliationDocListTaskDto.builder()
                            //对账类型
                            .accountReconciliationType(rule.getAccountReconciliationType())
                            //对账规则编码
                            .accountReconciliationRuleCode(rule.getAccountReconciliationRuleCode())
                            //计算时间
                            .calculationTime(DateUtil.parseDateTime(dataConfig.getCalculateDate()))
                            //业态编码
                            .businessFormatCode(rule.getBusinessFormatCode())
                            //业务单元编码
                            .businessUnitCode(rule.getBusinessUnitCode())
                            //计算结束时间
                            .endDate(DateUtil.parseDateTime(dataConfig.getEndDate()))
                            //计算开始时间
                            .startDate(DateUtil.parseDateTime(dataConfig.getStartDate()))
                            .build();
                        log.info("生成对账单：ReconciliationDocListTaskDto：{}", JSONObject.toJSONString(taskDto));
                        List<ReconciliationDocListVo> docListVos = reconciliationDocListTaskService.executeTask(taskDto);
                        idList.addAll(createBatch(docListVos));
                    });
                }
            });
        }
        return idList;
    }

    @Override
    public List<String> handleManualGenerate(Date date) {
        List<String> idList = Lists.newArrayList();
        if (Objects.isNull(date)) {
            date = new Date();
        }
        Pageable page = PageRequest.of(1, 20);
        Page<TpmAccountReconciliationRuleRespVo> pageResult = null;
        do {
            pageResult = this.accountReconciliationRuleService.findCurrentEffectiveRule(page);
            log.info("=====>    对账规则 page[{}/{}]    <=====", pageResult.getCurrent(), pageResult.getPages());
            idList.addAll(buildReconciliationForOut(date, pageResult.getRecords()));
            page = page.next();
            //每页20条,最多循环10000次,防止死循环
        } while (pageResult.hasNext()
                && page.getPageNumber() < ReconciliationConstant.LOOP_TIMES_MAX);
        return idList;
    }

    /**
     * 构建对账单
     *
     * @param date
     * @param ruleRespVos 对账规则
     */
    private void buildReconciliation(Date date, List<TpmAccountReconciliationRuleRespVo> ruleRespVos) {

        if (CollectionUtils.isEmpty(ruleRespVos)) {
            return;
        }

        if (Objects.isNull(date)) {
            date = new Date();
        }
        // 对账规则
        Map<String, List<TpmAccountReconciliationRuleRespVo>> reconciliationRuleMap = ruleRespVos.stream()
                .filter(k -> StringUtils.isNotBlank(k.getAccountReconciliationRuleCode()))
                .collect(Collectors.groupingBy(TpmAccountReconciliationRuleRespVo::getAccountReconciliationRuleCode));
        log.info("分组[{}]完成开始生成对账单", reconciliationRuleMap.size());
        // 1. 获取所有时间范围
        Date finalDate = date;
        List<TpmAccountReconciliationRuleDateConfigRespVo> ruleDateConfigRespVos = ruleRespVos.stream()
            .map(TpmAccountReconciliationRuleRespVo::getDateConfigList).flatMap(Collection::stream)
            .collect(Collectors.collectingAndThen(Collectors.toList(),
                a -> a.stream()
                    //获取需要当天执行的对账单规则
                    .filter(x -> StrUtil.isNotEmpty(x.getCalculateDate())
                        && DateUtil.isSameDay(finalDate, DateUtil.parseDateTime(x.getCalculateDate())))
                    .collect(Collectors.toList())));

        if (CollectionUtils.isEmpty(ruleDateConfigRespVos)) {
            log.error("对账规则无有效对账单可生成[{}]", DateUtil.format(finalDate, DatePattern.NORM_DATETIME_PATTERN));
            return;
        }
        // 2. 遍历对账规则时间配置
        ruleDateConfigRespVos.forEach(dataConfig -> {

            // 3. 获取对应的对账规则
            List<TpmAccountReconciliationRuleRespVo> rules =
                    reconciliationRuleMap.get(dataConfig.getAccountReconciliationRuleCode());
            if (CollectionUtils.isEmpty(rules)) {
                log.error("对账规则[{}]无有效对账单", dataConfig.getAccountReconciliationRuleCode());
                return;
            }
            log.error("对账规则[{}]开始生成对账单", dataConfig.getAccountReconciliationRuleCode());
            //执行任务
            rules.forEach(rule -> {
                ReconciliationDocListTaskDto taskDto = ReconciliationDocListTaskDto.builder()
                        //对账类型
                        .accountReconciliationType(rule.getAccountReconciliationType())
                        //对账规则编码
                        .accountReconciliationRuleCode(rule.getAccountReconciliationRuleCode())
                        //计算时间
                        .calculationTime(DateUtil.parseDateTime(dataConfig.getCalculateDate()))
                        //业态编码
                        .businessFormatCode(rule.getBusinessFormatCode())
                        //业务单元编码
                        .businessUnitCode(rule.getBusinessUnitCode())
                        //计算结束时间
                        .endDate(DateUtil.parseDateTime(dataConfig.getEndDate()))
                        //计算开始时间
                        .startDate(DateUtil.parseDateTime(dataConfig.getStartDate()))
                        .build();
                log.info("生成对账单：ReconciliationDocListTaskDto：{}", JSONObject.toJSONString(taskDto));
                List<ReconciliationDocListVo> docListVos = reconciliationDocListTaskService.executeTask(taskDto);
                createBatch(docListVos);
            });
        });

    }


    /**
     * 重新执行的失败的任务
     *
     * @param date
     */
    @Override
    public void autoBuildFailReconciliation(Date date) {

        Pageable page = PageRequest.of(1, 20);
        ReconciliationJobLogDto dto = new ReconciliationJobLogDto();
        dto.setIsSuccess(false);
        dto.setRetryTimes(ReconciliationConstant.RETRY_TIMES_MAX);
        // 扫描执行失败任务
        Page<ReconciliationJobLogVo> failJobPage = null;
        do {
            failJobPage = this.reconciliationJobLogVoService.findByConditions(page, dto);
            log.info("=====>    对账失败日志 page[{}/{}]    <=====", failJobPage.getCurrent(), failJobPage.getPages());
            buildFailReconciliation(failJobPage.getRecords());
            page = page.next();
            //每页20条,最多循环10000次,防止死循环
        } while (failJobPage.hasNext()
                && page.getPageNumber() < ReconciliationConstant.LOOP_TIMES_MAX);


    }

    /**
     * 对账执行失败记录,开始执行
     *
     * @param failJobInfos
     */
    private void buildFailReconciliation(List<ReconciliationJobLogVo> failJobInfos) {
        if (CollectionUtil.isEmpty(failJobInfos)) {
            return;
        }
        failJobInfos.forEach(failJobInfo -> {
            log.info("对账失败 信息{}", failJobInfo);
            //执行任务
            ReconciliationDocListTaskDto taskDto = ReconciliationDocListTaskDto.builder()
                .accountReconciliationType(failJobInfo.getAccountReconciliationType())
                .accountReconciliationRuleCode(failJobInfo.getAccountReconciliationRuleCode())
                .calculationTime(failJobInfo.getCalculationTime())
                .businessFormatCode(failJobInfo.getBusinessFormatCode())
                .businessUnitCode(failJobInfo.getBusinessUnitCode())
                .endDate(failJobInfo.getEndDate())
                .startDate(failJobInfo.getStartDate())
                .build();
            List<ReconciliationDocListVo> docListVos = reconciliationDocListTaskService.executeTask(taskDto);
            createBatch(docListVos);
        });
    }


}
