package com.biz.crm.tpm.business.reconciliation.doc.list.local.service.internal;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.constant.ReconciliationConstant;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.dto.ReconciliationDocListTaskDto;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.dto.ReconciliationGenerateDto;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.dto.ReconciliationJobLogDto;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.enums.ReconciliationBuesinssTypeEnum;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.service.ReconciliationJobLogVoService;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.strategy.AbstractReconciliationGenerateStrategy;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.vo.ReconciliationDocListVo;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.vo.ReconciliationJobLogVo;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StopWatch;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

@Component
@Slf4j
public class ReconciliationDocListTaskService {

    /**
     * 对账单任务日志记录
     */
    @Autowired(required = false)
    private ReconciliationJobLogVoService reconciliationJobLogVoService;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    /**
     * 对账单生成策略集合
     */
    @Autowired(required = false)
    protected List<AbstractReconciliationGenerateStrategy> reconciliationGenerateStrategies;


    /**
     * 执行任务
     *
     * @param reconciliationDocListTaskDto 任务dto
     * @return {@link List}<{@link ReconciliationDocListVo}>
     */
    public List<ReconciliationDocListVo> executeTask(ReconciliationDocListTaskDto reconciliationDocListTaskDto) {
        // 记录任务执行
        if (StrUtil.isEmpty(reconciliationDocListTaskDto.getAccountReconciliationRuleCode())) {
            return Lists.newArrayList();
        }
        String accountReconciliationRuleCode = reconciliationDocListTaskDto.getAccountReconciliationRuleCode();
        // 根据对账规则code做锁，后续可以根据具体业务变化细粒度一下
        String lock = ReconciliationConstant.REDIS_LOCK + accountReconciliationRuleCode;
        // 尝试加锁5秒，2小时自动释放锁
        boolean locked = redisLockService.tryLock(lock, TimeUnit.HOURS, 2, 3);
        if (!locked) {
            log.error("=====>      当前对账规则[{}]有任务执行中,请勿重复操作!      <=====", accountReconciliationRuleCode);
            return new ArrayList<>();
        }
        try {
            return this.runTask(reconciliationDocListTaskDto);
        } catch (Exception e) {
            log.error("", e);
            throw new IllegalArgumentException(e.getMessage(), e);
        } finally {
            redisLockService.unlock(lock);
        }
    }


    /**
     * 运行任务(不要改成public，这里不是入口,入口走executeTask,没有锁的操作不要执行  防止出现重复数据)
     *
     * @return
     */
    private List<ReconciliationDocListVo> runTask(ReconciliationDocListTaskDto dto) {
        StopWatch stopWatch = new StopWatch();
        stopWatch.start();

        // 构建查询参数
        ReconciliationJobLogDto jobLogDto = ReconciliationJobLogDto.builder()
                // 执行业务参数
                .businessUnitCode(dto.getBusinessUnitCode())
                .businessFormatCode(dto.getBusinessFormatCode())
                .accountReconciliationRuleCode(dto.getAccountReconciliationRuleCode())
                .accountReconciliationType(dto.getAccountReconciliationType())
                .startDate(dto.getStartDate())
                .endDate(dto.getEndDate())
                .calculationTime(Objects.isNull(dto.getCalculationTime()) ? null
                        : dto.getCalculationTime())
                .build();

        ReconciliationJobLogVo jobLogVo = this.reconciliationJobLogVoService.findUniqueJobInfo(jobLogDto);
        if (jobLogVo == null) {
            // 如果不存在此任务日志就代表第一次执行
            jobLogVo = ReconciliationJobLogVo.builder()
                    .executionStartTime(new Date())
                    .isSuccess(false).retryTimes(0)
                    // 执行业务参数
                    .businessUnitCode(dto.getBusinessUnitCode())
                    .businessFormatCode(dto.getBusinessFormatCode())
                    .accountReconciliationRuleCode(dto.getAccountReconciliationRuleCode())
                    .accountReconciliationType(dto.getAccountReconciliationType())
                    .startDate(dto.getStartDate())
                    .endDate(dto.getEndDate())
                    .calculationTime(Objects.isNull(dto.getCalculationTime()) ? null
                            : dto.getCalculationTime())
                    .build();

            this.reconciliationJobLogVoService.create(jobLogVo);
        } else {
            //如果标记为空,或失败  执行一次
            if (Objects.isNull(jobLogVo.getIsSuccess())
                    || !jobLogVo.getIsSuccess()) {

                jobLogVo.setRetryTimes(jobLogVo.getRetryTimes() + 1);
                this.reconciliationJobLogVoService.update(jobLogVo);
            }
            // 如果已经执行过了就不在执行
            else {
                log.info("生成对账单：已生成过对账单，跳过本次生成");
                return Lists.newArrayList();
            }
        }

        // 根据对账类型执行生成对账数据逻辑
        try {
            boolean smallBusinessUnit = BusinessUnitEnum.isLittleBusinessUnit(dto.getBusinessUnitCode());

            AbstractReconciliationGenerateStrategy strategy;
            //小业务单元，生成逻辑与主体相同
            if (smallBusinessUnit) {
                strategy = this.getGenerateStrategy(dto.getAccountReconciliationType(), BusinessUnitEnum.HEADQUARTERS.getCode());
            }else {
                strategy = this.getGenerateStrategy(dto.getAccountReconciliationType(), dto.getBusinessUnitCode());
            }
            Assert.notNull(strategy, "不存在的业务类型[" + dto.getAccountReconciliationType()
                    + "|" + dto.getBusinessUnitCode() + "],请检查代码业务编码是否有对应实现策略");
            ReconciliationGenerateDto generateDto = new ReconciliationGenerateDto();
            generateDto.setBusinessFormatCode(dto.getBusinessFormatCode());
            generateDto.setBusinessUnitCode(dto.getBusinessUnitCode());
            generateDto.setStartDate(DateUtil.formatDateTime(dto.getStartDate()));
            generateDto.setEndDate(DateUtil.formatDateTime(dto.getEndDate()));
            generateDto.setStartDateTime(dto.getStartDate());
            generateDto.setEndDateTime(dto.getEndDate());
            generateDto.setAccountReconciliationRuleCode(dto.getAccountReconciliationRuleCode());
            generateDto.setAccountReconciliationType(dto.getAccountReconciliationType());
            generateDto.setCustomerCode(dto.getCustomerCode());
            generateDto.setSmallBusinessUnit(smallBusinessUnit);

            // 记录任务执行成功
            jobLogVo.setIsSuccess(true);

            log.info("生成对账单：ReconciliationGenerateDto：{}", JSONObject.toJSONString(generateDto));

            List<ReconciliationDocListVo> resultList = strategy.generateDate(generateDto);
            if (smallBusinessUnit){
                resultList.forEach(r -> r.setBusinessUnitCode(dto.getBusinessUnitCode()));
            }
            return resultList;
        } catch (Exception e) {
            jobLogVo.setIsSuccess(false);
            jobLogVo.setErrorMsg(StrUtil.isEmpty(e.getMessage()) ? "空指针异常" : e.getMessage());
            log.error("对账单生成执行失败,{}", e.getMessage());
            log.error("", e);
            throw new RuntimeException(e.getMessage());
        } finally {
            stopWatch.stop();
            jobLogVo.setExecutionMillisecond((int) stopWatch.getTotalTimeMillis());
            jobLogVo.setExecutionEndTime(new Date());
            this.reconciliationJobLogVoService.update(jobLogVo);
        }

    }

    /**
     * 根据对账单类型+业务单元编码  匹配对对账单生产策略
     *
     * @param reconciliationType 对账单类型
     * @param businessUnitCode   业务单元编码
     * @return
     */
    public AbstractReconciliationGenerateStrategy getGenerateStrategy(String reconciliationType, String businessUnitCode) {
        List<AbstractReconciliationGenerateStrategy> generateStrategies =
                reconciliationGenerateStrategies.stream()
                        .filter(x -> x.getReconciliationType().equals(reconciliationType))
                        .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(generateStrategies)) {
            return null;
        }
        AbstractReconciliationGenerateStrategy strategy = null;
        if (generateStrategies.size() > 1) {
            // 如果存在多个 根据业务单元区分（业务单元唯一）
            ReconciliationBuesinssTypeEnum businessTypeEnum =
                    ReconciliationBuesinssTypeEnum.findByBusinessUnitCode(businessUnitCode);
            if (businessTypeEnum != null) {
                Optional<AbstractReconciliationGenerateStrategy> reconciliationGenerateStrategy = generateStrategies
                        .stream().filter(x -> businessTypeEnum.getCode().equals(x.getBusinessType()))
                        .findFirst();
                if (reconciliationGenerateStrategy.isPresent()) {
                    strategy = reconciliationGenerateStrategy.get();
                }
            } else {
                log.error("=====>   对账单策略匹配时,业务单元编码[{}]不存在<   =====", businessUnitCode);
            }
        } else {
            strategy = generateStrategies.get(0);
        }
        return strategy;
    }
}
