package com.biz.crm.tpm.business.reconciliation.doc.list.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.tpm.business.reconciliation.doc.list.local.entity.ReconciliationJobLogEntity;
import com.biz.crm.tpm.business.reconciliation.doc.list.local.repository.ReconciliationJobLogRepository;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.dto.ReconciliationJobLogDto;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.dto.ReconciliationJobLogLogEventDto;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.event.ReconciliationJobLogEventListener;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.event.log.ReconciliationJobLogLogEventListener;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.service.ReconciliationJobLogVoService;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.vo.ReconciliationJobLogVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import liquibase.pro.packaged.E;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

/**
 * TPM-任务执行日志(ReconciliationJobLog)表服务实现类
 *
 * @author liuyifan
 * @date 2023-02-22 11:04:04
 */
@Slf4j
@Service("reconciliationJobLogService")
public class ReconciliationJobLogVoServiceImpl implements ReconciliationJobLogVoService {

    @Autowired(required = false)
    private ReconciliationJobLogRepository reconciliationJobLogRepository;

    @Autowired(required = false)
    private List<ReconciliationJobLogEventListener> eventListeners;

    @Autowired
    @Qualifier("nebulaToolkitService")
    private NebulaToolkitService nebulaToolkitService;

    @Autowired
    private NebulaNetEventClient nebulaNetEventClient;

    @Override
    public Page<ReconciliationJobLogVo> findByConditions(Pageable pageable, ReconciliationJobLogDto dto) {
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(1, 50));
        dto = Optional.ofNullable(dto).orElse(new ReconciliationJobLogDto());
        Page<ReconciliationJobLogVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.reconciliationJobLogRepository.findByConditions(page, dto);
    }

    @Override
    public ReconciliationJobLogVo findDetailById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }

        ReconciliationJobLogEntity reconciliationJobLog = this.reconciliationJobLogRepository.findById(id);
        if (reconciliationJobLog == null) {
            return null;
        }
        return nebulaToolkitService.copyObjectByWhiteList(reconciliationJobLog, ReconciliationJobLogVo.class,
                HashSet.class, ArrayList.class);
    }

    @Override
    @Transactional
    public ReconciliationJobLogVo create(ReconciliationJobLogVo reconciliationJobLogVo) {
        this.createValidation(reconciliationJobLogVo);
        reconciliationJobLogVo.setTenantCode(TenantUtils.getTenantCode());
        reconciliationJobLogVo.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        reconciliationJobLogVo.setEnableStatus(EnableStatusEnum.ENABLE.getCode());

        ReconciliationJobLogEntity reconciliationJobLog = nebulaToolkitService
                .copyObjectByWhiteList(reconciliationJobLogVo, ReconciliationJobLogEntity.class, HashSet.class, ArrayList.class);
        this.reconciliationJobLogRepository.saveOrUpdate(reconciliationJobLog);

        reconciliationJobLogVo.setId(reconciliationJobLog.getId());

        // crud监听事件
        if (!CollectionUtils.isEmpty(eventListeners)) {
            for (ReconciliationJobLogEventListener reconciliationJobLogEventListener : eventListeners) {
                reconciliationJobLogEventListener.onCreate(reconciliationJobLogVo);
            }
        }

        // 发送通知
        ReconciliationJobLogLogEventDto logEventDto = new ReconciliationJobLogLogEventDto();
        logEventDto.setOriginal(null);
        logEventDto.setNewest(reconciliationJobLogVo);
        SerializableBiConsumer<ReconciliationJobLogLogEventListener, ReconciliationJobLogLogEventDto> onCreate =
                ReconciliationJobLogLogEventListener::onCreate;
        this.nebulaNetEventClient.publish(logEventDto, ReconciliationJobLogLogEventListener.class, onCreate);
        return reconciliationJobLogVo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class, propagation = Propagation.REQUIRES_NEW)
    public ReconciliationJobLogVo update(ReconciliationJobLogVo reconciliationJobLogVo) {
        this.updateValidation(reconciliationJobLogVo);
        String currentId = reconciliationJobLogVo.getId();
        ReconciliationJobLogEntity current = reconciliationJobLogRepository.findById(currentId);
        current = Validate.notNull(current, "修改信息不存在");
        ReconciliationJobLogVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(current,
                ReconciliationJobLogVo.class, HashSet.class, ArrayList.class);

        current.setExecutionEndTime(reconciliationJobLogVo.getExecutionEndTime());
        current.setExecutionMillisecond(reconciliationJobLogVo.getExecutionMillisecond());
        current.setIsSuccess(reconciliationJobLogVo.getIsSuccess());

        current.setAccountReconciliationRuleCode(reconciliationJobLogVo.getAccountReconciliationRuleCode());

        this.reconciliationJobLogRepository.saveOrUpdate(current);
        // crud监听事件
        if (!CollectionUtils.isEmpty(eventListeners)) {
            for (ReconciliationJobLogEventListener reconciliationJobLogEventListener : eventListeners) {
                reconciliationJobLogEventListener.onUpdate(oldVo, reconciliationJobLogVo);
            }
        }

        // 发送修改通知
        ReconciliationJobLogLogEventDto logEventDto = new ReconciliationJobLogLogEventDto();
        logEventDto.setOriginal(oldVo);
        logEventDto.setNewest(reconciliationJobLogVo);
        SerializableBiConsumer<ReconciliationJobLogLogEventListener, ReconciliationJobLogLogEventDto> onUpdate =
                ReconciliationJobLogLogEventListener::onUpdate;
        this.nebulaNetEventClient.publish(logEventDto, ReconciliationJobLogLogEventListener.class, onUpdate);
        return reconciliationJobLogVo;
    }

    @Override
    @Transactional
    public void enableBatch(List<String> ids) {
        Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
        this.reconciliationJobLogRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
    }

    @Override
    @Transactional
    public void disableBatch(List<String> ids) {
        Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
        this.reconciliationJobLogRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
    }

    @Override
    @Transactional
    public void updateDelFlagByIds(List<String> ids) {
        Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
        this.reconciliationJobLogRepository.updateDelFlagByIds(ids);
    }

    @Override
    public ReconciliationJobLogVo findUniqueJobInfo(ReconciliationJobLogDto dto) {
        Validate.notBlank(dto.getBusinessFormatCode(), "业态不能为空");
        Validate.notBlank(dto.getBusinessUnitCode(), "业务单元不能为空");
        Validate.notNull(dto.getStartDate(), "计算开始时间不能为空");
        Validate.notNull(dto.getEndDate(), "计算结束时间不能为空");
        Validate.notNull(dto.getAccountReconciliationRuleCode(), "对账规则编码不能为空");
        ReconciliationJobLogEntity logEntity = this.reconciliationJobLogRepository.findUniqueJobInfo(dto);
        if (Objects.isNull(logEntity)) {
            return null;
        }
        return nebulaToolkitService.copyObjectByWhiteList(logEntity, ReconciliationJobLogVo.class, HashSet.class, ArrayList.class);
    }

    private void createValidation(ReconciliationJobLogVo reconciliationJobLogVo) {
        // TODO 具体实现
        this.validation(reconciliationJobLogVo);
    }

    private void updateValidation(ReconciliationJobLogVo reconciliationJobLogVo) {
        // TODO 具体实现
        this.validation(reconciliationJobLogVo);
    }

    private void validation(ReconciliationJobLogVo reconciliationJobLogVo) {
        // TODO 具体实现
    }
}
