package com.biz.crm.tpm.business.reconciliation.doc.list.sdk.strategy;

import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgSubComOrgService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgSubComOrgVo;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalVo;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.dto.ReconciliationGenerateDto;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.dto.ReconciliationSummaryDataDisplayDto;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.service.ReconciliationGenerateAsyncService;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.service.ReconciliationSummaryDataDisplayService;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.vo.ReconciliationAsyncResult;
import com.biz.crm.tpm.business.reconciliation.doc.list.sdk.vo.ReconciliationDocListVo;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.security.sdk.login.UserIdentity;
import com.google.common.collect.Lists;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.concurrent.Future;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 对账单策略
 *
 * @param <T>
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2023-07-17 23:22
 */
@Slf4j
public abstract class AbstractReconciliationGenerateStrategy<T> {

    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private TerminalVoService terminalVoService;

    @Autowired(required = false)
    private ReconciliationSummaryDataDisplayService reconciliationSummaryDataDisplayService;

    @Autowired(required = false)
    private ReconciliationGenerateAsyncService reconciliationGenerateAsyncService;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Autowired(required = false)
    private SalesOrgSubComOrgService salesOrgSubComOrgService;

    /**
     * 对账类型
     *
     * @return 对账类型编码
     */
    public abstract String getReconciliationType();


    /**
     * 业务类型 用于同对账类型 不同数据源的扩展
     *
     * @return {@link String}
     */
    public abstract String getBusinessType();


    /**
     * 生成对账数据
     *
     * @param dto 生成对账单据dto参数
     * @return 对账数据列表
     */
    @SneakyThrows
    public List<ReconciliationDocListVo> generateDate(ReconciliationGenerateDto dto) {
        if (Objects.isNull(dto)) {
            log.info("=====>    生成对账单信息为空!执行跳过!    <=====");
            log.error("=====>    生成对账单信息为空!执行跳过!    <=====");
            return Collections.emptyList();
        }
        log.info("=====>    生成对账单信息 {}    <=====", JSONObject.toJSONString(dto));
        //这里将开始时间和结束时间单独取出来，用于存储到对账单上,这里的时间是对账单规则中的开始结束时间。之前存储的结束时间是转换后yyyyMMdd格式，
        // 导致后续功能查询不到数据
        Date docStartTime = dto.getStartDateTime();
        Date docEndTime = dto.getEndDateTime();
        List<ReconciliationDocListVo> resultList = new ArrayList<>();
        //1. 构建请求查询参数
        try {
            DateTime startDate = DateUtil.parseDateTime(dto.getStartDate());
            DateTime endDate = DateUtil.parseDateTime(dto.getEndDate());
            dto.setStartDateTime(startDate);
            dto.setEndDateTime(endDate);
        } catch (Exception e) {
            log.error("", e);
            log.error("=====>    生成对账单信息时,时间转换错误 {}    <=====", JSONObject.toJSONString(dto));
        }


        //2. 获取客户信息集合
        log.info("=====>    生成对账单：获取客户信息 start {}    <=====", JSONObject.toJSONString(dto));
        //策略实现只能返回一个  key,value 集合
        Map<String, Set<String>> customerCodeMap = this.findCustomerCodes(dto);
        log.info("=====>    生成对账单：获取客户信息 end {}     <=====", customerCodeMap);
        Set<String> customerCodes = new HashSet<>();
        String type = null;
        for (Map.Entry<String, Set<String>> entry : customerCodeMap.entrySet()) {
            type = entry.getKey();
            customerCodes = entry.getValue();
        }
        if (CollectionUtils.isEmpty(customerCodes)) {
            log.info("=====>    生成对账单时,客户信息为空!执行跳过!    <=====");
            log.error("=====>    生成对账单时,客户信息为空!执行跳过!    <=====");
            return Collections.emptyList();
        }
        //2.1 客户信息详情
        customerCodes = customerCodes.stream().filter(Objects::nonNull).collect(Collectors.toSet());
        log.info("=====>    生成对账单：共[{}]个客户    <=====", customerCodes.size());
        log.info("=====>    生成对账单：详细客户信息{}    <=====", customerCodes);
        if (CollectionUtils.isEmpty(customerCodes)) {
            log.info("=====>    生成对账单时,客户信息过滤空后为空!执行跳过!    <=====");
            log.error("=====>    生成对账单时,客户信息过滤空后为空!执行跳过!    <=====");
            return Collections.emptyList();
        }
        //这里别用findBaseByCustomerCodes方法 是个get请求  量大报错，巨坑
        //需要判断，如果是分子公司，从门店查询
        int size = customerCodes.size();
        List<TerminalVo> terminalVos = new ArrayList<>();
        Map<String, List<TerminalVo>> terminalMap = new HashMap<>(size);
        Map<String, List<CustomerVo>> customerMap = new HashMap<>(size);
        Map<String, SalesOrgSubComOrgVo> salesOrgSubComOrgVoMap = new HashMap<>();
        log.info("生成对账单类型:type:{}",type);
        if (BusinessUnitEnum.SON_COMPANY.getCode().equals(type)) {
            //23/7/31 分子改为在客户管理主数据查询客户信息
            List<CustomerVo> baseByCustomerCodes = customerVoService.
                    findForSubReconciliation(Lists.newArrayList(customerCodes));
            if (CollectionUtils.isEmpty(baseByCustomerCodes)) {
                log.error("=====>    生成对账单：业务单元为分子时,查询客户管理，客户信息不存在!    <=====");
                log.error("=====>    详细客户信息:{}    <=====", customerCodes);
                throw new IllegalArgumentException("不存在客户信息");
            }
            customerMap = baseByCustomerCodes.stream()
                    .collect(Collectors.groupingBy(e -> e.getErpCode()
                            + "-" + e.getSalesInstitutionErpCode()));
            //分子HR组织编码
            List<String> salesInstitutionErpCodes = baseByCustomerCodes.stream().map(CustomerVo::getSalesInstitutionErpCode).distinct().collect(Collectors.toList());
            log.info("生成对账单客户信息:baseByCustomerCodes:{}", JsonUtils.obj2JsonString(baseByCustomerCodes));
            if (!CollectionUtils.isEmpty(salesInstitutionErpCodes)) {
                log.info("生成对账单销售机构信息:salesInstitutionErpCodes:{}", JsonUtils.obj2JsonString(salesInstitutionErpCodes));
                List<SalesOrgSubComOrgVo> salesOrgSubComOrgVos = this.salesOrgSubComOrgService.findBySaleOrgErpCodes(salesInstitutionErpCodes);
                log.info("生成对账单分子公司hr组织和销售组织关系:salesOrgSubComOrgVos:{}", JsonUtils.obj2JsonString(salesOrgSubComOrgVos));
                if (!CollectionUtils.isEmpty(salesOrgSubComOrgVos)) {
                    salesOrgSubComOrgVoMap = salesOrgSubComOrgVos.stream().collect(Collectors.toMap(SalesOrgSubComOrgVo::getSalesOrgCode, Function.identity()));
                }
            }
        } else {
            List<CustomerVo> baseByCustomerCodes = customerVoService.
                    findCustomerAndContactByCustomerCodes(Lists.newArrayList(customerCodes));
            if (CollectionUtils.isEmpty(baseByCustomerCodes)) {
                log.error("=====>    生成对账单：业务单元为非分子时,客户信息不存在!    <=====");
                log.error("=====>    详细客户信息:{}    <=====", customerCodes);
                throw new IllegalArgumentException("不存在客户信息");
            }
            customerMap = baseByCustomerCodes.stream()
                    .collect(Collectors.groupingBy(CustomerVo::getCustomerCode));
        }

        //3. 生成对应数量的编码
        //4. 遍历客户生成pdf
        List<ReconciliationSummaryDataDisplayDto> displayDtos = new ArrayList<>();
        List<Future<ReconciliationAsyncResult>> futureList = new ArrayList<>();
        UserIdentity loginUser = loginUserService.getLoginUser();
        for (String customerCode : customerCodes) {
            Future<ReconciliationAsyncResult> future = reconciliationGenerateAsyncService.cal(
                    customerCode, type, docStartTime, docEndTime, dto, loginUser,
                    this, terminalMap, customerMap,salesOrgSubComOrgVoMap);
            futureList.add(future);
        }
        log.info("=====>    生成对账单：futureList：{}    <=====", futureList.size());
        List<Future<ReconciliationAsyncResult>> runningList = new ArrayList<>();
        //最大循环次数
        int loopTimes = 7200;
        while (loopTimes > 0) {
            loopFuture(futureList, runningList, displayDtos, resultList);
            log.info("=====>    生成对账单：resultList：{}    <=====", resultList.size());
            if (CollectionUtils.isEmpty(runningList)) {
                log.info("=====>    生成对账单：resultList：0    <=====");
                break;
            }
            log.info("=====>    生成对账单：runningList：{}    <=====", runningList.size());
            futureList = new ArrayList<>(runningList);
            runningList.clear();
            loopTimes--;
            Thread.sleep(1000);
        }
        this.reconciliationSummaryDataDisplayService.batchCreate(displayDtos);
        log.info("=====>    对账单生成结束    <=====");
        return resultList;

    }

    @SneakyThrows
    public void loopFuture(List<Future<ReconciliationAsyncResult>> futureList,
                           List<Future<ReconciliationAsyncResult>> runningList,
                           List<ReconciliationSummaryDataDisplayDto> displayDtos,
                           List<ReconciliationDocListVo> resultList) {
        for (Future<ReconciliationAsyncResult> future : futureList) {
            if (!future.isDone()) {
                runningList.add(future);
                continue;
            }
            try {
                ReconciliationAsyncResult result = future.get();
                if(!StringUtils.isEmpty(result.getExceptionMsg())){
                    throw new IllegalArgumentException(result.getExceptionMsg());
                }
                if (Objects.nonNull(result.getDisplayDto())) {
                    displayDtos.add(result.getDisplayDto());
                }
                if (Objects.nonNull(result.getReconciliationDocListVo())) {
                    resultList.add(result.getReconciliationDocListVo());
                }
            } catch (Exception e) {
                log.error("生成对账单：" + e.getMessage());
                log.error("", e);
                throw new RuntimeException("生成对账单异常：" + e.getMessage());
            }
        }
    }


    /**
     * 获取客户数据列表
     *
     * @param dto 生成对账单信息
     * @return {@link List}<{@link T}>
     */
    public abstract Map<String, Set<String>> findCustomerCodes(ReconciliationGenerateDto dto);

    /**
     * 获取数据列表
     *
     * @param dto          生成对账单信息
     * @param customerCode 客户编码
     * @return {@link List}<{@link T}>
     */
    public abstract List<T> findDateList(ReconciliationGenerateDto dto, String customerCode);

    /**
     * 转换
     *
     * @param propertyName  属性名
     * @param propertyClass 属性类
     * @return {@link String}
     */
    public abstract String convert(String propertyName, T propertyClass);


    /**
     * 转换头
     *
     * @param propertyName 属性名
     * @param propertyList 属性集合
     * @return {@link String}
     */
    public abstract String convertHead(String propertyName, List<Object> propertyList);


    /**
     * 转换尾部
     */
    public abstract void convertTail(LinkedHashMap<String, String> linkedHashMap);

}
