package com.biz.crm.tpm.business.sales.goal.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.dto.MasterDataMdgBaseDto;
import com.biz.crm.tpm.business.sales.goal.sdk.constant.SalesGoalConstant;
import com.biz.crm.tpm.business.sales.goal.sdk.dto.SalesGoalDiscountSalesStatisticsDto;
import com.biz.crm.tpm.business.sales.goal.sdk.dto.SalesGoalDto;
import com.biz.crm.tpm.business.sales.goal.sdk.dto.SalesGoalStatisticsDto;
import com.biz.crm.tpm.business.sales.goal.sdk.service.SalesGoalService;
import com.biz.crm.tpm.business.sales.goal.sdk.vo.SalesGoalStatisticsVo;
import com.biz.crm.tpm.business.sales.goal.sdk.vo.SalesGoalVo;
import com.bizunited.nebula.security.sdk.login.UserIdentity;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * @author huojia
 * @date 2022年10月26日 11:08
 */
@RestController
@RequestMapping("/v1/sales/goal")
@Slf4j
@Api(tags = "销售目标")
public class SalesGoalController {

    @Autowired(required = false)
    private SalesGoalService salesGoalService;

    @Autowired(required = false)
    private RedisService redisService;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    /**
     * 分页查询所有数据
     *
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询所有数据")
    @GetMapping("findByConditions")
    public Result<Page<SalesGoalVo>> findByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                      @ApiParam(name = "salesGoalDto", value = "销售目标信息") SalesGoalDto dto) {
        try {
            Page<SalesGoalVo> page = this.salesGoalService.findByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 通过主键查询单条数据
     *
     * @param id 主键
     * @return 单条数据
     */
    @ApiOperation(value = "通过主键查询单条数据")
    @GetMapping("detail")
    public Result<SalesGoalVo> detail(@RequestParam("id") @ApiParam(name = "id", value = "主键id") String id) {
        try {
            SalesGoalVo salesGoalVo = this.salesGoalService.findById(id);
            return Result.ok(salesGoalVo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 修改数据
     *
     * @param dto 实体对象
     * @return 修改结果
     */
    @ApiOperation(value = "修改数据")
    @PatchMapping
    public Result<SalesGoalVo> update(@ApiParam(name = "salesGoalDto", value = "TPM-销售目标") @RequestBody SalesGoalDto dto) {
        try {
            SalesGoalVo result = this.salesGoalService.update(dto);
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 删除数据
     *
     * @param ids 主键结合
     * @return 删除结果
     */
    @DeleteMapping
    @ApiOperation(value = "删除数据")
    public Result<?> delete(@RequestParam("ids")  List<String> ids) {
        try {
            this.salesGoalService.delete(ids);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 条件查询
     *
     * @param dto      查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "条件查询")
    @PostMapping("listByConditions")
    public Result<List<SalesGoalVo>> listByConditions(@RequestBody SalesGoalDto dto) {
        try {
            List<SalesGoalVo> result = this.salesGoalService.listByConditions(dto);
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    /**
     * 客户+产品+年月,统计数据的分子公司出库量
     *
     * @param dto      查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "客户+产品+年月,统计数据的分子公司出库量")
    @PostMapping("outStockNumber")
    public Result<BigDecimal> outStockNumber(@RequestBody SalesGoalDto dto) {
        try {
            BigDecimal result = this.salesGoalService.outStockNumber(dto);
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 客户+产品+年月,统计经销商进货量
     *
     * @param dto      查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "客户+产品+年月,统计经销商进货量")
    @PostMapping("inStockNumber")
    public Result<BigDecimal> inStockNumber(@RequestBody SalesGoalDto dto) {
        try {
            BigDecimal result = this.salesGoalService.inStockNumber(dto);
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 客户+产品+年月,统计销售数量
     *
     * @param dto      查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "客户+产品+年月,统计销售数量")
    @PostMapping("getSaleAmount")
    public Result<BigDecimal> getSaleAmount(@RequestBody SalesGoalDto dto) {
        try {
            BigDecimal result = this.salesGoalService.getSaleAmount(dto);
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 客户+产品+年月,统计任务量
     *
     * @param dto      查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "客户+产品+年月,统计任务量")
    @PostMapping("getTaskAmount")
    public Result<BigDecimal> getTaskAmount(@RequestBody SalesGoalDto dto) {
        try {
            BigDecimal result = this.salesGoalService.getTaskAmount(dto);
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 分页查询所有数据
     *
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询所有数据")
    @GetMapping("statistics/findByConditions")
    public Result<Page<SalesGoalStatisticsVo>> findByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                @ApiParam(name = "salesGoalDto", value = "销售目标信息") SalesGoalStatisticsDto dto) {
        try {
            Page<SalesGoalStatisticsVo> page = this.salesGoalService.findByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 查询按区域分组的数据
     *
     * @param dto      查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "查询按区域分组的数据")
    @PostMapping("findListGroupByRegion")
    public Result<List<SalesGoalVo>> findListGroupByRegion(@ApiParam(name = "salesGoalDto", value = "销售目标信息") SalesGoalDto dto) {
        try {
            List<SalesGoalVo> result = this.salesGoalService.findListGroupByRegion(dto);
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 导入测试
     *
     * @param importsVos 分页对象
     * @return 所有数据
     */
    @ApiOperation(value = "导入测试")
    @PostMapping("importTest")
    public Result<?> importTest(@RequestBody @ApiParam(name = "importsVos", value = "销售目标导入信息") List<SalesGoalDto> importsVos) {
        try {
            // this.salesGoalImportService.importTest(importsVos);
            this.salesGoalService.importSave(importsVos);
            return Result.ok("导入成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 拉取日销售日报数据
     *
     * @param dto 分页对象
     * @return 所有数据
     */
    @ApiOperation(value = "拉取日销售日报数据")
    @PostMapping("pullSalesDayReport")
    public Result<?> pullSalesDayReport(@RequestBody MasterDataMdgBaseDto dto) {
        try {
            String lockKey = salesGoalService.getLockKey(dto);
            if (redisService.hasKey(lockKey)) {
                return Result.error("数据[" + lockKey.replace(SalesGoalConstant.SALES_GOAL_LOCK, "") + "]还在拉取中,请稍后再试!");
            }
            UserIdentity userIdentity = this.loginUserService.getLoginUser();
            this.salesGoalService.pullSalesDayReportAsync(dto,userIdentity);
            return Result.ok("数据异步拉取中");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 信息获取
     * @param salesGoal
     * @return
     */
    @PostMapping("findSalesGoal")
    public Result<?> findSalesGoal(@RequestBody SalesGoalDto salesGoal){

        try {

            List<SalesGoalVo> list = this.salesGoalService.findSalesGoal(salesGoal);

            return Result.ok(list);
        }catch (Exception e){

            log.error(e.getMessage(), e);

            return Result.error(e.getMessage());
        }
    }


    @ApiOperation(value = "指标计算查询")
    @GetMapping("listForVariableCal")
    public Result<List<SalesGoalVo>> listForVariableCal(@ApiParam(name = "salesGoalDto", value = "销售目标信息") SalesGoalDto dto) {
        try {
            return Result.ok(this.salesGoalService.listForVariableCal(dto));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("获取汇总金额")
    @PostMapping("getSalesGoalAmount")
    public Result<BigDecimal> getSalesGoalAmount(@RequestBody SalesGoalDto salesGoal){

        try {

            return Result.ok(this.salesGoalService.getSalesGoalAmount(salesGoal));
        }catch (Exception e){
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据业态，业务单元，分组，年月，统计折前销量，按照零售商+区域分组，-隔开
     *
     * @param dto
     * @return Map<String,BigDecimal>
     */
    @ApiOperation(value = "根据业态，业务单元，分组，年月，统计折前销量，按照零售商+区域分组，-隔开")
    @PostMapping("getSalesBeforeDiscount")
    public Result<Map<String,BigDecimal>> getSalesBeforeDiscount(@RequestBody SalesGoalDiscountSalesStatisticsDto dto) {
        try {
            Map<String,BigDecimal> map = this.salesGoalService.getSalesBeforeDiscount(dto);
            return Result.ok(map);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据业态，业务单元，分组，年月，统计折后销量，按照零售商+区域分组，-隔开
     *
     * @param dto
     * @return Map<String,BigDecimal>
     */
    @ApiOperation(value = "根据业态，业务单元，分组，年月，统计折后销量，按照零售商+区域分组，-隔开")
    @PostMapping("getSalesAfterDiscount")
    public Result<Map<String,BigDecimal>> getSalesAfterDiscount(@RequestBody SalesGoalDiscountSalesStatisticsDto dto) {
        try {
            Map<String,BigDecimal> map = this.salesGoalService.getSalesAfterDiscount(dto);
            return Result.ok(map);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }
}