package com.biz.crm.tpm.business.sales.goal.local.repository;

import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.local.entity.TenantFlagOpEntity;
import com.biz.crm.business.common.local.entity.UuidEntity;
import com.biz.crm.business.common.local.entity.UuidFlagOpEntity;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.dto.MasterDataMdgBaseDto;
import com.biz.crm.tpm.business.sales.goal.local.entity.SalesGoalEntity;
import com.biz.crm.tpm.business.sales.goal.local.mapper.SalesGoalMapper;
import com.biz.crm.tpm.business.sales.goal.sdk.dto.SalesGoalDiscountSalesStatisticsDto;
import com.biz.crm.tpm.business.sales.goal.sdk.dto.SalesGoalDto;
import com.biz.crm.tpm.business.sales.goal.sdk.vo.SalesGoalSumVo;
import com.biz.crm.tpm.business.sales.goal.sdk.vo.SalesGoalVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;

/**
 * @author huojia
 * @date 2022年10月26日 11:38
 */
@Component
public class SalesGoalRepository extends ServiceImpl<SalesGoalMapper, SalesGoalEntity> {

    @Resource
    private SalesGoalMapper salesGoalMapper;

    /**
     * 根据id、删除标记查询销售目标
     *
     * @param id
     * @param delFlag
     * @return com.biz.crm.tpm.business.sales.goal.local.entity.SalesGoalEntity
     * @author huojia
     * @date 2022/10/27 11:09
     **/
    public SalesGoalEntity getById(String id, String delFlag) {
        if (StringUtils.isEmpty(id)) {
            return null;
        }
        return this.lambdaQuery()
                .eq(UuidEntity::getId, id)
                .eq(StringUtils.isNotEmpty(delFlag), UuidFlagOpEntity::getDelFlag, delFlag)
                .one();
    }

    /**
     * 根据配置条件查询
     *
     * @param salesGoalDto
     * @return java.util.List<com.biz.crm.tpm.business.sales.goal.local.entity.SalesGoalEntity>
     * @author huojia
     * @date 2022/11/3 20:04
     **/
    public List<SalesGoalEntity> listByConditions(SalesGoalDto salesGoalDto) {
        return this.lambdaQuery()
                .eq(SalesGoalEntity::getBusinessFormatCode, salesGoalDto.getBusinessFormatCode())
                .eq(SalesGoalEntity::getBusinessUnitCode, salesGoalDto.getBusinessUnitCode())
                .eq(SalesGoalEntity::getSalesOrgCode, salesGoalDto.getSalesOrgCode())
                .notIn(!CollectionUtils.isEmpty(salesGoalDto.getExcludeCustomerCodeList()), SalesGoalEntity::getCustomerCode, salesGoalDto.getExcludeCustomerCodeList())
                .notIn(!CollectionUtils.isEmpty(salesGoalDto.getExcludeTerminalCodeList()), SalesGoalEntity::getTerminalCode, salesGoalDto.getExcludeCustomerCodeList())
                .notIn(!CollectionUtils.isEmpty(salesGoalDto.getExcludeProductCodeList()), SalesGoalEntity::getProductCode, salesGoalDto.getExcludeProductCodeList())
                .list();
    }

    public BigDecimal outStockNumber(SalesGoalDto salesGoalDto) {

        List<SalesGoalEntity> list = this.lambdaQuery().eq(SalesGoalEntity::getCustomerCode, salesGoalDto.getCustomerCode())
                .eq(SalesGoalEntity::getProductCode, salesGoalDto.getProductCode())
                .eq(SalesGoalEntity::getYearMonthLy, salesGoalDto.getYearMonthLy()).list();
        if (CollectionUtils.isEmpty(list)) {
            return BigDecimal.ZERO;
        } else {
            return list.stream().map(SalesGoalEntity::getDeliveryQuantity).filter(Objects::nonNull).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
        }
    }

    public BigDecimal inStockNumber(SalesGoalDto salesGoalDto) {
        List<SalesGoalEntity> list = this.lambdaQuery().eq(SalesGoalEntity::getCustomerCode, salesGoalDto.getCustomerCode())
                .eq(SalesGoalEntity::getProductCode, salesGoalDto.getProductCode())
                .eq(SalesGoalEntity::getYearMonthLy, salesGoalDto.getYearMonthLy()).list();
        if (CollectionUtils.isEmpty(list)) {
            return BigDecimal.ZERO;
        } else {
            //todo
            return list.stream().map(SalesGoalEntity::getDeliveryQuantity).filter(Objects::nonNull).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
        }
    }

    public BigDecimal getSaleAmount(SalesGoalDto salesGoalDto) {
        List<SalesGoalEntity> list = this.lambdaQuery().eq(SalesGoalEntity::getCustomerCode, salesGoalDto.getCustomerCode())
                .eq(SalesGoalEntity::getProductCode, salesGoalDto.getProductCode())
                .eq(SalesGoalEntity::getYearMonthLy, salesGoalDto.getYearMonthLy()).list();
        if (CollectionUtils.isEmpty(list)) {
            return BigDecimal.ZERO;
        } else {
            //todo
            return list.stream().map(SalesGoalEntity::getDeliveryQuantity).filter(Objects::nonNull).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
        }
    }

    public BigDecimal getTaskAmount(SalesGoalDto salesGoalDto) {
        List<SalesGoalEntity> list = this.lambdaQuery().eq(SalesGoalEntity::getCustomerCode, salesGoalDto.getCustomerCode())
                .eq(SalesGoalEntity::getProductCode, salesGoalDto.getProductCode())
                .eq(SalesGoalEntity::getYearMonthLy, salesGoalDto.getYearMonthLy()).list();
        if (CollectionUtils.isEmpty(list)) {
            return BigDecimal.ZERO;
        } else {
            return list.stream().map(SalesGoalEntity::getTaskQuantity).filter(Objects::nonNull).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
        }
    }

    /**
     * 根据日历日查询
     *
     * @param calendarDay
     * @return java.util.List<com.biz.crm.tpm.business.sales.goal.local.entity.SalesGoalEntity>
     * @author huojia
     * @date 2023/1/5 16:51
     **/
    public List<SalesGoalEntity> listByCalendar(String calendarDay) {
        if (StringUtils.isEmpty(calendarDay)) {
            return null;
        }
        return this.lambdaQuery()
                .like(SalesGoalEntity::getCalendarDay, calendarDay)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .list();
    }

    /**
     * 条件批量删除
     *
     * @param calender
     * @author huojia
     * @date 2023/1/17 2:31
     **/
    public void delByCalendar(String calender, MasterDataMdgBaseDto dto) {
        salesGoalMapper.delByCalendar(calender, dto);
    }

    /**
     * 导入，大批量数据问题
     *
     * @param salesGoalEntities
     * @author huojia
     * @date 2023/1/28 17:41
     **/
    public void importSaveBatch(List<SalesGoalEntity> salesGoalEntities) {
        salesGoalMapper.importSaveBatch(salesGoalEntities);
    }

    /**
     * 根据维度组装的key查询销售任务数据
     *
     * @param onlyKeys   维度组装的key
     * @param tenantCode 租户编码
     * @return List<SalesGoalEntity>
     **/
    public List<SalesGoalEntity> getSalesGoalByKeys(List<String> onlyKeys, String tenantCode) {
        if (CollectionUtils.isEmpty(onlyKeys) || StringUtils.isBlank(tenantCode)) {
            return Lists.newArrayList();
        }
        List<List<String>> codes = Lists.partition(onlyKeys, 500);
        return this.salesGoalMapper.getSalesGoalByKeys(codes, tenantCode);
    }

    /**
     * 根据业态，业务单元，分组，年月，统计折前销量，按照零售商+区域分组，-隔开
     *
     * @param dto 统计参数
     * @return List<SalesGoalVo>
     */
    public List<SalesGoalVo> getSalesBeforeDiscount(SalesGoalDiscountSalesStatisticsDto dto) {
        if (StringUtils.isBlank(dto.getBusinessFormatCode())) {
            return new ArrayList<>();
        }
        if (StringUtils.isBlank(dto.getBusinessUnitCode())) {
            return new ArrayList<>();
        }
        if (StringUtils.isBlank(dto.getYearSalesTypeCode())) {
            return new ArrayList<>();
        }
        if (Objects.isNull(dto.getYearMonth())) {
            return new ArrayList<>();
        }
        return this.salesGoalMapper.getSalesBeforeDiscount(dto);
    }

    public List<SalesGoalVo> restoreReplaySalesAfterDiscount(SalesGoalDto dto) {
        dto.setTenantCode(TenantUtils.getTenantCode());
        return this.salesGoalMapper.restoreReplaySalesAfterDiscount(dto);
    }

    /**
     * 根据业态，业务单元，分组，年月，统计折后销量，按照零售商+区域分组，-隔开
     *
     * @param dto 统计参数
     * @return List<SalesGoalVo>
     */
    public List<SalesGoalVo> getSalesAfterDiscount(SalesGoalDiscountSalesStatisticsDto dto) {
        if (StringUtils.isBlank(dto.getBusinessFormatCode())) {
            return new ArrayList<>();
        }
        if (StringUtils.isBlank(dto.getBusinessUnitCode())) {
            return new ArrayList<>();
        }
        if (StringUtils.isBlank(dto.getYearSalesTypeCode())) {
            return new ArrayList<>();
        }
        if (Objects.isNull(dto.getYearMonth())) {
            return new ArrayList<>();
        }
        return this.salesGoalMapper.getSalesAfterDiscount(dto);
    }

    /**
     * 根据参数统计折后额和出库数
     *
     * @param salesGoalDto 参数
     * @return SalesGoalSumVo
     */
    public SalesGoalSumVo findDiscountAfterAndDeliveryQuantity(SalesGoalDto salesGoalDto) {
        return this.salesGoalMapper.findDiscountAfterAndDeliveryQuantity(salesGoalDto);
    }

    public BigDecimal findDiscountAfter(SalesGoalDto salesGoalDto) {
        return this.salesGoalMapper.findDiscountAfter(salesGoalDto);
    }

    /**
     * 销售任务：统计“年月+区域+系统”维度内控版的销售任务折后金额
     * @param salesGoalDtos
     * @return
     */
    public List<SalesGoalVo> findListForFR(List<SalesGoalDto> salesGoalDtos) {
        return this.salesGoalMapper.findListForFR(salesGoalDtos);
    }
}