package com.biz.crm.tpm.business.sales.goal.local.service.internal;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.eunm.DataFromEnum;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.constants.MasterDataMdgConstants;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.dto.MasterDataMdgBaseDto;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.service.MasterDataMdgService;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.vo.MasterDataMdgSalesDayReportVo;
import com.biz.crm.tpm.business.sales.goal.local.entity.SalesGoalEntity;
import com.biz.crm.tpm.business.sales.goal.local.mapper.SalesGoalMapper;
import com.biz.crm.tpm.business.sales.goal.local.repository.SalesGoalRepository;
import com.biz.crm.tpm.business.sales.goal.local.service.SalesGoaEditService;
import com.biz.crm.tpm.business.sales.goal.sdk.constant.SalesGoalConstant;
import com.biz.crm.tpm.business.sales.goal.sdk.dto.*;
import com.biz.crm.tpm.business.sales.goal.sdk.eunm.YearSalesTypeEnum;
import com.biz.crm.tpm.business.sales.goal.sdk.event.SalesGoalForMainOneDaySalesListener;
import com.biz.crm.tpm.business.sales.goal.sdk.event.SalesGoalLogEventListener;
import com.biz.crm.tpm.business.sales.goal.sdk.service.SalesGoalService;
import com.biz.crm.tpm.business.sales.goal.sdk.vo.SalesGoalStatisticsVo;
import com.biz.crm.tpm.business.sales.goal.sdk.vo.SalesGoalSumVo;
import com.biz.crm.tpm.business.sales.goal.sdk.vo.SalesGoalVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.bizunited.nebula.security.sdk.login.UserIdentity;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author huojia
 * @date 2022年10月26日 11:10
 */
@Slf4j
@Service
public class SalesGoalServiceImpl implements SalesGoalService {

    @Autowired(required = false)
    private SalesGoalMapper salesGoalMapper;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    @Autowired(required = false)
    private SalesGoalRepository salesGoalRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private MasterDataMdgService masterDataMdgService;

    @Autowired(required = false)
    private List<SalesGoalForMainOneDaySalesListener> salesGoalForMainOneDaySalesListeners;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Autowired(required = false)
    private SalesGoaEditService salesGoaEditService;

    @Autowired(required = false)
    private ProductVoService productVoService;

    @Autowired(required = false)
    private SalesOrgVoService salesOrgVoService;

    /**
     * 分页查询销售目标
     *
     * @param pageable
     * @param dto
     * @return com.baomidou.mybatisplus.extension.plugins.pagination.Page<com.biz.crm.tpm.business.sales.goal.sdk.vo.SalesGoalVo>
     * @author huojia
     * @date 2022/11/1 21:10
     **/
    @Override
    public Page<SalesGoalVo> findByConditions(Pageable pageable, SalesGoalDto dto) {
        ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(dto)) {
            dto = new SalesGoalDto();
        }
        Page<SalesGoalVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.salesGoalMapper.findByConditions(page, dto);
    }

    /**
     * 根据主键查询销售目标
     *
     * @param id
     * @return com.biz.crm.tpm.business.sales.goal.sdk.vo.SalesGoalVo
     * @author huojia
     * @date 2022/11/1 21:10
     **/
    @Override
    public SalesGoalVo findById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        SalesGoalEntity salesGoalEntity = this.salesGoalRepository.getById(id, DelFlagStatusEnum.NORMAL.getCode());
        Validate.notNull(salesGoalEntity, "数据不存在，请刷新后重试！");
        return nebulaToolkitService.copyObjectByBlankList(salesGoalEntity, SalesGoalVo.class, LinkedHashSet.class, ArrayList.class);
    }

    /**
     * 编辑销售目标
     *
     * @param dto
     * @return com.biz.crm.tpm.business.sales.goal.sdk.vo.SalesGoalVo
     * @author huojia
     * @date 2022/11/1 21:11
     **/
    @Override
    @Transactional(rollbackFor = Exception.class)
    public SalesGoalVo update(SalesGoalDto dto) {
        // 根据配置校验必填字段
        this.updateValidate(dto);
        SalesGoalVo salesGoalVo = this.findById(dto.getId());
        if (ObjectUtils.isEmpty(salesGoalVo)) {
            throw new RuntimeException("修改数据失败，原数据不存在！");
        }
        // 保存销售目标
        SalesGoalDto oldVo = this.nebulaToolkitService.copyObjectByWhiteList(salesGoalVo, SalesGoalDto.class, LinkedHashSet.class, ArrayList.class);
        SalesGoalEntity salesGoalEntity = this.nebulaToolkitService.copyObjectByWhiteList(dto, SalesGoalEntity.class, LinkedHashSet.class, ArrayList.class);
        this.salesGoalRepository.saveOrUpdate(salesGoalEntity);
        // 业务日志编辑
        SalesGoalLogEventDto logEventDto = new SalesGoalLogEventDto();
        logEventDto.setOriginal(oldVo);
        logEventDto.setNewest(dto);
        SerializableBiConsumer<SalesGoalLogEventListener, SalesGoalLogEventDto> onUpdate =
                SalesGoalLogEventListener::onUpdate;
        this.nebulaNetEventClient.publish(logEventDto, SalesGoalLogEventListener.class, onUpdate);
        return this.nebulaToolkitService.copyObjectByWhiteList(dto, SalesGoalVo.class, LinkedHashSet.class, ArrayList.class);
    }

    /**
     * 校验参数
     *
     * @param dto
     * @author huojia
     * @date 2022/10/26 11:48
     **/
    private void updateValidate(SalesGoalDto dto) {
        if (StringUtils.isEmpty(dto.getTenantCode())) {
            dto.setTenantCode(TenantUtils.getTenantCode());
        }
        /*DimensionInformationQueryData dimensionVo = new DimensionInformationQueryData();
        dimensionVo.setBusinessState(dto.getBusinessFormatCode());
        dimensionVo.setBusinessUnit(dto.getBusinessUnitCode());
        dimensionVo.setMenu(MenuCodeEnum.SALES_GOAL.getCode());
        dimensionVo.setSalesOrgCodeList(Arrays.asList(dto.getSalesOrgCode()));
        List<DimensionDimensionInformationVo> dimensionVoList = dimensionDimensionInformationService.findDetailsByCodes(dimensionVo);
        if (CollectionUtils.isEmpty(dimensionVoList)) {
            throw new RuntimeException("未维护对应维度配置数据，请检查！");
        }
        Class<SalesGoalDto> clazz = SalesGoalDto.class;
        // 根据维度配置，判断必填字段
        dimensionVoList.forEach(informationVo -> {
            try {
                if (StringUtils.isEmpty(informationVo.getFieldCoding())) {
                    return;
                }
                Field field = clazz.getDeclaredField(informationVo.getFieldCoding());
                field.setAccessible(true);
                Object o = field.get(dto);
                if (ObjectUtils.isEmpty(o)) {
                    throw new RuntimeException(informationVo.getFieldName() + "不能为空！");
                }
            } catch (NoSuchFieldException | IllegalAccessException e) {
                throw new RuntimeException(e.getMessage());
            }
        });*/
    }

    /**
     * 批量删除
     *
     * @param ids
     * @author huojia
     * @date 2022/11/1 21:11
     **/
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delete(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            throw new RuntimeException("请选择要删除的数据！");
        }
        List<SalesGoalEntity> salesGoalEntities = this.salesGoalRepository.listByIds(ids);
        if (!CollectionUtils.isEmpty(salesGoalEntities)) {
            salesGoalEntities.forEach(salesGoalEntity -> {
                Validate.isTrue(!DataFromEnum.FOREIGN_SYSTEM.getCode().equals(salesGoalEntity.getDataFromCode()), "通过系统接口同步过来的数据不支持删除！");
                salesGoalEntity.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
            });
            this.salesGoalRepository.updateBatchById(salesGoalEntities);
        }
        // 业务日志删除
        salesGoalEntities.forEach(salesGoalEntity -> {
            SalesGoalLogEventDto logEventDto = new SalesGoalLogEventDto();
            logEventDto.setOriginal(this.nebulaToolkitService.copyObjectByWhiteList(
                    salesGoalEntity, SalesGoalDto.class, LinkedHashSet.class, ArrayList.class
            ));
            logEventDto.setNewest(null);
            SerializableBiConsumer<SalesGoalLogEventListener, SalesGoalLogEventDto> onDelete =
                    SalesGoalLogEventListener::onDelete;
            this.nebulaNetEventClient.publish(logEventDto, SalesGoalLogEventListener.class, onDelete);
        });
    }

    /**
     * 批量导入
     *
     * @param importList
     * @author huojia
     * @date 2022/11/1 21:11
     **/
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void importSave(List<SalesGoalDto> importList) {
        if (CollectionUtils.isEmpty(importList)) {
            return;
        }
        importList.forEach(dto -> {
            dto.setId(UUID.randomUUID().toString().replace("-", ""));
            dto.setTenantCode(TenantUtils.getTenantCode());
            dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            dto.setDataFromCode(DataFromEnum.IMPORT.getCode());
        });
        List<SalesGoalEntity> salesGoalEntities = (List<SalesGoalEntity>) this.nebulaToolkitService.copyCollectionByWhiteList(
                importList, SalesGoalDto.class, SalesGoalEntity.class, LinkedHashSet.class, ArrayList.class
        );
        this.salesGoalRepository.importSaveBatch(salesGoalEntities);
        // 业务日志创建
        importList.forEach(dto -> {
            SalesGoalLogEventDto logEventDto = new SalesGoalLogEventDto();
            logEventDto.setOriginal(null);
            logEventDto.setNewest(dto);
            SerializableBiConsumer<SalesGoalLogEventListener, SalesGoalLogEventDto> onCreate =
                    SalesGoalLogEventListener::onCreate;
            this.nebulaNetEventClient.publish(logEventDto, SalesGoalLogEventListener.class, onCreate);
        });
    }

    @Override
    public List<SalesGoalVo> listByConditions(SalesGoalDto salesGoalDto) {
        List<SalesGoalEntity> salesGoalEntities = salesGoalMapper.listByConditions(salesGoalDto);
        // List<SalesGoalEntity> salesGoalEntities = salesGoalRepository.listByConditions(salesGoalDto);
        if (CollectionUtils.isEmpty(salesGoalEntities)) {
            return Lists.newArrayList();
        }
        return new ArrayList<>(this.nebulaToolkitService.copyCollectionByWhiteList(salesGoalEntities, SalesGoalEntity.class, SalesGoalVo.class, LinkedHashSet.class, ArrayList.class));
    }

    @Override
    public BigDecimal getSalesGoalAmount(SalesGoalDto salesGoal) {
        return this.salesGoalMapper.getSalesGoalAmount(salesGoal);
    }

    @Override
    public BigDecimal outStockNumber(SalesGoalDto salesGoalDto) {

        return this.salesGoalRepository.outStockNumber(salesGoalDto);
    }

    @Override
    public BigDecimal inStockNumber(SalesGoalDto salesGoalDto) {

        return this.salesGoalRepository.inStockNumber(salesGoalDto);
    }

    @Override
    public BigDecimal getSaleAmount(SalesGoalDto salesGoalDto) {
        return this.salesGoalRepository.getSaleAmount(salesGoalDto);
    }

    @Override
    public BigDecimal getTaskAmount(SalesGoalDto salesGoalDto) {
        return this.salesGoalRepository.getTaskAmount(salesGoalDto);
    }

    /**
     * 分页统计销售目标任务量
     *
     * @param pageable 分页参数
     * @param dto      筛选参数
     * @return Page<SalesGoalStatisticsVo>
     **/
    @Override
    public Page<SalesGoalStatisticsVo> findByConditions(Pageable pageable, SalesGoalStatisticsDto dto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(dto)) {
            dto = new SalesGoalStatisticsDto();
        }
        Page<SalesGoalStatisticsVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.salesGoalMapper.listTaskQuantityPage(page, dto);
    }

    /**
     * 查询按区域分组的数据
     */
    @Override
    public List<SalesGoalVo> findListGroupByRegion(SalesGoalDto dto) {
        return this.salesGoalMapper.findListGroupByRegion(dto);
    }

    /**
     * 批量拉取
     *
     * @param dto
     * @author huojia
     * @date 2023/1/2 16:38
     **/
    @Override
    @Async
    public void pullSalesDayReportAsync(MasterDataMdgBaseDto dto, UserIdentity userIdentity) {
        loginUserService.refreshAuthentication(userIdentity);
        this.pullSalesDayReport(dto);
    }

    /**
     * 批量拉取
     *
     * @param dto
     * @author huojia
     * @date 2023/1/2 16:38
     **/
    @Override
    public void pullSalesDayReport(MasterDataMdgBaseDto dto) {
        Validate.notBlank(dto.getSales_organization(), "销售机构erp编码不能为空");
        Validate.notBlank(dto.getY_id(), "年[y_id]不能为空");
        Validate.notBlank(dto.getM_id(), "月[m_id]不能为空");
        Validate.notBlank(dto.getSales_organization(), "销售机构erp编码不能为空");
        if (ObjectUtils.isEmpty(dto)) {
            dto = new MasterDataMdgBaseDto();
        }
        if (StringUtils.isEmpty(dto.getPageNum())) {
            dto.setPageNum("1");
        }
        if (StringUtils.isEmpty(dto.getPageSize())) {
            dto.setPageSize("2000");
        }
        String lockKey = this.getLockKey(dto);
        boolean lock = true;
        try {
            lock = this.lock(lockKey);
            if (!lock) {
                return;
            }
            int maxPage = 0;
            if (BooleanEnum.TRUE.getCapital().equals(dto.getFullPullFlag())) {
                maxPage = this.countPageMax(dto);
            } else {
                // 一次最多200万数据
                maxPage = 1000;
            }

            List<MasterDataMdgSalesDayReportVo> masterDataMdgSalesDayReportVos = new ArrayList<>();
            if (StringUtils.isBlank(dto.getD_id())) {
                //先删除销售任务数据
                LambdaQueryWrapper<SalesGoalEntity> wrapper = new LambdaQueryWrapper<SalesGoalEntity>()
                        .eq(SalesGoalEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                        .eq(SalesGoalEntity::getBusinessUnitCode, BusinessUnitEnum.HEADQUARTERS.getCode())
                        .eq(SalesGoalEntity::getSalesInstitutionErpCode, dto.getSales_organization())
                        .eq(SalesGoalEntity::getYearMonthLy, dto.getY_id() + "-" + dto.getM_id());
                this.salesGoalRepository.remove(wrapper);
            }
            for (int pageNum = Integer.parseInt(dto.getPageNum()); pageNum <= maxPage; pageNum++) {
                dto.setPageNum(String.valueOf(pageNum));
                masterDataMdgSalesDayReportVos = masterDataMdgService.pullSalesDayReportList(dto);
                if (CollectionUtils.isEmpty(masterDataMdgSalesDayReportVos)) {
                    return;
                }
                // 数据校验
                //销售任务数据
                List<SalesGoalEntity> pullList = new ArrayList<>();
                //主体日销售数据
                List<MainOneDaySalesDto> dayList = new ArrayList<>();
                //拆分数据
                this.salesGoalValidate(masterDataMdgSalesDayReportVos, pullList, dayList);

                //更新保存销售任务数据
                if (!CollectionUtils.isEmpty(pullList)) {
                    this.salesGoaEditService.editBatch(pullList);
                }

                if (!CollectionUtils.isEmpty(dayList)) {
                    //推送到主体日销售数据中
                    if (!CollectionUtils.isEmpty(salesGoalForMainOneDaySalesListeners)) {
                        for (SalesGoalForMainOneDaySalesListener salesGoalForMainOneDaySalesListener : salesGoalForMainOneDaySalesListeners) {
                            salesGoalForMainOneDaySalesListener.saveOrUpdate(dayList);
                        }
                    }
                }
            }
        } finally {
            if (lock) {
                this.unLock(lockKey);
            }
        }
    }

    /**
     * 垂直批量查询
     *
     * @param salesGoalDto
     * @return java.util.List<com.biz.crm.tpm.business.sales.goal.sdk.vo.SalesGoalVo>
     * @author huojia
     * @date 2023/1/17 0:51
     **/
    @Override
    public List<SalesGoalVo> listVerticalByConditions(SalesGoalDto salesGoalDto) {
        List<SalesGoalEntity> salesGoalEntities = salesGoalMapper.listVerticalByConditions(salesGoalDto);
        if (CollectionUtils.isEmpty(salesGoalEntities)) {
            return Lists.newArrayList();
        }
        return new ArrayList<>(this.nebulaToolkitService.copyCollectionByWhiteList(
                salesGoalEntities, SalesGoalEntity.class, SalesGoalVo.class, LinkedHashSet.class, ArrayList.class
        ));
    }

    /**
     * 汇总垂直销售任务
     *
     * @param salesGoalDto
     * @return java.util.List<com.biz.crm.tpm.business.sales.goal.sdk.vo.SalesGoalSumVo>
     * @author huojia
     * @date 2023/2/9 10:59
     **/
    @Override
    public List<SalesGoalSumVo> listVerticalSumByConditions(SalesGoalDto salesGoalDto) {
        return salesGoalMapper.listVerticalSumByConditions(salesGoalDto);
    }

    @Override
    public List<SalesGoalVo> findSalesGoal(SalesGoalDto salesGoal) {

        List<SalesGoalEntity> salesGoalList = this.salesGoalMapper.selectList(Wrappers.lambdaQuery(SalesGoalEntity.class)
                .eq(SalesGoalEntity::getCustomerCode, salesGoal.getCustomerCode())
                .eq(SalesGoalEntity::getBusinessFormatCode, salesGoal.getBusinessFormatCode())
                .eq(SalesGoalEntity::getBusinessUnitCode, salesGoal.getBusinessUnitCode())
                .eq(SalesGoalEntity::getYearMonthLy, salesGoal.getYearMonthLy()));

        if (salesGoalList == null || salesGoalList.isEmpty()) {

            return Lists.newArrayList();

        }

        return (List<SalesGoalVo>) this.nebulaToolkitService.copyCollectionByWhiteList(salesGoalList
                , SalesGoalEntity.class
                , SalesGoalVo.class
                , HashSet.class
                , ArrayList.class);

    }

    /**
     * 指标计算查询
     *
     * @param dto
     * @return
     */
    @Override
    public List<SalesGoalVo> listForVariableCal(SalesGoalDto dto) {
        if (StringUtils.isEmpty(dto.getSalesOrgCode())
                && StringUtils.isEmpty(dto.getCustomerCode())
                && StringUtils.isEmpty(dto.getYearMonthLy())
                && StringUtils.isEmpty(dto.getYearLy())
                && StringUtils.isEmpty(dto.getCustomerChannelCode())
                && StringUtils.isEmpty(dto.getChannelCode())
                && StringUtils.isEmpty(dto.getBusinessFormatCode())
                && StringUtils.isEmpty(dto.getBusinessUnitCode())
                && StringUtils.isEmpty(dto.getSalesInstitutionCode())
                && StringUtils.isEmpty(dto.getSalesOrgRegionCode())
                && StringUtils.isEmpty(dto.getSalesOrgProvinceCode())
                && StringUtils.isEmpty(dto.getYearSalesTypeCode())
                && CollectionUtils.isEmpty(dto.getExcludeProductCodeList())
                && CollectionUtils.isEmpty(dto.getExcludeProductBrandCodeList())
                && CollectionUtils.isEmpty(dto.getExcludeProductCategoryCodeList())
                && CollectionUtils.isEmpty(dto.getExcludeProductItemCodeList())
                && CollectionUtils.isEmpty(dto.getIncludeProductCodeList())
                && CollectionUtils.isEmpty(dto.getYearMonthLySet())
                && CollectionUtils.isEmpty(dto.getCustomerCodeList())
                && StringUtils.isEmpty(dto.getCustomerErpCode())) {
            //防止全表扫，下面增了过滤条件，一定要在这里加上
            return Lists.newArrayList();
        }
        List<SalesGoalEntity> entityList = this.salesGoalRepository.lambdaQuery()
                .eq(SalesGoalEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(StringUtils.isNotEmpty(dto.getSalesOrgCode()), SalesGoalEntity::getSalesOrgCode, dto.getSalesOrgCode())
                .eq(StringUtils.isNotEmpty(dto.getCustomerErpCode()), SalesGoalEntity::getCustomerErpCode, dto.getCustomerErpCode())
                .eq(StringUtils.isNotEmpty(dto.getCustomerCode()), SalesGoalEntity::getCustomerCode, dto.getCustomerCode())
                .in(!CollectionUtils.isEmpty(dto.getCustomerCodeList()), SalesGoalEntity::getCustomerCode, dto.getCustomerCodeList())
                .eq(StringUtils.isNotEmpty(dto.getYearMonthLy()), SalesGoalEntity::getYearMonthLy, dto.getYearMonthLy())
                .eq(StringUtils.isNotEmpty(dto.getYearLy()), SalesGoalEntity::getYearMonthLy, dto.getYearLy())
                .in(!CollectionUtils.isEmpty(dto.getYearMonthLySet()), SalesGoalEntity::getYearMonthLy, dto.getYearMonthLySet())
                .eq(StringUtils.isNotEmpty(dto.getCustomerChannelCode()), SalesGoalEntity::getCustomerChannelCode, dto.getCustomerChannelCode())
                .eq(StringUtils.isNotEmpty(dto.getChannelCode()), SalesGoalEntity::getChannelCode, dto.getChannelCode())
                .eq(StringUtils.isNotEmpty(dto.getBusinessFormatCode()), SalesGoalEntity::getBusinessFormatCode, dto.getBusinessFormatCode())
                .eq(StringUtils.isNotEmpty(dto.getBusinessUnitCode()), SalesGoalEntity::getBusinessUnitCode, dto.getBusinessUnitCode())
                .eq(StringUtils.isNotEmpty(dto.getSalesInstitutionCode()), SalesGoalEntity::getSalesInstitutionCode, dto.getSalesInstitutionCode())
                .eq(StringUtils.isNotEmpty(dto.getSalesOrgRegionCode()), SalesGoalEntity::getSalesOrgRegionCode, dto.getSalesOrgRegionCode())
                .eq(StringUtils.isNotEmpty(dto.getSalesOrgProvinceCode()), SalesGoalEntity::getSalesOrgProvinceCode, dto.getSalesOrgProvinceCode())
                .eq(StringUtils.isNotEmpty(dto.getYearSalesTypeCode()), SalesGoalEntity::getYearSalesTypeCode, dto.getYearSalesTypeCode())
                .in(!CollectionUtils.isEmpty(dto.getIncludeProductCodeList()), SalesGoalEntity::getProductCode, dto.getIncludeProductCodeList())
                .notIn(!CollectionUtils.isEmpty(dto.getExcludeProductCodeList()), SalesGoalEntity::getProductCode, dto.getExcludeProductCodeList())
                .notIn(!CollectionUtils.isEmpty(dto.getExcludeProductBrandCodeList()), SalesGoalEntity::getProductBrandCode, dto.getExcludeProductBrandCodeList())
                .notIn(!CollectionUtils.isEmpty(dto.getExcludeProductCategoryCodeList()), SalesGoalEntity::getProductCategoryCode, dto.getExcludeProductCategoryCodeList())
                .notIn(!CollectionUtils.isEmpty(dto.getExcludeProductItemCodeList()), SalesGoalEntity::getProductItemCode, dto.getExcludeProductItemCodeList())
                .list();
        return (List<SalesGoalVo>) nebulaToolkitService.copyCollectionByWhiteList(entityList, SalesGoalEntity.class, SalesGoalVo.class, HashSet.class, ArrayList.class);
    }

    @Override
    public List<SalesGoalVo> listForVariableCalNew(SalesGoalDto dto) {
        if (StringUtils.isEmpty(dto.getSalesOrgCode())
                && StringUtils.isEmpty(dto.getCustomerCode())
                && StringUtils.isEmpty(dto.getYearMonthLy())
                && StringUtils.isEmpty(dto.getCustomerChannelCode())
                && StringUtils.isEmpty(dto.getChannelCode())
                && StringUtils.isEmpty(dto.getBusinessFormatCode())
                && StringUtils.isEmpty(dto.getBusinessUnitCode())
                && StringUtils.isEmpty(dto.getSalesInstitutionCode())
                && StringUtils.isEmpty(dto.getSalesOrgRegionCode())
                && StringUtils.isEmpty(dto.getSalesOrgProvinceCode())
                && StringUtils.isEmpty(dto.getYearSalesTypeCode())
                && CollectionUtils.isEmpty(dto.getExcludeProductCodeList())
                && CollectionUtils.isEmpty(dto.getIncludeProductCodeList())
                && StringUtils.isEmpty(dto.getCustomerErpCode())) {
            //防止全表扫，下面增了过滤条件，一定要在这里加上
            return Lists.newArrayList();
        }
        String year = dto.getYearMonthLy().substring(0, 4);
        String month = dto.getYearMonthLy().substring(5, dto.getYearMonthLy().length());
        if (!month.equals("01")) {
            Integer m = Integer.parseInt(month) - 1;
            if (m < 10) {
                month = "0" + m;
            } else {
                month = m.toString();
            }
        }
        String startDateStr = year + "01";
        String endDateStr = year + month;
        List<SalesGoalEntity> entityList = this.salesGoalRepository.lambdaQuery()
                .eq(SalesGoalEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(StringUtils.isNotEmpty(dto.getSalesOrgCode()), SalesGoalEntity::getSalesOrgCode, dto.getSalesOrgCode())
                .eq(StringUtils.isNotEmpty(dto.getCustomerErpCode()), SalesGoalEntity::getCustomerErpCode, dto.getCustomerErpCode())
                .eq(StringUtils.isNotEmpty(dto.getCustomerCode()), SalesGoalEntity::getCustomerCode, dto.getCustomerCode())
                .eq(StringUtils.isNotEmpty(dto.getCustomerChannelCode()), SalesGoalEntity::getCustomerChannelCode, dto.getCustomerChannelCode())
                .eq(StringUtils.isNotEmpty(dto.getChannelCode()), SalesGoalEntity::getChannelCode, dto.getChannelCode())
                .eq(StringUtils.isNotEmpty(dto.getBusinessFormatCode()), SalesGoalEntity::getBusinessFormatCode, dto.getBusinessFormatCode())
                .eq(StringUtils.isNotEmpty(dto.getBusinessUnitCode()), SalesGoalEntity::getBusinessUnitCode, dto.getBusinessUnitCode())
                .eq(StringUtils.isNotEmpty(dto.getSalesInstitutionCode()), SalesGoalEntity::getSalesInstitutionCode, dto.getSalesInstitutionCode())
                .eq(StringUtils.isNotEmpty(dto.getSalesOrgRegionCode()), SalesGoalEntity::getSalesOrgRegionCode, dto.getSalesOrgRegionCode())
                .eq(StringUtils.isNotEmpty(dto.getSalesOrgProvinceCode()), SalesGoalEntity::getSalesOrgProvinceCode, dto.getSalesOrgProvinceCode())
                .eq(StringUtils.isNotEmpty(dto.getYearSalesTypeCode()), SalesGoalEntity::getYearSalesTypeCode, dto.getYearSalesTypeCode())
                .between(StringUtils.isNotEmpty(startDateStr) && StringUtils.isNotEmpty(endDateStr), SalesGoalEntity::getYearMonthLy, startDateStr, endDateStr)
                .in(!CollectionUtils.isEmpty(dto.getIncludeProductCodeList()), SalesGoalEntity::getProductCode, dto.getIncludeProductCodeList())
                .notIn(!CollectionUtils.isEmpty(dto.getExcludeProductCodeList()), SalesGoalEntity::getProductCode, dto.getExcludeProductCodeList())
                .list();
        return (List<SalesGoalVo>) nebulaToolkitService.copyCollectionByWhiteList(entityList, SalesGoalEntity.class, SalesGoalVo.class, HashSet.class, ArrayList.class);
    }

    /**
     * 授权预算项目折后销售
     *
     * @param dto
     * @return
     */
    @Override
    public List<SalesGoalVo> restoreReplaySalesAfterDiscount(SalesGoalDto dto) {
        return salesGoalRepository.restoreReplaySalesAfterDiscount(dto);
    }

    /**
     * 根据业态，业务单元，分组，年月，统计折前销量，按照零售商+区域分组，-隔开
     *
     * @param dto
     * @return Map<String, BigDecimal>
     */
    @Override
    public Map<String, BigDecimal> getSalesBeforeDiscount(SalesGoalDiscountSalesStatisticsDto dto) {
        if (StringUtils.isBlank(dto.getBusinessFormatCode())) {
            return new HashMap<>();
        }
        if (StringUtils.isBlank(dto.getBusinessUnitCode())) {
            return new HashMap<>();
        }
        if (StringUtils.isBlank(dto.getYearSalesTypeCode())) {
            return new HashMap<>();
        }
        if (Objects.isNull(dto.getYearMonth())) {
            return new HashMap<>();
        }
        Map<String, BigDecimal> map = new HashMap<>();
        List<SalesGoalVo> feeVos = salesGoalRepository.getSalesBeforeDiscount(dto);
        if (CollectionUtils.isEmpty(feeVos)) {
            return map;
        }
        for (SalesGoalVo feeVo : feeVos) {
            String key = feeVo.getSystemCode() + "-" + feeVo.getRegionCode();
            if (map.containsKey(key)) {
                BigDecimal b = map.get(key).add(feeVo.getDeliverySalesAmount());
                map.put(key, b);
            } else {
                map.put(key, feeVo.getDeliverySalesAmount());
            }
        }
        return map;
    }

    /**
     * 根据业态，业务单元，分组，年月，统计折后销量，按照零售商+区域分组，-隔开
     *
     * @param dto
     * @return Map<String, BigDecimal>
     */
    @Override
    public Map<String, BigDecimal> getSalesAfterDiscount(SalesGoalDiscountSalesStatisticsDto dto) {
        if (StringUtils.isBlank(dto.getBusinessFormatCode())) {
            return new HashMap<>();
        }
        if (StringUtils.isBlank(dto.getBusinessUnitCode())) {
            return new HashMap<>();
        }
        if (StringUtils.isBlank(dto.getYearSalesTypeCode())) {
            return new HashMap<>();
        }
        if (Objects.isNull(dto.getYearMonth())) {
            return new HashMap<>();
        }
        Map<String, BigDecimal> map = new HashMap<>();
        List<SalesGoalVo> feeVos = salesGoalRepository.getSalesAfterDiscount(dto);
        if (CollectionUtils.isEmpty(feeVos)) {
            return map;
        }
        for (SalesGoalVo feeVo : feeVos) {
            String key = feeVo.getSystemCode() + "-" + feeVo.getRegionCode();
            if (map.containsKey(key)) {
                BigDecimal b = map.get(key).add(feeVo.getDeliveryDiscountSalesAmount());
                map.put(key, b);
            } else {
                map.put(key, feeVo.getDeliveryDiscountSalesAmount());
            }
        }
        return map;
    }

    /**
     * 计数
     *
     * @param dto
     * @return int
     * @author huojia
     * @date 2023/1/12 1:16
     **/
    private int countPageMax(MasterDataMdgBaseDto dto) {
        Integer totalNum = masterDataMdgService.countList(MasterDataMdgConstants.API_ADS_MN_SBI_CHL_SALES_DAILY_NEW_D_TPM);
        int maxPage = 1;
        int sizeInt = Integer.parseInt(dto.getPageSize());
        if (totalNum > sizeInt) {
            maxPage = totalNum % sizeInt > 0 ? (totalNum / sizeInt) + 1 : totalNum / sizeInt;
        }
        return maxPage;
    }

    private List<MainOneDaySalesDto> buildMainOneDaySalesDto(List<MasterDataMdgSalesDayReportVo> masterDataMdgSalesDayReportVos) {

        List<MainOneDaySalesDto> mainOneDaySalesDtoList = new ArrayList<>();
        for (MasterDataMdgSalesDayReportVo masterDataMdgSalesDayReportVo : masterDataMdgSalesDayReportVos) {
            MainOneDaySalesDto mainOneDaySalesDto = new MainOneDaySalesDto();
            mainOneDaySalesDto.setYearMonthDay(masterDataMdgSalesDayReportVo.getCalendar_day());
            mainOneDaySalesDto.setBusinessFormatCode(masterDataMdgSalesDayReportVo.getFormats());
            mainOneDaySalesDto.setBusinessFormatName(null);

            mainOneDaySalesDto.setSalesInstitutionErpCode(masterDataMdgSalesDayReportVo.getSales_organization());
            mainOneDaySalesDto.setSalesInstitutionName(masterDataMdgSalesDayReportVo.getSales_organization_txt());
            mainOneDaySalesDto.setSalesInstitutionCode(masterDataMdgSalesDayReportVo.getSales_organization_tpm());


            mainOneDaySalesDto.setSalesOrgRegionErpCode(masterDataMdgSalesDayReportVo.getManagement_area());
            mainOneDaySalesDto.setSalesOrgRegionName(masterDataMdgSalesDayReportVo.getManagement_area_txt());
            mainOneDaySalesDto.setSalesOrgRegionCode(masterDataMdgSalesDayReportVo.getManagement_area_tpm());

            mainOneDaySalesDto.setSalesOrgProvinceErpCode(masterDataMdgSalesDayReportVo.getProvincial_area());
            mainOneDaySalesDto.setSalesOrgProvinceName(masterDataMdgSalesDayReportVo.getZsalegrp_t());
            mainOneDaySalesDto.setSalesOrgProvinceCode(masterDataMdgSalesDayReportVo.getProvincial_area_tpm());

            mainOneDaySalesDto.setCustomerErpCode(masterDataMdgSalesDayReportVo.getCustomer());
            mainOneDaySalesDto.setCustomerName(masterDataMdgSalesDayReportVo.getCustomer_txt());
            mainOneDaySalesDto.setCustomerCode(masterDataMdgSalesDayReportVo.getCustomer() + masterDataMdgSalesDayReportVo.getSales_organization()
                    + masterDataMdgSalesDayReportVo.getChannel_type() + masterDataMdgSalesDayReportVo.getFormats());
            mainOneDaySalesDto.setChannelCode(masterDataMdgSalesDayReportVo.getChannel_type());
            mainOneDaySalesDto.setChannelName(masterDataMdgSalesDayReportVo.getChannel_type_txt());
            mainOneDaySalesDto.setCustomerClassifyCode(masterDataMdgSalesDayReportVo.getCustomer_category());
            mainOneDaySalesDto.setCustomerClassifyName(masterDataMdgSalesDayReportVo.getCustomer_category_txt());
            mainOneDaySalesDto.setProductCode(masterDataMdgSalesDayReportVo.getProduct());
            mainOneDaySalesDto.setProductName(masterDataMdgSalesDayReportVo.getProduct_txt());
            mainOneDaySalesDto.setNewAndOldProduct(masterDataMdgSalesDayReportVo.getZxlp());
            //金额字段赋值
            mainOneDaySalesDto.setSaleQuantity(new BigDecimal(Optional.ofNullable(masterDataMdgSalesDayReportVo.getActual_qty_3_1_1d()).orElse("0")));
            mainOneDaySalesDto.setDiscountFrontSaleAmount(new BigDecimal(Optional.ofNullable(masterDataMdgSalesDayReportVo.getSales_amt_3_1_1d()).orElse("0")));
            mainOneDaySalesDto.setDiscountBehindSaleAmount(new BigDecimal(Optional.ofNullable(masterDataMdgSalesDayReportVo.getDiscount_sales_amt_3_1_1d()).orElse("0")));
            mainOneDaySalesDto.setMilkCardDiscountBehindOrderAmountToB(new BigDecimal(Optional.ofNullable(masterDataMdgSalesDayReportVo.getDiscount_mc_order_sales_amt_3_1_1d()).orElse("0")));
            mainOneDaySalesDto.setMilkCardDiscountBehindOrderAmountToC(new BigDecimal(Optional.ofNullable(masterDataMdgSalesDayReportVo.getToc_mc_area_order_sales_amt_1d()).orElse("0")));

            mainOneDaySalesDto.setProductBrandCode(masterDataMdgSalesDayReportVo.getProductBrandCode());
            mainOneDaySalesDto.setProductBrandName(masterDataMdgSalesDayReportVo.getProductBrandName());
            mainOneDaySalesDto.setProductCategoryCode(masterDataMdgSalesDayReportVo.getProductCategoryCode());
            mainOneDaySalesDto.setProductCategoryName(masterDataMdgSalesDayReportVo.getProductCategoryName());
            mainOneDaySalesDto.setProductItemCode(masterDataMdgSalesDayReportVo.getProductItemCode());
            mainOneDaySalesDto.setProductItemName(masterDataMdgSalesDayReportVo.getProductItemName());
            mainOneDaySalesDtoList.add(mainOneDaySalesDto);
        }
        return mainOneDaySalesDtoList;
    }

    /**
     * 批量保存
     *
     * @param dto
     * @param pullList
     * @author huojia
     * @date 2023/1/5 14:42
     **/
    @Transactional(rollbackFor = Exception.class)
    void saveMdgBatch(MasterDataMdgBaseDto dto, List<SalesGoalEntity> pullList) {
        this.salesGoalRepository.saveBatch(pullList);
    }

    /**
     * 批量拉取销售日报数据
     *
     * @param masterDataMdgSalesDayReportVos
     * @return java.util.List<com.biz.crm.tpm.business.sales.goal.local.entity.SalesGoalEntity>
     * @author huojia
     * @date 2023/1/2 16:50
     **/
    private void salesGoalValidate(List<MasterDataMdgSalesDayReportVo> masterDataMdgSalesDayReportVos,
                                   List<SalesGoalEntity> pullList, List<MainOneDaySalesDto> dayList) {
        List<String> list = YearSalesTypeEnum.concertSalesDayToList();

        //先循环一次，查产品信息,并同纬度数据聚合
        Set<String> productCodeSet = new HashSet<>();
        Set<String> salesOrgCodeSet = new HashSet<>();
        Map<String, MasterDataMdgSalesDayReportVo> sumMap = new HashMap<>();
        for (MasterDataMdgSalesDayReportVo reportVo : masterDataMdgSalesDayReportVos) {
            //处理商品编码
            if (StringUtils.isNotBlank(reportVo.getProduct()) && reportVo.getProduct().startsWith("000000")) {
                reportVo.setProduct(reportVo.getProduct().substring(6));
            }
            productCodeSet.add(reportVo.getProduct());

            String saleOrgKey = reportVo.getChannel_type() + reportVo.getFormats();
            String insKey = saleOrgKey + reportVo.getSales_organization();
            salesOrgCodeSet.add(insKey);
            String regKey = insKey + reportVo.getManagement_area();
            salesOrgCodeSet.add(regKey);
            String provKey = regKey + reportVo.getProvincial_area();
            salesOrgCodeSet.add(provKey);

            //判断是否同纬度，进行求和操作
            String sumKey = reportVo.getFormats() + reportVo.getSales_organization() + reportVo.getManagement_area()
                    + reportVo.getProvincial_area() + reportVo.getChannel_type() + reportVo.getCustomer() + reportVo.getProduct()
                    + reportVo.getY_id() + reportVo.getM_id();
            if (StringUtils.isNotBlank(reportVo.getD_id())) {
                //                1038586 主体日销售数据，更新逻辑改变    业态编码+销售机构编码+渠道类型+客户编码+产品编码+年月日
                sumKey = reportVo.getFormats() + reportVo.getSales_organization()
                        + reportVo.getChannel_type() + reportVo.getCustomer() + reportVo.getProduct()
                        + reportVo.getY_id() + reportVo.getM_id() + reportVo.getD_id();
            }
            if (sumMap.containsKey(sumKey)) {
                MasterDataMdgSalesDayReportVo sumVo = sumMap.get(sumKey);
                //折前销售额-内控
                String dcm = new BigDecimal(Optional.ofNullable(sumVo.getSales_target_d_cm()).orElse("0").trim()).add(
                        new BigDecimal(Optional.ofNullable(reportVo.getSales_target_d_cm()).orElse("0").trim())).toPlainString();
                sumVo.setSales_target_d_cm(dcm);
                //折前销售额-合同
                String bcm = new BigDecimal(Optional.ofNullable(sumVo.getSales_target_b_cm()).orElse("0").trim()).add(
                        new BigDecimal(Optional.ofNullable(reportVo.getSales_target_b_cm()).orElse("0").trim())).toPlainString();
                sumVo.setSales_target_b_cm(bcm);
                //任务量-内控
                String tdcm = new BigDecimal(Optional.ofNullable(sumVo.getTarget_qty_d_cm()).orElse("0").trim()).add(
                        new BigDecimal(Optional.ofNullable(reportVo.getTarget_qty_d_cm()).orElse("0").trim())).toPlainString();
                sumVo.setTarget_qty_d_cm(tdcm);
                //任务量-合同
                String tbcm = new BigDecimal(Optional.ofNullable(sumVo.getTarget_qty_b_cm()).orElse("0").trim()).add(
                        new BigDecimal(Optional.ofNullable(reportVo.getTarget_qty_b_cm()).orElse("0").trim())).toPlainString();
                sumVo.setTarget_qty_b_cm(tbcm);
                //折后销售额-内控
                String ddcm = new BigDecimal(Optional.ofNullable(sumVo.getDiscount_sales_target_d_cm()).orElse("0").trim()).add(
                        new BigDecimal(Optional.ofNullable(reportVo.getDiscount_sales_target_d_cm()).orElse("0").trim())).toPlainString();
                sumVo.setDiscount_sales_target_d_cm(ddcm);
                //折后销售额-合同
                String dbcm = new BigDecimal(Optional.ofNullable(sumVo.getDiscount_sales_target_b_cm()).orElse("0").trim()).add(
                        new BigDecimal(Optional.ofNullable(reportVo.getDiscount_sales_target_b_cm()).orElse("0").trim())).toPlainString();
                sumVo.setDiscount_sales_target_b_cm(dbcm);
                //销量
                String act1d = new BigDecimal(Optional.ofNullable(sumVo.getActual_qty_3_1_1d()).orElse("0").trim()).add(
                        new BigDecimal(Optional.ofNullable(reportVo.getActual_qty_3_1_1d()).orElse("0").trim())).toPlainString();
                sumVo.setActual_qty_3_1_1d(act1d);
                //折前销售额
                String sal1d = new BigDecimal(Optional.ofNullable(sumVo.getSales_amt_3_1_1d()).orElse("0").trim()).add(
                        new BigDecimal(Optional.ofNullable(reportVo.getSales_amt_3_1_1d()).orElse("0").trim())).toPlainString();
                sumVo.setSales_amt_3_1_1d(sal1d);
                //折后销售额（不含奶卡）
                String dis1d = new BigDecimal(Optional.ofNullable(sumVo.getDiscount_sales_amt_3_1_1d()).orElse("0").trim()).add(
                        new BigDecimal(Optional.ofNullable(reportVo.getDiscount_sales_amt_3_1_1d()).orElse("0").trim())).toPlainString();
                sumVo.setDiscount_sales_amt_3_1_1d(dis1d);
                //奶卡折后订单额（toB）
                String tob = new BigDecimal(Optional.ofNullable(sumVo.getDiscount_mc_order_sales_amt_3_1_1d()).orElse("0").trim()).add(
                        new BigDecimal(Optional.ofNullable(reportVo.getDiscount_mc_order_sales_amt_3_1_1d()).orElse("0").trim())).toPlainString();
                sumVo.setDiscount_mc_order_sales_amt_3_1_1d(tob);
                //奶卡折后订单额（toC）
                String toc = new BigDecimal(Optional.ofNullable(sumVo.getToc_mc_area_order_sales_amt_1d()).orElse("0").trim()).add(
                        new BigDecimal(Optional.ofNullable(reportVo.getToc_mc_area_order_sales_amt_1d()).orElse("0").trim())).toPlainString();
                sumVo.setToc_mc_area_order_sales_amt_1d(toc);
            } else {
                sumMap.put(sumKey, reportVo);
            }
        }
        Map<String, ProductVo> productVoMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(productCodeSet)) {
            List<ProductVo> productVoList = productVoService.findByProductCodes(new ArrayList<>(productCodeSet));
            productVoMap = productVoList.stream().collect(Collectors.toMap(ProductVo::getProductCode, Function.identity()));
        }
        Map<String, ProductVo> finalProductVoMap = productVoMap;

        Map<String, String> salesOrgMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(salesOrgCodeSet)) {
            List<List<String>> salesOrgCodeList = Lists.partition(new ArrayList<>(salesOrgCodeSet), 500);
            salesOrgMap = salesOrgVoService.findNameBySalesOrgCodes(salesOrgCodeList);
        }
        Map<String, String> finalSalesOrgMap = salesOrgMap;

        sumMap.values().forEach(masterDataMdgSalesDayReportVo -> {
            //客户tpm编码
            String cusKey = masterDataMdgSalesDayReportVo.getCustomer() + masterDataMdgSalesDayReportVo.getSales_organization()
                    + masterDataMdgSalesDayReportVo.getChannel_type() + masterDataMdgSalesDayReportVo.getFormats();
            //销售组织tpm编码前缀,销售组织优先使用tpm里的名称
            String saleOrgKey = masterDataMdgSalesDayReportVo.getChannel_type() + masterDataMdgSalesDayReportVo.getFormats();
            String insKey = saleOrgKey + masterDataMdgSalesDayReportVo.getSales_organization();
            masterDataMdgSalesDayReportVo.setSales_organization_tpm(insKey);
            masterDataMdgSalesDayReportVo.setSales_organization_txt(finalSalesOrgMap.getOrDefault(insKey, masterDataMdgSalesDayReportVo.getSales_organization_txt()));

            String regKey = insKey + masterDataMdgSalesDayReportVo.getManagement_area();
            masterDataMdgSalesDayReportVo.setManagement_area_tpm(regKey);
            masterDataMdgSalesDayReportVo.setManagement_area_txt(finalSalesOrgMap.getOrDefault(regKey, masterDataMdgSalesDayReportVo.getManagement_area_txt()));

            String provKey = regKey + masterDataMdgSalesDayReportVo.getProvincial_area();
            masterDataMdgSalesDayReportVo.setProvincial_area_tpm(provKey);
            masterDataMdgSalesDayReportVo.setZsalegrp_t(finalSalesOrgMap.getOrDefault(provKey, masterDataMdgSalesDayReportVo.getZsalegrp_t()));

            //产品信息
            ProductVo productVo = finalProductVoMap.get(masterDataMdgSalesDayReportVo.getProduct());
            if (productVo != null) {
                masterDataMdgSalesDayReportVo.setProduct_txt(productVo.getProductName());
                masterDataMdgSalesDayReportVo.setProductBrandCode(productVo.getProductBrandCode());
                masterDataMdgSalesDayReportVo.setProductBrandName(productVo.getProductBrandName());
                masterDataMdgSalesDayReportVo.setProductCategoryCode(productVo.getProductCategoryCode());
                masterDataMdgSalesDayReportVo.setProductCategoryName(productVo.getProductCategoryName());
                masterDataMdgSalesDayReportVo.setProductItemCode(productVo.getProductLevelCode());
                masterDataMdgSalesDayReportVo.setProductItemName(productVo.getProductLevelName());
            }

            //判断d_id为null的是销售任务数据，有值得为主体日销售数据
            if (StringUtils.isNotBlank(masterDataMdgSalesDayReportVo.getD_id())) {
                MainOneDaySalesDto mainOneDaySalesDto = new MainOneDaySalesDto();
                mainOneDaySalesDto.setYearMonthDay(masterDataMdgSalesDayReportVo.getCalendar_day());
                mainOneDaySalesDto.setYearMonthLy(mainOneDaySalesDto.getYearMonthDay().substring(0, 6));
                mainOneDaySalesDto.setBusinessFormatCode(masterDataMdgSalesDayReportVo.getFormats());
                mainOneDaySalesDto.setBusinessFormatName(null);

                mainOneDaySalesDto.setSalesInstitutionErpCode(masterDataMdgSalesDayReportVo.getSales_organization());
                mainOneDaySalesDto.setSalesInstitutionName(masterDataMdgSalesDayReportVo.getSales_organization_txt());
                mainOneDaySalesDto.setSalesInstitutionCode(masterDataMdgSalesDayReportVo.getSales_organization_tpm());


                mainOneDaySalesDto.setSalesOrgRegionErpCode(masterDataMdgSalesDayReportVo.getManagement_area());
                mainOneDaySalesDto.setSalesOrgRegionName(masterDataMdgSalesDayReportVo.getManagement_area_txt());
                mainOneDaySalesDto.setSalesOrgRegionCode(masterDataMdgSalesDayReportVo.getManagement_area_tpm());

                mainOneDaySalesDto.setSalesOrgProvinceErpCode(masterDataMdgSalesDayReportVo.getProvincial_area());
                mainOneDaySalesDto.setSalesOrgProvinceName(masterDataMdgSalesDayReportVo.getZsalegrp_t());
                mainOneDaySalesDto.setSalesOrgProvinceCode(masterDataMdgSalesDayReportVo.getProvincial_area_tpm());

                mainOneDaySalesDto.setCustomerErpCode(masterDataMdgSalesDayReportVo.getCustomer());
                mainOneDaySalesDto.setCustomerName(masterDataMdgSalesDayReportVo.getCustomer_txt());
                mainOneDaySalesDto.setCustomerCode(masterDataMdgSalesDayReportVo.getCustomer() + masterDataMdgSalesDayReportVo.getSales_organization()
                        + masterDataMdgSalesDayReportVo.getChannel_type() + masterDataMdgSalesDayReportVo.getFormats());
                mainOneDaySalesDto.setChannelCode(masterDataMdgSalesDayReportVo.getChannel_type());
                mainOneDaySalesDto.setChannelName(masterDataMdgSalesDayReportVo.getChannel_type_txt());
                mainOneDaySalesDto.setCustomerClassifyCode(masterDataMdgSalesDayReportVo.getCustomer_category());
                mainOneDaySalesDto.setCustomerClassifyName(masterDataMdgSalesDayReportVo.getCustomer_category_txt());
                mainOneDaySalesDto.setProductCode(masterDataMdgSalesDayReportVo.getProduct());
                mainOneDaySalesDto.setProductName(masterDataMdgSalesDayReportVo.getProduct_txt());
                mainOneDaySalesDto.setNewAndOldProduct(masterDataMdgSalesDayReportVo.getZxlp());
                //金额字段赋值
                mainOneDaySalesDto.setSaleQuantity(new BigDecimal(Optional.ofNullable(masterDataMdgSalesDayReportVo.getActual_qty_3_1_1d()).orElse("0")));
                mainOneDaySalesDto.setDiscountFrontSaleAmount(new BigDecimal(Optional.ofNullable(masterDataMdgSalesDayReportVo.getSales_amt_3_1_1d()).orElse("0")));
                mainOneDaySalesDto.setDiscountBehindSaleAmount(new BigDecimal(Optional.ofNullable(masterDataMdgSalesDayReportVo.getDiscount_sales_amt_3_1_1d()).orElse("0")));
                mainOneDaySalesDto.setMilkCardDiscountBehindOrderAmountToB(new BigDecimal(Optional.ofNullable(masterDataMdgSalesDayReportVo.getDiscount_mc_order_sales_amt_3_1_1d()).orElse("0")));
                mainOneDaySalesDto.setMilkCardDiscountBehindOrderAmountToC(new BigDecimal(Optional.ofNullable(masterDataMdgSalesDayReportVo.getToc_mc_area_order_sales_amt_1d()).orElse("0")));

                mainOneDaySalesDto.setProductBrandCode(masterDataMdgSalesDayReportVo.getProductBrandCode());
                mainOneDaySalesDto.setProductBrandName(masterDataMdgSalesDayReportVo.getProductBrandName());
                mainOneDaySalesDto.setProductCategoryCode(masterDataMdgSalesDayReportVo.getProductCategoryCode());
                mainOneDaySalesDto.setProductCategoryName(masterDataMdgSalesDayReportVo.getProductCategoryName());
                mainOneDaySalesDto.setProductItemCode(masterDataMdgSalesDayReportVo.getProductItemCode());
                mainOneDaySalesDto.setProductItemName(masterDataMdgSalesDayReportVo.getProductItemName());

//                mainOneDaySalesDto.setOnlyKey(mainOneDaySalesDto.getYearMonthDay() + mainOneDaySalesDto.getSalesInstitutionCode() + mainOneDaySalesDto.getSalesOrgRegionCode()
//                        + mainOneDaySalesDto.getSalesOrgProvinceCode() + mainOneDaySalesDto.getCustomerCode() + mainOneDaySalesDto.getCustomerClassifyCode() + mainOneDaySalesDto.getProductCode());
//                1038586 主体日销售数据，更新逻辑改变    业态编码+销售机构编码+渠道类型+客户编码+产品编码+年月日
                mainOneDaySalesDto.setOnlyKey(mainOneDaySalesDto.getYearMonthDay() + mainOneDaySalesDto.getSalesInstitutionCode()
                        + mainOneDaySalesDto.getCustomerCode() +  mainOneDaySalesDto.getProductCode());
                dayList.add(mainOneDaySalesDto);
            } else {
                list.forEach(groupCode -> {
                    SalesGoalEntity salesGoalEntity = new SalesGoalEntity();
                    salesGoalEntity.setYearMonthLy(masterDataMdgSalesDayReportVo.getY_id() + "-" + masterDataMdgSalesDayReportVo.getM_id());
                    salesGoalEntity.setCalendarDay(masterDataMdgSalesDayReportVo.getCalendar_day());
                    salesGoalEntity.setBusinessFormatCode(masterDataMdgSalesDayReportVo.getFormats());
                    salesGoalEntity.setBusinessUnitCode(BusinessUnitEnum.HEADQUARTERS.getCode());

                    //销售机构
                    salesGoalEntity.setSalesInstitutionErpCode(masterDataMdgSalesDayReportVo.getSales_organization());
                    salesGoalEntity.setSalesInstitutionName(masterDataMdgSalesDayReportVo.getSales_organization_txt());
                    salesGoalEntity.setSalesInstitutionCode(masterDataMdgSalesDayReportVo.getSales_organization_tpm());
                    //销售大区
                    salesGoalEntity.setSalesOrgRegionErpCode(masterDataMdgSalesDayReportVo.getManagement_area());
                    salesGoalEntity.setSalesOrgRegionName(masterDataMdgSalesDayReportVo.getManagement_area_txt());
                    salesGoalEntity.setSalesOrgRegionCode(masterDataMdgSalesDayReportVo.getManagement_area_tpm());
                    //销售省区
                    salesGoalEntity.setSalesOrgProvinceErpCode(masterDataMdgSalesDayReportVo.getProvincial_area());
                    salesGoalEntity.setSalesOrgProvinceName(masterDataMdgSalesDayReportVo.getZsalegrp_t());
                    salesGoalEntity.setSalesOrgProvinceCode(masterDataMdgSalesDayReportVo.getProvincial_area_tpm());
                    //客户信息
                    salesGoalEntity.setCustomerCode(cusKey);
                    salesGoalEntity.setCustomerName(masterDataMdgSalesDayReportVo.getCustomer_txt());
                    salesGoalEntity.setCustomerErpCode(masterDataMdgSalesDayReportVo.getCustomer());
                    salesGoalEntity.setCustomerType(masterDataMdgSalesDayReportVo.getCustomer_category());
                    salesGoalEntity.setCustomerChannelCode(masterDataMdgSalesDayReportVo.getChannel_type());
                    salesGoalEntity.setCustomerChannelName(masterDataMdgSalesDayReportVo.getChannel_type_txt());

                    salesGoalEntity.setProductCode(masterDataMdgSalesDayReportVo.getProduct());
                    salesGoalEntity.setProductName(masterDataMdgSalesDayReportVo.getProduct_txt());
                    salesGoalEntity.setNewOldProduct(masterDataMdgSalesDayReportVo.getZxlp());
                    salesGoalEntity.setDataFromCode(DataFromEnum.FOREIGN_SYSTEM.getCode());
                    salesGoalEntity.setYearSalesTypeCode(groupCode);

                    salesGoalEntity.setProductBrandCode(masterDataMdgSalesDayReportVo.getProductBrandCode());
                    salesGoalEntity.setProductBrandName(masterDataMdgSalesDayReportVo.getProductBrandName());
                    salesGoalEntity.setProductCategoryCode(masterDataMdgSalesDayReportVo.getProductCategoryCode());
                    salesGoalEntity.setProductCategoryName(masterDataMdgSalesDayReportVo.getProductCategoryName());
                    salesGoalEntity.setProductItemCode(masterDataMdgSalesDayReportVo.getProductItemCode());
                    salesGoalEntity.setProductItemName(masterDataMdgSalesDayReportVo.getProductItemName());

                    if (StringUtils.isNotEmpty(salesGoalEntity.getSalesOrgProvinceCode())) {
                        salesGoalEntity.setSalesOrgCode(salesGoalEntity.getSalesOrgProvinceCode());
                        salesGoalEntity.setSalesOrgName(salesGoalEntity.getSalesOrgProvinceName());
                    } else if (StringUtils.isNotEmpty(salesGoalEntity.getSalesOrgRegionCode())) {
                        salesGoalEntity.setSalesOrgCode(salesGoalEntity.getSalesOrgRegionCode());
                        salesGoalEntity.setSalesOrgName(salesGoalEntity.getSalesOrgRegionName());
                    } else if (StringUtils.isNotEmpty(salesGoalEntity.getSalesInstitutionCode())) {
                        salesGoalEntity.setSalesOrgCode(salesGoalEntity.getSalesInstitutionCode());
                        salesGoalEntity.setSalesOrgName(salesGoalEntity.getSalesInstitutionName());
                    } else {
                        salesGoalEntity.setSalesOrgCode("");
                        salesGoalEntity.setSalesOrgName("");
                    }

                    if (YearSalesTypeEnum.INTERNAL_CONTROL.getCode().equals(groupCode)) {
                        salesGoalEntity.setDeliveryDiscountSalesAmount(new BigDecimal(Optional.ofNullable(masterDataMdgSalesDayReportVo.getDiscount_sales_target_d_cm()).orElse("0").trim()));
                        salesGoalEntity.setDeliverySalesAmount(new BigDecimal(Optional.ofNullable(masterDataMdgSalesDayReportVo.getSales_target_d_cm()).orElse("0").trim()));
                        salesGoalEntity.setTaskQuantity(new BigDecimal(Optional.ofNullable(masterDataMdgSalesDayReportVo.getTarget_qty_d_cm()).orElse("0").trim()));
                    }
                    if (YearSalesTypeEnum.CONTRACT.getCode().equals(groupCode)) {
                        salesGoalEntity.setDeliveryDiscountSalesAmount(new BigDecimal(Optional.ofNullable(masterDataMdgSalesDayReportVo.getDiscount_sales_target_b_cm()).orElse("0").trim()));
                        salesGoalEntity.setDeliverySalesAmount(new BigDecimal(Optional.ofNullable(masterDataMdgSalesDayReportVo.getSales_target_b_cm()).orElse("0").trim()));
                        salesGoalEntity.setTaskQuantity(new BigDecimal(Optional.ofNullable(masterDataMdgSalesDayReportVo.getTarget_qty_b_cm()).orElse("0").trim()));
                    }
                    salesGoalEntity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                    salesGoalEntity.setTenantCode(TenantUtils.getTenantCode());
                    pullList.add(salesGoalEntity);
                });
            }
        });
    }

    /**
     * 获取锁定的key
     *
     * @param dto
     * @return
     */
    @Override
    public String getLockKey(MasterDataMdgBaseDto dto) {
        String lockKey = DateUtil.format(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY);
        if (!StringUtils.isEmpty(dto.getUdate())) {
            lockKey = dto.getUdate();
        }
        return SalesGoalConstant.SALES_GOAL_LOCK + lockKey;
    }

    /**
     * 根据参数统计折后额和出库数
     *
     * @param salesGoalDto 参数
     * @return SalesGoalSumVo
     */
    @Override
    public SalesGoalSumVo findDiscountAfterAndDeliveryQuantity(SalesGoalDto salesGoalDto) {
        if (StringUtils.isBlank(salesGoalDto.getBusinessFormatCode())
                || StringUtils.isBlank(salesGoalDto.getBusinessUnitCode())
                || StringUtils.isBlank(salesGoalDto.getYearMonthLy())) {
            return null;
        }
        return this.salesGoalRepository.findDiscountAfterAndDeliveryQuantity(salesGoalDto);
    }

    @Override
    public BigDecimal findDiscountAfter(SalesGoalDto salesGoalDto) {
        return this.salesGoalRepository.findDiscountAfter(salesGoalDto);
    }

    /**
     * 解锁
     *
     * @param lockKey
     * @author huojia
     * @date 2023/1/2 16:48
     **/
    private void unLock(String lockKey) {
        if (StringUtils.isEmpty(lockKey)) {
            throw new RuntimeException("拉取销售日报数据解锁失败，日期不能为空！");
        }
        redisLockService.unlock(lockKey);
    }

    /**
     * @param lockKey
     * @return boolean
     * @author huojia
     * @date 2023/1/2 16:47
     **/
    private boolean lock(String lockKey) {
        if (StringUtils.isEmpty(lockKey)) {
            throw new RuntimeException("拉取销售日报数据加锁失败，日期不能为空！");
        }
        return this.redisLockService.tryLock(lockKey, TimeUnit.HOURS, 12);
    }

    /**
     * 销售任务：统计“年月+区域+系统”维度内控版的销售任务折后金额
     * @param salesGoalDtos
     * @return
     */
    @Override
    public List<SalesGoalVo> findListForFR(List<SalesGoalDto> salesGoalDtos) {
        return this.salesGoalRepository.findListForFR(salesGoalDtos);
    }
}
