package com.biz.crm.tpm.business.sales.goal.local.service.process;

import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.channel.sdk.service.CustomerChannelVoService;
import com.biz.crm.mdm.business.customer.channel.sdk.vo.CustomerChannelVo;
import com.biz.crm.mdm.business.customer.retailer.sdk.service.CustomerRetailerVoService;
import com.biz.crm.mdm.business.customer.retailer.sdk.vo.CustomerRetailerVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mdm.business.product.brand.sdk.service.ProductBrandService;
import com.biz.crm.mdm.business.product.brand.sdk.vo.ProductBrandVo;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalVo;
import com.biz.crm.mn.common.base.eunm.DataFromEnum;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.sales.goal.sdk.constant.DictTypeCodeConstant;
import com.biz.crm.tpm.business.sales.goal.sdk.dto.SalesGoalDto;
import com.biz.crm.tpm.business.sales.goal.sdk.service.SalesGoalService;
import com.biz.crm.tpm.business.sales.goal.sdk.vo.SalesGoalEstoreImportsVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author huojia
 * @date 2022年10月28日 11:38
 */
@Slf4j
@Component
public class SalesGoalEstoreImportsProcess implements ImportProcess<SalesGoalEstoreImportsVo> {

    @Autowired(required = false)
    private SalesGoalService salesGoalService;

    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private DictToolkitService dictToolkitService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private CustomerChannelVoService customerChannelVoService;

    @Autowired(required = false)
    private SalesOrgVoService salesOrgVoService;

    /**
     * 数据分片
     *
     * @return 分片长度
     */
    @Override
    public Integer getBatchCount() {
        return 2000;
    }

    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, SalesGoalEstoreImportsVo> data,
                                        TaskGlobalParamsVo paramsVo,
                                        Map<String, Object> params) {
        try {
            Validate.notEmpty(data, "导入数据不能为空！");

            Map<String, String> mdmBusinessFormatMap = new HashMap<>();
            Map<String, String> mdmBusinessUnitMap = new HashMap<>();
            Map<String, List<CustomerVo>> customerVoMap = new HashMap<>();
            Map<String, SalesOrgVo> orgVoMap = new HashMap<>();
            Map<String, CustomerChannelVo> channelVoMap = new HashMap<>();

            log.info("---------------------------->开始校验数据，开始时间<------------------------:" + DateUtil.format(new Date(), DateUtil.DEFAULT_DATE_ALL_PATTERN));
            List<SalesGoalDto> importList = this.validate(data, mdmBusinessFormatMap, mdmBusinessUnitMap, customerVoMap, orgVoMap, channelVoMap);
            log.info("---------------------------->开始校验数据，结束时间<------------------------:" + DateUtil.format(new Date(), DateUtil.DEFAULT_DATE_ALL_PATTERN));

            log.info("---------------------------->开始保存数据，开始时间<------------------------:" + DateUtil.format(new Date(), DateUtil.DEFAULT_DATE_ALL_PATTERN));
            salesGoalService.importSave(importList);
            log.info("---------------------------->开始保存数据，结束时间<------------------------:" + DateUtil.format(new Date(), DateUtil.DEFAULT_DATE_ALL_PATTERN));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            e.printStackTrace();
            throw e;
        }
        return null;
    }

    /**
     * 校验数据
     *
     * @param data
     * @author huojia
     * @date 2022/10/28 14:04
     **/
    private List<SalesGoalDto> validate(LinkedHashMap<Integer, SalesGoalEstoreImportsVo> data,
                                        Map<String, String> mdmBusinessFormatMap,
                                        Map<String, String> mdmBusinessUnitMap,
                                        Map<String, List<CustomerVo>> customerVoMap,
                                        Map<String, SalesOrgVo> orgVoMap,
                                        Map<String, CustomerChannelVo> channelVoMap) {
        List<SalesGoalEstoreImportsVo> salesGoalVerticalImportsVos = new ArrayList<>(data.values());


        // 数据验空
        log.info("---------------------------->开始数据验空，开始时间<------------------------:" + DateUtil.format(new Date(), DateUtil.DEFAULT_DATE_ALL_PATTERN));
        salesGoalVerticalImportsVos.forEach(importsVo ->validateNull(importsVo));
        log.info("---------------------------->开始数据验空，结束时间<------------------------:" + DateUtil.format(new Date(), DateUtil.DEFAULT_DATE_ALL_PATTERN));

        // 构建基础数据map
        this.buildBaseMap(salesGoalVerticalImportsVos, mdmBusinessFormatMap, mdmBusinessUnitMap, customerVoMap, orgVoMap, channelVoMap);

        salesGoalVerticalImportsVos.forEach(importsVo -> {

            // 校验基础数据
            log.info("---------------------------->开始校验基础数据，开始时间<------------------------:" + DateUtil.format(new Date(), DateUtil.DEFAULT_DATE_ALL_PATTERN));
            this.buildBaseData(importsVo, mdmBusinessFormatMap, mdmBusinessUnitMap, customerVoMap, orgVoMap, channelVoMap);
            log.info("---------------------------->开始校验基础数据，结束时间<------------------------:" + DateUtil.format(new Date(), DateUtil.DEFAULT_DATE_ALL_PATTERN));

            // 校验数字
            log.info("---------------------------->开始校验数字，开始时间<------------------------:" + DateUtil.format(new Date(), DateUtil.DEFAULT_DATE_ALL_PATTERN));
            this.buildDecimal(importsVo);
            log.info("---------------------------->开始校验数字，结束时间<------------------------:" + DateUtil.format(new Date(), DateUtil.DEFAULT_DATE_ALL_PATTERN));
        });
        Collection<SalesGoalDto> salesGoalDtoList = nebulaToolkitService.copyCollectionByWhiteList(salesGoalVerticalImportsVos, SalesGoalEstoreImportsVo.class, SalesGoalDto.class, LinkedHashSet.class, ArrayList.class);
        return new ArrayList<>(salesGoalDtoList);
    }

    /**
     * 数据验空
     *
     * @param importsVo
     * @author huojia
     * @date 2023/1/2 11:03
     **/
    private void validateNull(SalesGoalEstoreImportsVo importsVo) {
        if (StringUtils.isEmpty(importsVo.getBusinessFormatCode())) {
            throw new NullPointerException("业态不能为空！");
        }
        if (StringUtils.isEmpty(importsVo.getBusinessUnitCode())) {
            throw new NullPointerException("业务单元不能为空！");
        }
        Validate.notBlank(importsVo.getSalesInstitutionCode(), "销售组织编码不能为空！");
        Validate.notBlank(importsVo.getChannelCode(), "渠道编码不能为空！");
        Validate.notBlank(importsVo.getYearMonthLy(), "年月不能为空！");
        Validate.notBlank(importsVo.getCustomerCode(), "客户编码不能为空！");
        Validate.notBlank(importsVo.getDeliverySalesAmountStr(), "折前目标销售额不能为空！");
    }

    /**
     * 校验数字
     *
     * @param importsVo
     * @author huojia
     * @date 2022/11/10 21:42
     **/
    private void buildDecimal(SalesGoalEstoreImportsVo importsVo) {
        // 出库折前销售额
        try {
            if (!StringUtils.isEmpty(importsVo.getDeliverySalesAmountStr())) {
                importsVo.setDeliverySalesAmount(new BigDecimal(importsVo.getDeliverySalesAmountStr()));
            }
        } catch (Exception e) {
            throw new IllegalArgumentException("折前销售额" + importsVo.getDeliverySalesAmountStr() + "格式错误，请检查！");
        }
    }

    /**
     * 校验基础数据
     *
     * @param importsVo
     * @author huojia
     * @date 2022/11/10 21:41
     **/
    private void buildBaseData(SalesGoalEstoreImportsVo importsVo,
                               Map<String, String> mdmBusinessFormatMap,
                               Map<String, String> mdmBusinessUnitMap,
                               Map<String, List<CustomerVo>> customerVoMap,
                               Map<String, SalesOrgVo> orgVoMap,
                               Map<String, CustomerChannelVo> channelVoMap) {
        // 设置数据来源
        importsVo.setDataFromCode(DataFromEnum.IMPORT.getCode());
        // 业态
        if (!StringUtils.isEmpty(importsVo.getBusinessFormatCode())) {
            if (!mdmBusinessFormatMap.containsKey(importsVo.getBusinessFormatCode())) {
                throw new IllegalArgumentException("业态" + importsVo.getBusinessFormatCode() + "错误，请检查！");
            } else {
                importsVo.setBusinessFormatCode(mdmBusinessFormatMap.get(importsVo.getBusinessFormatCode()));
            }
        }
        // 业务单元
        if (!StringUtils.isEmpty(importsVo.getBusinessUnitCode())) {
            if (!mdmBusinessUnitMap.containsKey(importsVo.getBusinessUnitCode())) {
                throw new IllegalArgumentException("业务单元" + importsVo.getBusinessUnitCode() + "错误，请检查！");
            } else {
                importsVo.setBusinessUnitCode(mdmBusinessUnitMap.get(importsVo.getBusinessUnitCode()));
            }
        }
        // 年月
        if (!StringUtils.isEmpty(importsVo.getYearMonthLy())) {
            try {
                DateUtil.parseDate(importsVo.getYearMonthLy(), DateUtil.DEFAULT_YEAR_MONTH);
            } catch (Exception ignored) {
                throw new IllegalArgumentException("年月" + importsVo.getYearMonthLy() + "格式错误，应该为" + DateUtil.DEFAULT_YEAR_MONTH);
            }
        }
        //渠道
        if (channelVoMap.containsKey(importsVo.getChannelCode())) {
            importsVo.setChannelName(channelVoMap.get(importsVo.getChannelCode()).getCustomerChannelName());
        } else {
            throw new IllegalArgumentException("渠道编码【" + importsVo.getChannelCode() + "】错误，请检查！");
        }
        //销售机构
        if (orgVoMap.containsKey(importsVo.getSalesInstitutionCode())) {
            importsVo.setSalesInstitutionName(orgVoMap.get(importsVo.getSalesInstitutionCode()).getSalesOrgName());
        } else {
            throw new IllegalArgumentException("销售机构编码【" + importsVo.getSalesInstitutionCode() + "】错误，请检查！");
        }
        //销售部门
        if (StringUtils.isNotBlank(importsVo.getSalesOrgRegionCode())) {
            if (orgVoMap.containsKey(importsVo.getSalesOrgRegionCode())) {
                importsVo.setSalesOrgRegionName(orgVoMap.get(importsVo.getSalesOrgRegionCode()).getSalesOrgName());
            } else {
                throw new IllegalArgumentException("销售部门编码【" + importsVo.getSalesOrgRegionCode() + "】错误，请检查！");
            }
        }
        //销售组
        if (StringUtils.isNotBlank(importsVo.getSalesOrgProvinceCode())) {
            if (orgVoMap.containsKey(importsVo.getSalesOrgProvinceCode())) {
                importsVo.setSalesOrgProvinceName(orgVoMap.get(importsVo.getSalesOrgProvinceCode()).getSalesOrgName());
            } else {
                throw new IllegalArgumentException("销售组编码【" + importsVo.getSalesOrgProvinceCode() + "】错误，请检查！");
            }
        }
        // 客户
        if (customerVoMap.containsKey(importsVo.getCustomerCode())) {
            List<CustomerVo> list = customerVoMap.get(importsVo.getCustomerCode());
            CustomerVo customerVo = list.stream().filter(e -> importsVo.getChannelCode().equals(e.getCustomerChannelCode()) && importsVo.getSalesInstitutionCode().equals(e.getSalesInstitutionCode()) && importsVo.getBusinessFormatCode().equals(e.getBusinessFormatCode()))
                    .findFirst().orElse(null);
            if (customerVo == null) {
                throw new IllegalArgumentException("客户编码【" + importsVo.getCustomerCode() + "】未找到对应客户，请检查！");
            }
            importsVo.setCustomerErpCode(importsVo.getCustomerCode());
            importsVo.setCustomerName(customerVo.getCustomerName());
            importsVo.setCustomerCode(customerVo.getCustomerCode());
        } else {
            throw new IllegalArgumentException("客户编码【" + importsVo.getCustomerCode() + "】错误，请检查！");
        }
    }

    /**
     * 构建基础数据map
     *
     * @param salesGoalVerticalImportsVos
     * @author huojia
     * @date 2022/11/10 21:40
     **/
    private void buildBaseMap(List<SalesGoalEstoreImportsVo> salesGoalVerticalImportsVos,
                              Map<String, String> mdmBusinessFormatMap,
                              Map<String, String> mdmBusinessUnitMap,
                              Map<String, List<CustomerVo>> customerVoMap,
                              Map<String, SalesOrgVo> orgVoMap,
                              Map<String, CustomerChannelVo> channelVoMap) {
        // 查询数据字典
        mdmBusinessFormatMap.putAll(dictToolkitService.findConvertMapByDictTypeCode(DictTypeCodeConstant.MDM_BUSINESS_FORMAT));
        mdmBusinessUnitMap.putAll(dictToolkitService.findConvertMapByDictTypeCode(DictTypeCodeConstant.MDM_BUSINESS_UNIT));

        // 查询主数据数据，校验
        Set<String> customerCodeSet = new HashSet<>();
        Set<String> channelCodeSet = new HashSet<>();
        Set<String> orgCodeSet = new HashSet<>();
        for (SalesGoalEstoreImportsVo vo : salesGoalVerticalImportsVos) {
            customerCodeSet.add(vo.getCustomerCode());
            channelCodeSet.add(vo.getChannelCode());
            orgCodeSet.add(vo.getSalesInstitutionCode());
            if (StringUtils.isNotBlank(vo.getSalesOrgProvinceCode())) {
                orgCodeSet.add(vo.getSalesOrgProvinceCode());
            }
            if (StringUtils.isNotBlank(vo.getSalesOrgRegionCode())) {
                orgCodeSet.add(vo.getSalesOrgRegionCode());
            }
        }

        // 客户
        List<CustomerVo> customerVos = customerVoService.findByCustomerMdgCodes(new ArrayList<>(customerCodeSet));
        if (!CollectionUtils.isEmpty(customerVos)) {
            customerVoMap.putAll(customerVos.stream().collect(Collectors.groupingBy(e -> e.getErpCode())));
        } else {
            throw new IllegalArgumentException("未找到任意客户，请检查！");
        }
        //销售机构
        List<SalesOrgVo> orgVos = salesOrgVoService.findBySalesOrgCodes(new ArrayList<>(orgCodeSet));
        if (org.apache.commons.collections.CollectionUtils.isNotEmpty(orgVos)) {
            orgVoMap.putAll(orgVos.stream().collect(Collectors.toMap(SalesOrgVo::getSalesOrgCode, Function.identity())));
        } else {
            throw new IllegalArgumentException("未找到任意销售机构，请检查！");
        }
        //查询渠道
        List<CustomerChannelVo> channelList = customerChannelVoService.findByCodes(new ArrayList<>(channelCodeSet));
        if (!CollectionUtils.isEmpty(channelList)) {
            channelVoMap.putAll(channelList.stream().collect(Collectors.toMap(e -> e.getCustomerChannelCode(), Function.identity())));
        } else {
            throw new IllegalArgumentException("未找到任意渠道，请检查！");
        }

    }

    @Override
    public Class<SalesGoalEstoreImportsVo> findCrmExcelVoClass() {
        return SalesGoalEstoreImportsVo.class;
    }

    @Override
    public String getTemplateCode() {
        return "TPM_SALES_GOAL_ESTORE_IMPORT";
    }

    @Override
    public String getTemplateName() {
        return "TPM电商销售任务导入";
    }

    @Override
    public String getBusinessCode() {
        return "TPM_SALES_GOAL_VERTICAL_IMPORT";
    }

    @Override
    public String getBusinessName() {
        return "TPM垂直销售目标导入";
    }
}
