package com.biz.crm.tpm.business.sales.goal.local.task.imp;

import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.dto.MasterDataMdgSalesPerformanceDto;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.service.MasterDataMdgService;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.vo.MasterDataMdgSalesPerformanceVo;
import com.biz.crm.tpm.business.sales.goal.local.service.SalesPerformanceService;
import com.biz.crm.tpm.business.sales.goal.sdk.constant.SalesPerformanceConstant;
import com.biz.crm.tpm.business.sales.goal.sdk.task.PullSalesPerformanceTask;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @author yangshen
 * @version 1.0
 * @date 2023/1/9 16:20
 * @ClassName PullSalesPerformanceTaskImpl 定时获取垂直销售业绩
 * @Description TODO
 */
@Component
@Slf4j
public class PullSalesPerformanceTaskImpl implements PullSalesPerformanceTask {

    @Value("${salesPerformance.pageSize:}")
    private String pageSize;

    @Value("${salesPerformance.appCode:}")
    private String appCode;

    @Value("${salesPerformance.advancedDay:}")
    private Integer advancedDay;

    @Autowired(required = false)
    private MasterDataMdgService masterDataMdgService;

    @Autowired(required = false)
    private SalesPerformanceService salesPerformanceService;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Override
    @Transactional
    public void timedTaskPull(){
        log.info("----> {}执行定时任务 <----",this.getDateAndTime(SalesPerformanceConstant.DATE_TIME));
        MasterDataMdgSalesPerformanceDto dto = new MasterDataMdgSalesPerformanceDto();
        //上个月数据
        this.pullPriorMonthDate(dto);
        //当前月数据
        this.pullCurrentMonthData(dto);
        log.info("----> {}定时任务执行完毕 <----",this.getDateAndTime(SalesPerformanceConstant.DATE_TIME));
    }

    /**
     * 拉取前五天的数据定时任务
     */
    @Override
    @DynamicTaskService(cornExpression = "0 0 6,13,18 * * ?", taskDesc = "6，13，18点拉垂直日销售数据")
    public void timedTaskPullDay(){

        loginUserService.refreshAuthentication(null);
        Date date = new Date();
        long startTime = date.getTime();
        log.info("----> {}执行定时任务 <----",this.getDateAndTime(SalesPerformanceConstant.DATE_TIME));
        MasterDataMdgSalesPerformanceDto dto = new MasterDataMdgSalesPerformanceDto();
        //获取前五天数据
        this.pullPriorFiveDay(dto);
        log.info("----> {}定时任务执行完毕 <----",this.getDateAndTime(SalesPerformanceConstant.DATE_TIME));
        long endTime = date.getTime();
        int minutes = (int) ((endTime - startTime)/(1000 * 60));
        log.info("总共耗时 {} 分钟",minutes);
    }

    /**
     * 手动同步某一天数据
     *
     * @param data
     */
    @Override
    public void pullDayData(String data) {
        try {
            DateUtil.date_yyyyMMdd.parse(data);
        } catch (Exception e) {
            throw new IllegalArgumentException("日期格式错误【yyyyMMdd 例：20230206】");
        }
        MasterDataMdgSalesPerformanceDto dto = new MasterDataMdgSalesPerformanceDto();
        this.setDate(dto);
        boolean hasLock = false;
        String key = SalesPerformanceConstant.VERTICAL_SALES_PERFORMANCE_LOCK;
        try {
            hasLock = redisLockService.tryLock(key, TimeUnit.DAYS, 1);
            if (hasLock) {
                dto.setSalesDay(data);
                dto.setSalesMonth(data.substring(0,6));
                dto.setSalesYear(data.substring(0,4));
                this.pullSalesPerformance(dto);
            }
        }catch (Exception e){
            log.error("拉取垂直销售业绩数据失败",e);
            throw e;
        } finally{
            if (hasLock) {
                redisLockService.unlock(SalesPerformanceConstant.VERTICAL_SALES_PERFORMANCE_LOCK);
            }
        }
    }

    /**
     * 获取前五天数据
     * @param dto
     */
    private void pullPriorFiveDay(MasterDataMdgSalesPerformanceDto dto) {
        Integer advancedDay = this.advancedDay;
//        advancedDay = 1;
        this.setDate(dto);
        //获取5天前的日期
        boolean hasLock = false;
        String key = SalesPerformanceConstant.VERTICAL_SALES_PERFORMANCE_LOCK;
        try {
            hasLock = redisLockService.tryLock(key, TimeUnit.DAYS, 1);
            if (hasLock) {
                while (advancedDay > 0) {
                    LocalDate date = LocalDate.now().minusDays(advancedDay);
//                    LocalDate ld = LocalDate.of(2023, 1, 18);
//                    LocalDate date = ld.minusDays(advancedDay);
                    String salesDay = date.format(DateTimeFormatter.ofPattern("yyyyMMdd"));
                    dto.setSalesDay(salesDay);
                    dto.setSalesYear(String.valueOf(date.getYear()));
                    String salesMonth = date.format(DateTimeFormatter.ofPattern("yyyyMM"));
                    dto.setSalesMonth(salesMonth);
                    this.pullSalesPerformance(dto);
                    advancedDay--;
                }
            }
        }catch (Exception e){
            log.error("拉取垂直销售业绩数据失败",e);
            throw e;
        } finally{
            if (hasLock) {
                redisLockService.unlock(SalesPerformanceConstant.VERTICAL_SALES_PERFORMANCE_LOCK);
            }
        }
    }

    /**
     *
     * @param dto
     */
    private void setDate(MasterDataMdgSalesPerformanceDto dto) {
        dto.setPageSize(pageSize);
        dto.setAppCode(appCode);
        //前一天日期 yyyyMMdd
//        String date = getDateAndTime(SalesPerformanceConstant.THE_DAY_BEFORE);
        String date =DateUtil.format(new Date(),DateUtil.DEFAULT_YEAR_MONTH_DAY_NO_CH);
        dto.setDs(date);
    }


    /**
     * 拉取上个月数据
     * @param dto
     */
    private void pullPriorMonthDate(MasterDataMdgSalesPerformanceDto dto) {
        dto.setPageSize(pageSize);
        dto.setAppCode(appCode);
        //前一天日期 yyyyMMdd
        dto.setDs(getDateAndTime(SalesPerformanceConstant.THE_DAY_BEFORE));
        //如果当前月份为1月
        if (SalesPerformanceConstant.ONE_MONTH.equals(getDateAndTime(SalesPerformanceConstant.MONTH))){
            dto.setSalesYear(getDateAndTime(SalesPerformanceConstant.THE_YEAR_BEFORE));
        }
        dto.setSalesMonth(getDateAndTime(SalesPerformanceConstant.THE_MONTH_BEFORE));
        //获取数据
        this.pullSalesPerformance(dto);
    }

    /**
     * 拉取当前月数据
     * @param dto
     */
    private void pullCurrentMonthData(MasterDataMdgSalesPerformanceDto dto) {
        dto.setPageSize(pageSize);
        dto.setAppCode(appCode);
        //前一天日期 yyyyMMdd
        dto.setDs(getDateAndTime(SalesPerformanceConstant.THE_DAY_BEFORE));
        //如果当前月份为1月
        dto.setSalesMonth(getDateAndTime(SalesPerformanceConstant.MONTH));
        dto.setSalesYear(getDateAndTime(SalesPerformanceConstant.YEAR));
        //获取数据
        this.pullSalesPerformance(dto);
    }

    /**
     * 取垂直销售业绩数据
     */
    private void pullSalesPerformance(MasterDataMdgSalesPerformanceDto dto){
        //需要拉取数据总数
        Integer total = this.masterDataMdgService.getSalesPerformanceTotal(dto);
        Integer size = Integer.valueOf(pageSize);
        Integer page = total / size;
        page = total % size == 0 ? page : page + 1;
        log.info("日期为{}, 共{}条数据, 需调用{}次, 每次获取{}条数据",dto.getSalesDay(),total,page,size);
        //先进行删除数据 -----数据库总数-----
//        Integer count = salesPerformanceService.count(new LambdaQueryWrapper<SalesPerformanceEntity>()
//                .eq(SalesPerformanceEntity::getYearCol, dto.getSalesYear())
//                .eq(SalesPerformanceEntity::getSalesDate, dto.getSalesDay()));//获得总数
        Integer count = salesPerformanceService.totalCount(dto);
        //分批次删除数据
        Integer delPage = count / size;
        delPage = count % size == 0 ? delPage : delPage + 1;
        salesPerformanceService.removeBatch(dto,delPage,size);
        //分批次查询
        for (int i = 1; i <= page; i++) {
            Integer pageNumber = i;
            dto.setPageNum(pageNumber.toString());
            log.info("{},开始拉取垂直销售业绩",this.getDateAndTime(SalesPerformanceConstant.DATE_TIME));
            List<MasterDataMdgSalesPerformanceVo> masterDataMdgSalesPerformanceVos = null;
            int flag = 0;
            do{
                try{
                    masterDataMdgSalesPerformanceVos = masterDataMdgService.pullSalesPerformance(dto);
                    flag = 3;
                }catch(Exception e){
                    e.printStackTrace();
                    flag ++;
                    log.error("取垂直销售业绩数据失败异常",e);
                }
            }while(flag < 3);
            log.info("{},拉取数据成功，拉取数据为{}",this.getDateAndTime(SalesPerformanceConstant.DATE_TIME));JSON.toJSONString(masterDataMdgSalesPerformanceVos);
            //数据拷贝
            //使用分批次保存数据
            salesPerformanceService.saveBatchSalesPerformance(masterDataMdgSalesPerformanceVos,dto);
            log.info("{} 第{}批数据保存成功",dto.getSalesDay(), dto.getPageNum());
        }
    }

    /**
     * 获取当前时间
     * @return
     */
    private String getDateAndTime(String flag) {

        if (SalesPerformanceConstant.THE_DAY_BEFORE.equals(flag)){
            LocalDate localDate = LocalDate.now().minusDays(1);
            // 获取当前日期前一天的日期
            return localDate.format(DateTimeFormatter.ofPattern("yyyyMMdd"));
        }

        if (SalesPerformanceConstant.THE_MONTH_BEFORE.equals(flag)){
            LocalDate localDate = LocalDate.now().minusMonths(1);
            return String.valueOf(localDate.getMonthValue());
        }

        if (SalesPerformanceConstant.THE_YEAR_BEFORE.equals(flag)){
            LocalDate localDate = LocalDate.now().minusYears(1);
            return localDate.getYear() + String.valueOf(localDate.getMonthValue());
        }

        if (SalesPerformanceConstant.YEAR.equals(flag)){
            LocalDate localDate = LocalDate.now();
            return String.valueOf(localDate.getYear());
        }

        if (SalesPerformanceConstant.MONTH.equals(flag)){
            LocalDate localDate = LocalDate.now();
            return localDate.getYear() + String.valueOf(localDate.getYear());
        }
        Date date = new Date();
        SimpleDateFormat format;
        if (SalesPerformanceConstant.DATE_TIME.equals(flag)){
            format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            return format.format(date);
        }
        format = new SimpleDateFormat("yyyyMMdd");
        return format.format(date);
    }

}
