package com.biz.crm.tpm.business.sales.plan.local.service.internal;

import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerOrgChannelStoresVo;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.third.system.ecrm.sdk.dto.MainPlanAndReplyDto;
import com.biz.crm.mn.third.system.ecrm.sdk.dto.MainPlanAndReplyItemDto;
import com.biz.crm.mn.third.system.ecrm.sdk.service.MainPlanAndReplyService;
import com.biz.crm.tpm.business.sales.plan.local.entity.SalesPlanEntity;
import com.biz.crm.tpm.business.sales.plan.local.service.MainPlanAndReplyLogService;
import com.biz.crm.tpm.business.sales.plan.local.service.SalesPlanEntityService;
import com.biz.crm.tpm.business.sales.plan.sdk.constant.SalesPlanConstant;
import com.biz.crm.tpm.business.sales.plan.sdk.dto.SalesPlanDto;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author: chenlong
 * @date: 2023/1/12 10:36
 * @description: 主体计划量和回复量服务接口实现
 */
@Service("mainPlanAndReplyService")
@Slf4j
public class MainPlanAndReplyServiceImpl implements MainPlanAndReplyService {

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;
    @Autowired(required = false)
    private RedisLockService redisLockService;
    @Autowired(required = false)
    private SalesPlanEntityService salesPlanEntityService;
    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;
    @Autowired(required = false)
    private SalesOrgVoService salesOrgVoService;
    @Autowired(required = false)
    private CustomerVoService customerVoService;
    @Autowired(required = false)
    private ProductVoService productVoService;
    @Autowired(required = false)
    private MainPlanAndReplyLogService mainPlanAndReplyLogService;
    @Autowired(required = false)
    private LoginUserService loginUserService;

    /**
     * 主体计划量和回复量同步（批量）
     *
     * @param dto 合同dto
     */
    @Override
    public void mainPlanAndReplySync(MainPlanAndReplyDto dto) {
        //校验数据
        Validate.notNull(dto, "数据对象为空！");
        Validate.notEmpty(dto.getItemList(), "明细列表[itemList]为空！");
        //客户渠道默认为20
        String cusChannelCode = "20";
        //开始分片处理数据
        List<List<MainPlanAndReplyItemDto>> sliceList = Lists.partition(dto.getItemList(), 2000);
        for (List<MainPlanAndReplyItemDto> itemDtos : sliceList) {
            Set<String> noSet = new HashSet<>();
            Set<String> sapCodes = new HashSet<>();
            List<MainPlanAndReplyItemDto> curList = new ArrayList<>();
            for (MainPlanAndReplyItemDto item : itemDtos) {
                //产品编码去空格
                item.setProductCode(item.getProductCode().trim());
                try {
                    this.createValidate(item);
                } catch (Exception e) {
                    log.error("同步数据异常：{}", e.getMessage());
                    continue;
                }
                item.setCustomerChannelCode(cusChannelCode);
                noSet.add(item.getOnlyKey());
                sapCodes.add(cusChannelCode + item.getBusinessFormatCode() + item.getSalesInstitutionCode());
                if (StringUtils.isNotBlank(item.getSalesOrgRegionCode())) {
                    sapCodes.add(cusChannelCode + item.getBusinessFormatCode() +
                            item.getSalesInstitutionCode() + item.getSalesOrgRegionCode());
                    if (StringUtils.isNotBlank(item.getSalesOrgProvinceCode())) {
                        sapCodes.add(cusChannelCode + item.getBusinessFormatCode() +
                                item.getSalesInstitutionCode() + item.getSalesOrgRegionCode() + item.getSalesOrgProvinceCode());
                    }
                }
                curList.add(item);
            }
            if (CollectionUtils.isEmpty(curList)) {
                return;
            }
            //获取销售组织信息和产品信息,校验并填充
            Map<String, SalesOrgVo> salesMap = this.salesOrgVoService.getSalesMapByTpmCodes(new ArrayList<>(sapCodes));
            //循环判断组织信息，并填充
            Set<String> cusCodes = new HashSet<>();
            //过滤销售组织有问题的数据
            List<MainPlanAndReplyItemDto> itemList1 = new ArrayList<>();
            for (MainPlanAndReplyItemDto item : curList) {
                this.validateSalesOrg(item, salesMap, itemList1);
                item.setCustomerChannelCode(cusChannelCode);
                String cusCode = item.getCustomerCode() + item.getSalesInstitutionCodeSap() + cusChannelCode + item.getBusinessFormatCode();
                cusCodes.add(cusCode);
            }
            if (CollectionUtils.isEmpty(itemList1)) {
                return;
            }

            sapCodes.clear();
            salesMap.clear();

            //校验客户信息,组装数据
            List<SalesPlanDto> dtoList = new ArrayList<>();
            List<List<String>> cusList = Lists.partition(new ArrayList<>(cusCodes), 500);
            Map<String, CustomerOrgChannelStoresVo> cusMap = customerVoService.getCustomerByCodes(cusList);
            Set<String> onlyKeySet = new HashSet<>();
            for (MainPlanAndReplyItemDto item : itemList1) {
                String cusCode = item.getCustomerCode() + item.getSalesInstitutionCodeSap() + item.getCustomerChannelCode() + item.getBusinessFormatCode();
                if (!cusMap.containsKey(cusCode)) {
                    log.error("在TPM中，客户编码{}未找到客户信息", item.getCustomerCode());
                    continue;
                }
                CustomerOrgChannelStoresVo storesVo = cusMap.get(cusCode);
                item.setCustomerCode(cusCode);
                item.setSalesOrgRegionCode(storesVo.getSalesRegionCode());
                item.setSalesOrgRegionName(storesVo.getSalesRegionName());
                item.setSalesOrgProvinceCode(storesVo.getSalesOrgCode());
                item.setSalesOrgProvinceName(storesVo.getSalesOrgName());
                item.setCustomerChannelName(storesVo.getChannelName());
                SalesPlanDto planDto = this.nebulaToolkitService.copyObjectByBlankList(item, SalesPlanDto.class, null, null);
                planDto.setSalesInstitutionErpCode(item.getSalesInstitutionCodeSap());
                planDto.setSalesOrgProvinceErpCode(item.getSalesOrgProvinceCodeSap());
                planDto.setSalesOrgRegionErpCode(item.getSalesOrgRegionCodeSap());
                planDto.setSalesOrgCode(item.getSalesOrgRegionCode());
                planDto.setSalesOrgName(item.getSalesOrgRegionName());
                String onlyKey = item.getBusinessFormatCode() + item.getBusinessUnitCode() + item.getYearMonthLy() + item.getCustomerCode() + item.getProductCode() + item.getUnitCode();
                onlyKeySet.add(onlyKey);
                planDto.setOnlyKey(onlyKey);
                dtoList.add(planDto);
            }
            if (CollectionUtils.isEmpty(dtoList)) {
                return;
            }
            cusCodes.clear();
            cusMap.clear();

            //当前版本
            String version = DateUtil.getDataString(DateUtil.date_yyyyMMdd);
            //获取最大版本的数据，根据onlyKey
            List<List<String>> onlyKeys = Lists.partition(new ArrayList<>(onlyKeySet), 500);
            List<SalesPlanEntity> entities = this.salesPlanEntityService.getMaxVersionSalesPlan(onlyKeys);
            Map<String, SalesPlanEntity> planEntityMap = new HashMap<>();
            if (!CollectionUtils.isEmpty(entities)) {
                planEntityMap = entities.stream().collect(Collectors.toMap(SalesPlanEntity::getOnlyKey, Function.identity()));
            }
            List<SalesPlanDto> newList = new ArrayList<>();
            //循环判断数据是否是新版本
            for (SalesPlanDto planDto : dtoList) {
                if (!planEntityMap.containsKey(planDto.getOnlyKey())) {
                    planDto.setVersionNumber(version);
                    newList.add(planDto);
                    continue;
                }
                //若维度相同，直接更新数据
                SalesPlanEntity entity = planEntityMap.get(planDto.getOnlyKey());
                planDto.setId(entity.getId());
                planDto.setVersionNumber(version);
                newList.add(planDto);
            }
            entities.clear();
            planEntityMap.clear();
            //批量加锁
            boolean hasLock = false;
            try {
                hasLock = redisLockService.batchLock(SalesPlanConstant.MONTH_PLAN_ECRM_LOCK, new ArrayList<>(noSet), TimeUnit.SECONDS, 10);
                if (!hasLock) {
                    throw new RuntimeException("主体计划量和回复量数据正在录入中,请勿重复推送");
                }
                this.saveBatchAndLog(newList);
            } catch (Exception e) {
                log.error("", e);
                throw e;
            } finally {
                if (hasLock) {
                    redisLockService.batchUnLock(SalesPlanConstant.MONTH_PLAN_ECRM_LOCK, new ArrayList<>(noSet));
                }
            }
        }
    }

    /**
     * 数据验证，格式校验
     *
     * @param itemDto dto对象
     */
    private void createValidate(MainPlanAndReplyItemDto itemDto) {
        Validate.notBlank(itemDto.getBusinessFormatCode(), "业态编码[businessFormatCode]为空！");
        Validate.notBlank(itemDto.getBusinessFormatName(), "业态名称[businessFormatName]为空！");
        if (StringUtils.isNotBlank(itemDto.getBusinessUnitCode())) {
            itemDto.setBusinessUnitCode(BusinessUnitEnum.descToEnum(itemDto.getBusinessUnitCode()).getCode());
        } else {
            itemDto.setBusinessUnitCode(BusinessUnitEnum.HEADQUARTERS.getCode());
        }
        Validate.notBlank(itemDto.getYearMonthLy(), "年月[yearMonthLy]为空！");
        //年月格式转换
        try {
            itemDto.setYearMonthLy(DateUtil.formatDate(DateUtil.date_yyyyMM.parse(itemDto.getYearMonthLy()), DateUtil.DEFAULT_YEAR_MONTH));
        } catch (Exception e) {
            throw new IllegalArgumentException("年月格式错误");
        }
        Validate.notBlank(itemDto.getSalesInstitutionCode(), "销售机构编码[salesInstitutionCode]为空！");
        Validate.notBlank(itemDto.getSalesInstitutionName(), "销售机构名称[salesInstitutionName]为空！");
        if (StringUtils.isNotBlank(itemDto.getSalesOrgProvinceCode())) {
            Validate.notBlank(itemDto.getSalesOrgProvinceName(), "销售组织名称（省区）[salesOrgProvinceName]为空！");
            Validate.notBlank(itemDto.getSalesOrgRegionCode(), "销售组织编码（大区）[salesOrgRegionCode]为空！");
            Validate.notBlank(itemDto.getSalesOrgRegionName(), "销售组织名称（大区）[salesOrgRegionName]为空！");
        }
//        Validate.notBlank(itemDto.getSalesOrgRegionCode(), "销售组织编码（大区）[salesOrgRegionCode]为空！");
//        Validate.notBlank(itemDto.getSalesOrgRegionName(), "销售组织名称（大区）[salesOrgRegionName]为空！");
//        Validate.notBlank(itemDto.getSalesOrgProvinceCode(), "销售组织编码（省区）[salesOrgProvinceCode]为空！");
//        Validate.notBlank(itemDto.getSalesOrgProvinceName(), "销售组织名称（省区）[salesOrgProvinceName]为空！");

//        Validate.notBlank(itemDto.getCustomerChannelCode(), "客户渠道编码[customerChannelCode]为空！");
//        Validate.notBlank(itemDto.getCustomerChannelName(), "客户渠道名称[customerChannelName]为空！");

        Validate.notBlank(itemDto.getCustomerCode(), "客户编码[customerCode]为空！");
        Validate.notBlank(itemDto.getCustomerName(), "客户名称[customerName]为空！");

        Validate.notBlank(itemDto.getProductCode(), "产品编码[productCode]为空！");
        Validate.notBlank(itemDto.getProductName(), "产品名称[productName]为空！");
        Validate.notBlank(itemDto.getUnitCode(), "单位编码[unitCode]为空！");
        Validate.notBlank(itemDto.getUnitName(), "单位名称[unitName]为空！");
        Validate.notNull(itemDto.getPrice(), "单价[price]为空！");


        Validate.notNull(itemDto.getPlanAmount(), "计划量[planQuantity]为空！");
        Validate.notNull(itemDto.getPlanAmount(), "计划量[planQuantity]为空！");
        Validate.notNull(itemDto.getRestoreQuantity(), "回复数量[restoreQuantity]为空！");
        Validate.notNull(itemDto.getRestoreAmount(), "回复金额[restoreAmount]为空！");

        String onlyKey = itemDto.getBusinessFormatCode() + itemDto.getBusinessUnitCode() + itemDto.getYearMonthLy() + itemDto.getSalesInstitutionCode()
//                + itemDto.getCustomerChannelCode()
                + itemDto.getCustomerCode() + itemDto.getProductCode() + itemDto.getUnitCode();
        itemDto.setOnlyKey(onlyKey);
    }

    /**
     * 数据验证，验证sap组织
     *
     * @param itemDto dto对象
     */
    private void validateSalesOrg(MainPlanAndReplyItemDto itemDto, Map<String, SalesOrgVo> salesMap, List<MainPlanAndReplyItemDto> itemList) {
        //先判断销售组织编码（省区）
        if (StringUtils.isNotBlank(itemDto.getSalesOrgProvinceCode())) {
            String key1 = itemDto.getCustomerChannelCode() + itemDto.getBusinessFormatCode()
                    + itemDto.getSalesInstitutionCode() + itemDto.getSalesOrgRegionCode() + itemDto.getSalesOrgProvinceCode();
            SalesOrgVo vo1 = salesMap.getOrDefault(key1, null);
            if (Objects.isNull(vo1)) {
                log.error("在TPM中，此销售组织编码（省区）[{}}]未查找到数据", itemDto.getSalesOrgProvinceCode());
                return;
            }
            itemDto.setSalesOrgProvinceCode(vo1.getSalesOrgCode());
            itemDto.setSalesOrgProvinceName(vo1.getSalesOrgName());
            itemDto.setSalesOrgProvinceCodeSap(vo1.getErpCode());
            if (StringUtils.isBlank(itemDto.getBusinessUnitCode())) {
                itemDto.setBusinessUnitCode(vo1.getBusinessUnitCode());
            }
            //判断销售组织编码（大区）
            String key2 = itemDto.getCustomerChannelCode() + itemDto.getBusinessFormatCode()
                    + itemDto.getSalesInstitutionCode() + itemDto.getSalesOrgRegionCode();
            SalesOrgVo vo2 = salesMap.getOrDefault(key2, null);
            if (Objects.isNull(vo2)) {
                log.error("在TPM中，此销售组织编码（大区）[{}]未查找到数据", itemDto.getSalesOrgRegionCode());
                return;
            }
            if (StringUtils.isBlank(vo2.getErpCode()) || !vo2.getErpCode().equals(vo1.getParentErpCode())) {
                log.error("在TPM中，[{}]不是[{}]的上级组织",
                        itemDto.getSalesOrgRegionCode(), itemDto.getSalesOrgProvinceCode());
                return;
            }
            itemDto.setSalesOrgRegionCode(vo2.getSalesOrgCode());
            itemDto.setSalesOrgRegionName(vo2.getSalesOrgName());
            itemDto.setSalesOrgRegionCodeSap(vo2.getErpCode());
            //判断销售机构编码
            String key3 = itemDto.getCustomerChannelCode() + itemDto.getBusinessFormatCode() + itemDto.getSalesInstitutionCode();
            SalesOrgVo vo3 = salesMap.getOrDefault(key3, null);
            if (Objects.isNull(vo3)) {
                log.error("在TPM中，此销售机构编码[{}]未查找到数据", itemDto.getSalesInstitutionCode());
                return;
            }
            if (StringUtils.isBlank(vo3.getErpCode()) || !vo3.getErpCode().equals(vo2.getParentErpCode())) {
                log.error("在TPM中，[{}]不是[{}]的上级组织",
                        itemDto.getSalesInstitutionCode(), itemDto.getSalesOrgRegionCode());
                return;
            }
            itemDto.setSalesInstitutionCode(vo3.getSalesOrgCode());
            itemDto.setSalesInstitutionName(vo3.getSalesOrgName());
            itemDto.setSalesInstitutionCodeSap(vo3.getErpCode());
        } else if (StringUtils.isNotBlank(itemDto.getSalesOrgRegionCode())) {
            //判断销售组织编码（大区）
            String key2 = itemDto.getCustomerChannelCode() + itemDto.getBusinessFormatCode()
                    + itemDto.getSalesInstitutionCode() + itemDto.getSalesOrgRegionCode();
            SalesOrgVo vo2 = salesMap.getOrDefault(key2, null);
            if (Objects.isNull(vo2)) {
                log.error("在TPM中，此销售组织编码（大区）[{}]未查找到数据", itemDto.getSalesOrgRegionCode());
                return;
            }
            itemDto.setSalesOrgRegionCode(vo2.getSalesOrgCode());
            itemDto.setSalesOrgRegionName(vo2.getSalesOrgName());
            itemDto.setSalesOrgRegionCodeSap(vo2.getErpCode());
            if (StringUtils.isBlank(itemDto.getBusinessUnitCode())) {
                itemDto.setBusinessUnitCode(vo2.getBusinessUnitCode());
            }
            //判断销售机构编码
            String key3 = itemDto.getCustomerChannelCode() + itemDto.getBusinessFormatCode() + itemDto.getSalesInstitutionCode();
            SalesOrgVo vo3 = salesMap.getOrDefault(key3, null);
            if (Objects.isNull(vo3)) {
                log.error("在TPM中，此销售机构编码[{}]未查找到数据", itemDto.getSalesInstitutionCode());
                return;
            }
            if (StringUtils.isBlank(vo3.getErpCode()) || !vo3.getErpCode().equals(vo2.getParentErpCode())) {
                log.error("在TPM中，[{}]不是[{}]的上级组织",
                        itemDto.getSalesInstitutionCode(), itemDto.getSalesOrgRegionCode());
                return;
            }
            itemDto.setSalesInstitutionCode(vo3.getSalesOrgCode());
            itemDto.setSalesInstitutionName(vo3.getSalesOrgName());
            itemDto.setSalesInstitutionCodeSap(vo3.getErpCode());
        } else if (StringUtils.isNotBlank(itemDto.getSalesInstitutionCode())) {
            //判断销售机构编码
            String key3 = itemDto.getCustomerChannelCode() + itemDto.getBusinessFormatCode() + itemDto.getSalesInstitutionCode();
            SalesOrgVo vo3 = salesMap.getOrDefault(key3, null);
            if (Objects.isNull(vo3)) {
                log.error("在TPM中，此销售机构编码[{}]未查找到数据", itemDto.getSalesInstitutionCode());
                return;
            }
            itemDto.setSalesInstitutionCode(vo3.getSalesOrgCode());
            itemDto.setSalesInstitutionName(vo3.getSalesOrgName());
            itemDto.setSalesInstitutionCodeSap(vo3.getErpCode());
            if (StringUtils.isBlank(itemDto.getBusinessUnitCode())) {
                itemDto.setBusinessUnitCode(vo3.getBusinessUnitCode());
            }
        } else {
            log.error("销售组织数据异常");
            return;
        }
        itemList.add(itemDto);
    }

    /**
     * 保存数据，并添加日志
     *
     * @param list dto对象
     */
    public void saveBatchAndLog(List<SalesPlanDto> list) {
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        //拆分新增和编辑数据
        List<SalesPlanDto> updateList = new ArrayList<>();
        List<SalesPlanDto> addList = new ArrayList<>();
        for (SalesPlanDto planDto : list) {
            if (StringUtils.isNotBlank(planDto.getId())) {
                updateList.add(planDto);
            } else {
                addList.add(planDto);
            }
        }
        //获取编辑的历史数据
        List<SalesPlanEntity> oldList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(updateList)) {
            List<String> ids = updateList.stream().map(SalesPlanDto::getId).collect(Collectors.toList());
            List<List<String>> idList = Lists.partition(ids, 1200);
            oldList = salesPlanEntityService.getSalesPlanByIds(idList);
        }

        List<SalesPlanEntity> planList = salesPlanEntityService.saveThirdSystemBatch(addList, updateList);

        if (CollectionUtils.isEmpty(planList)) {
            return;
        }

        Map<String, SalesPlanEntity> oldMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(oldList)) {
            oldMap = oldList.stream().collect(Collectors.toMap(SalesPlanEntity::getId, Function.identity()));
        }
        log.info("主体计划量回复量异步保存日志=====================》");
        AbstractCrmUserIdentity loginDetails = this.loginUserService.getAbstractLoginUser();
        //异步保存日志
        this.mainPlanAndReplyLogService.editLogAsync(planList, oldMap, loginDetails);

        log.info("主体计划量回复量异步更新计算折扣金额=====================》");
        //异步更新计算折扣金额
        this.salesPlanEntityService.syncCalDiscount(planList);
    }
}
