package com.biz.crm.tpm.business.sales.plan.local.service.internal;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.dto.DiscountRateDto;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.enums.DiscountRateVersionEnum;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.service.DiscountRateSdkService;
import com.biz.crm.tpm.business.sales.plan.local.entity.SalesPlanEntity;
import com.biz.crm.tpm.business.sales.plan.local.repository.SalesPlanRepository;
import com.biz.crm.tpm.business.sales.plan.local.service.SalesPlanEntityService;
import com.biz.crm.tpm.business.sales.plan.sdk.dto.SalesPlanDto;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author: chenlong
 * @date: 2023/1/12 14:58
 * @description: 月销售计划(SalesPlanEntity)表服务接口
 */
@Service("salesPlanEntityService")
@Slf4j
public class SalesPlanEntityServiceImpl implements SalesPlanEntityService {

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;
    @Autowired(required = false)
    private SalesPlanRepository salesPlanRepository;
    @Autowired(required = false)
    private DiscountRateSdkService discountRateSdkService;

    /**
     * 保存数据
     *
     * @param addList    新增对象
     * @param updateList 更新对象
     * @return List<SalesPlanEntity>
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public List<SalesPlanEntity> saveThirdSystemBatch(List<SalesPlanDto> addList, List<SalesPlanDto> updateList) {
        List<SalesPlanEntity> newList = new ArrayList<>();
        String tenantCode = TenantUtils.getTenantCode();
        if (!CollectionUtils.isEmpty(addList)) {
            for (SalesPlanDto thirdDto : addList) {
                thirdDto.setTenantCode(tenantCode);
                thirdDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                thirdDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            }
            newList.addAll(this.nebulaToolkitService.copyCollectionByWhiteList(addList, SalesPlanDto.class,
                    SalesPlanEntity.class, HashSet.class, ArrayList.class));
            //组装基本数据
            salesPlanRepository.saveBatch(newList);
        }

        if (!CollectionUtils.isEmpty(updateList)) {
            //组装基本数据
            newList.addAll(this.nebulaToolkitService.copyCollectionByWhiteList(updateList, SalesPlanDto.class,
                    SalesPlanEntity.class, HashSet.class, ArrayList.class));
            salesPlanRepository.updateBatchById(newList);
        }
        return newList;
    }

    /**
     * 异步计算折后数据
     *
     * @param planList 销售计划数据
     */
    @Async
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void syncCalDiscount(List<SalesPlanEntity> planList) {
//        log.error("开始异步计算销售计划折后数据，数据={}", JSONObject.toJSONString(planList));
        if (CollectionUtils.isEmpty(planList)) {
            return;
        }
        //根据业态+业务单元+客户编码+产品编码查询折扣率
        List<String> keys = new ArrayList<>();
        List<String> keys2 = new ArrayList<>();
        for (SalesPlanEntity entity : planList) {
            String key = entity.getBusinessFormatCode() + entity.getBusinessUnitCode() + entity.getYearMonthLy() + entity.getCustomerCode() + entity.getProductCode();
            keys.add(key);
            String key2 = entity.getBusinessFormatCode() + entity.getBusinessUnitCode() + entity.getYearMonthLy() + entity.getCustomerCode();
            keys2.add(key2);
        }
        //查询折扣率
        List<List<String>> keyList = Lists.partition(keys, 500);
        List<DiscountRateDto> list = discountRateSdkService.getDiscountRateByKeys(keyList);
        Map<String, List<DiscountRateDto>> map = new HashMap<>();
        if (!CollectionUtils.isEmpty(list)) {
            map = list.stream().collect(Collectors.groupingBy(
                    i -> i.getBusinessFormatCode() + i.getBusinessUnitCode() + DateUtil.dateToStr(i.getYearAndMonth(), DateUtil.date_yyyy_MM) + i.getCustomerCode() + i.getProductCode()));
        }

        List<List<String>> keyList2 = Lists.partition(keys2, 500);
        List<DiscountRateDto> list2 = discountRateSdkService.getDiscountRateByKeysNoPro(keyList2);
        Map<String, List<DiscountRateDto>> map2 = new HashMap<>();
        if (!CollectionUtils.isEmpty(list2)) {
            map2 = list2.stream().collect(Collectors.groupingBy(
                    i -> i.getBusinessFormatCode() + i.getBusinessUnitCode() + DateUtil.dateToStr(i.getYearAndMonth(), DateUtil.date_yyyy_MM) + i.getCustomerCode()));
        }

        //循环计算折扣率
        List<SalesPlanEntity> ens = new ArrayList<>();
        for (SalesPlanEntity entity : planList) {
//            log.error("销售计划循环，年月={}，客户={}，产品={}",entity.getYearMonthLy(),entity.getCustomerCode(),entity.getProductCode());
            String key = entity.getBusinessFormatCode() + entity.getBusinessUnitCode() + entity.getYearMonthLy() + entity.getCustomerCode() + entity.getProductCode();
            String key2 = entity.getBusinessFormatCode() + entity.getBusinessUnitCode() + entity.getYearMonthLy() + entity.getCustomerCode();
            List<DiscountRateDto> dtos = new ArrayList<>();
            if (map.containsKey(key)) {
                dtos = map.get(key);
            } else if (map2.containsKey(key2)) {
                dtos = map2.get(key2);
            } else {
                log.error("销售计划同步未匹配到折扣率，id={},key={},key2={}", entity.getId(), key, key2);
                continue;
            }
            Map<String, List<DiscountRateDto>> dtoMap = dtos.stream().collect(Collectors.groupingBy(DiscountRateDto::getRateVersion));
            if (dtoMap.containsKey(DiscountRateVersionEnum.CORRECTION.getCode())) {
                BigDecimal rate = dtoMap.get(DiscountRateVersionEnum.CORRECTION.getCode()).get(0).getSystemRate();
                entity.setDiscountPlanAmount(entity.getPlanAmount()
                        .multiply(new BigDecimal(1).subtract(rate)).setScale(2, BigDecimal.ROUND_HALF_UP));
                entity.setDiscountRestoreAmount(entity.getRestoreAmount()
                        .multiply(new BigDecimal(1).subtract(rate)).setScale(2, BigDecimal.ROUND_HALF_UP));
                ens.add(entity);
            } else if (dtoMap.containsKey(DiscountRateVersionEnum.PLAN.getCode())) {
                BigDecimal rate = dtoMap.get(DiscountRateVersionEnum.PLAN.getCode()).get(0).getSystemRate();
                entity.setDiscountPlanAmount(entity.getPlanAmount()
                        .multiply(new BigDecimal(1).subtract(rate)).setScale(2, BigDecimal.ROUND_HALF_UP));
                entity.setDiscountRestoreAmount(entity.getRestoreAmount()
                        .multiply(new BigDecimal(1).subtract(rate)).setScale(2, BigDecimal.ROUND_HALF_UP));
                ens.add(entity);
            }
        }
        if (!CollectionUtils.isEmpty(ens)) {
            salesPlanRepository.updateBatchById(ens);
        }
    }

    /**
     * 通过唯一键集合查询最大版本的销售计划
     *
     * @param onlyKeys 唯一键集合
     * @return List<SalesPlanEntity>
     */
    @Override
    public List<SalesPlanEntity> getMaxVersionSalesPlan(List<List<String>> onlyKeys) {
        if (CollectionUtils.isEmpty(onlyKeys)) {
            return Lists.newArrayList();
        }
        return this.salesPlanRepository.getMaxVersionSalesPlan(onlyKeys, TenantUtils.getTenantCode());
    }

    /**
     * 通过id查询销售计划
     *
     * @param ids id集合
     * @return List<SalesPlanEntity>
     */
    @Override
    public List<SalesPlanEntity> getSalesPlanByIds(List<List<String>> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return Lists.newArrayList();
        }
        return this.salesPlanRepository.getSalesPlanByIds(ids, TenantUtils.getTenantCode());
    }
}
