package com.biz.crm.tpm.business.sales.plan.local.service.internal;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgSubComOrgService;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.org.sdk.service.OrgVoService;
import com.biz.crm.mdm.business.org.sdk.vo.OrgVo;
import com.biz.crm.mdm.business.sales.org.sdk.enums.SalesOrgLevelTypeEnum;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgSubComOrgService;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgAllParentVo;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgSubComOrgVo;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.eunm.DataFromEnum;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.common.rocketmq.service.RocketMqProducer;
import com.biz.crm.mn.common.rocketmq.vo.MqMessageVo;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.service.DiscountRateSdkService;
import com.biz.crm.tpm.business.daily.sales.data.sdk.dto.TpmSapDaySalesDto;
import com.biz.crm.tpm.business.daily.sales.data.sdk.service.TpmSapDaySalesService;
import com.biz.crm.tpm.business.daily.sales.data.sdk.vo.TpmSapDaySalesVo;
import com.biz.crm.tpm.business.profit.goal.discount.sdk.constant.ProfitGoalDiscountConstant;
import com.biz.crm.tpm.business.sales.plan.local.entity.SalesPlanEntity;
import com.biz.crm.tpm.business.sales.plan.local.mapper.SalesPlanMapper;
import com.biz.crm.tpm.business.sales.plan.local.repository.SalesPlanRepository;
import com.biz.crm.tpm.business.sales.plan.local.service.SalesPlanSyncService;
import com.biz.crm.tpm.business.sales.plan.sdk.constant.SalesPlanConstant;
import com.biz.crm.tpm.business.sales.plan.sdk.dto.*;
import com.biz.crm.tpm.business.sales.plan.sdk.enums.ConfirmEunm;
import com.biz.crm.tpm.business.sales.plan.sdk.enums.SummaryDimensionEunm;
import com.biz.crm.tpm.business.sales.plan.sdk.event.SalesPlanLogEventListener;
import com.biz.crm.tpm.business.sales.plan.sdk.service.SalesPlanService;
import com.biz.crm.tpm.business.sales.plan.sdk.vo.SalesPlanVo;
import com.biz.crm.tpm.business.sales.plan.sdk.vo.SubSaleMonitorVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.ibatis.annotations.Param;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author huojia
 * @date 2022年10月26日 14:20
 */
@Slf4j
@Service
public class SalesPlanServiceImpl implements SalesPlanService {

    @Autowired(required = false)
    private SalesOrgVoService salesOrgVoService;

    @Autowired(required = false)
    private SalesPlanMapper salesPlanMapper;

    @Autowired(required = false)
    private SalesPlanRepository salesPlanRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;
    @Autowired(required = false)
    private DiscountRateSdkService discountRateSdkService;
    @Autowired(required = false)
    private RedisLockService redisLockService;
    @Autowired(required = false)
    private SalesPlanSyncService salesPlanSyncService;

    @Autowired
    private SalesOrgSubComOrgService salesOrgSubComOrgService;
    @Autowired(required = false)
    private LoginUserService loginUserService;
    @Autowired(required = false)
    private OrgVoService orgVoService;

    @Autowired(required = false)
    private RocketMqProducer rocketMqProducer;

    @Autowired(required = false)
    private TpmSapDaySalesService tpmSapDaySalesService;

    /**
     * 分页查询销售计划
     *
     * @param pageable
     * @param dto
     * @return com.baomidou.mybatisplus.extension.plugins.pagination.Page<com.biz.crm.tpm.business.sales.plan.sdk.vo.SalesPlanVo>
     * @author huojia
     * @date 2022/11/1 21:12
     **/
    @Override
    public Page<SalesPlanVo> findByConditions(Pageable pageable, SalesPlanDto dto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        dto = ObjectUtils.defaultIfNull(dto, new SalesPlanDto());
        Page<SalesPlanVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.salesPlanMapper.findByConditions(page, dto);
    }

    @Override
    public List<SalesPlanVo> findByConditions(SalesPlanDto dto) {
        if (dto == null) {
            return Lists.newArrayList();
        }
        List<SalesPlanEntity> salesPlans = this.salesPlanRepository.findByConditions(dto);
        return Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(salesPlans, SalesPlanEntity.class, SalesPlanVo.class, HashSet.class, ArrayList.class));
    }

    /**
     * 根据主键查询销售计划
     *
     * @param id
     * @return com.biz.crm.tpm.business.sales.plan.sdk.vo.SalesPlanVo
     * @author huojia
     * @date 2022/11/1 21:12
     **/
    @Override
    public SalesPlanVo findById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        SalesPlanEntity salesPlanEntity = this.salesPlanRepository.getById(id, DelFlagStatusEnum.NORMAL.getCode());
        Validate.notNull(salesPlanEntity, "数据不存在，请刷新后重试！");
        return nebulaToolkitService.copyObjectByBlankList(salesPlanEntity, SalesPlanVo.class, LinkedHashSet.class, ArrayList.class);
    }

    /**
     * 编辑销售计划
     *
     * @param dto
     * @return com.biz.crm.tpm.business.sales.plan.sdk.vo.SalesPlanVo
     * @author huojia
     * @date 2022/11/1 21:12
     **/
    @Override
    @Transactional(rollbackFor = Exception.class)
    public SalesPlanVo update(SalesPlanDto dto) {
        // 根据配置校验必填字段
        this.updateValidate(dto);
        SalesPlanVo salesPlanVo = this.findById(dto.getId());
        if (ObjectUtils.isEmpty(salesPlanVo)) {
            throw new RuntimeException("修改数据失败，原数据不存在！");
        }
        //分子转换出hr组织
        dto = this.convertOrg(Lists.newArrayList(dto)).get(0);
        // 保存销售计划
        SalesPlanDto oldVo = this.nebulaToolkitService.copyObjectByWhiteList(salesPlanVo, SalesPlanDto.class, LinkedHashSet.class, ArrayList.class);
        SalesPlanEntity salesPlanEntity = this.nebulaToolkitService.copyObjectByWhiteList(dto, SalesPlanEntity.class, LinkedHashSet.class, ArrayList.class);
        this.salesPlanRepository.saveOrUpdate(salesPlanEntity);
        // 业务日志编辑
        SalesPlanLogEventDto logEventDto = new SalesPlanLogEventDto();
        logEventDto.setOriginal(oldVo);
        logEventDto.setNewest(dto);
        SerializableBiConsumer<SalesPlanLogEventListener, SalesPlanLogEventDto> onUpdate =
                SalesPlanLogEventListener::onUpdate;
        this.nebulaNetEventClient.publish(logEventDto, SalesPlanLogEventListener.class, onUpdate);
        return this.nebulaToolkitService.copyObjectByWhiteList(dto, SalesPlanVo.class, LinkedHashSet.class, ArrayList.class);
    }

    /**
     * 校验必填参数
     *
     * @param dto
     * @author huojia
     * @date 2022/10/26 14:43
     **/
    private void updateValidate(SalesPlanDto dto) {
        if (StringUtils.isEmpty(dto.getTenantCode())) {
            dto.setTenantCode(TenantUtils.getTenantCode());
        }
        this.buildData(dto);
    }

    /**
     * 构建数据信息
     *
     * @param dto
     * @author huojia
     * @date 2023/1/28 10:32
     **/
    private void buildData(SalesPlanDto dto) {
        List<String> salesOrgCodeList = new ArrayList<>();
        if (!StringUtils.isEmpty(dto.getSalesInstitutionCode())) {
            salesOrgCodeList.add(dto.getSalesInstitutionCode());
        }
        if (!StringUtils.isEmpty(dto.getSalesOrgRegionCode())) {
            salesOrgCodeList.add(dto.getSalesOrgRegionCode());
        }
        if (!StringUtils.isEmpty(dto.getSalesOrgProvinceCode())) {
            salesOrgCodeList.add(dto.getSalesOrgProvinceCode());
        }
        List<SalesOrgVo> salesOrgVos = salesOrgVoService.findBySalesOrgCodes(salesOrgCodeList);
        if (!CollectionUtils.isEmpty(salesOrgVos)) {
            Map<String, SalesOrgVo> salesOrgMap = salesOrgVos.stream()
                    .collect(Collectors.toMap(SalesOrgVo::getSalesOrgCode, Function.identity(), (oldVo, newVo) -> newVo));
            if (salesOrgMap.containsKey(dto.getSalesInstitutionCode())) {
                dto.setSalesInstitutionErpCode(salesOrgMap.get(dto.getSalesInstitutionCode()).getErpCode());
            }
            if (salesOrgMap.containsKey(dto.getSalesOrgRegionCode())) {
                dto.setSalesOrgRegionErpCode(salesOrgMap.get(dto.getSalesOrgRegionCode()).getErpCode());
            }
            if (salesOrgMap.containsKey(dto.getSalesOrgProvinceCode())) {
                dto.setSalesOrgProvinceErpCode(salesOrgMap.get(dto.getSalesOrgProvinceCode()).getErpCode());
            }
        }
        //计算税金附加（元）
        if (BusinessUnitEnum.SON_COMPANY.getCode().equals(dto.getBusinessUnitCode())) {
            //税金附加=增值税*增值税及附加税税率
            BigDecimal vat = dto.getVat();
            BigDecimal vatRate = BigDecimal.ZERO;
            //销售机构编码
            String salesInstitutionCode = dto.getSalesInstitutionCode();
            if (org.springframework.util.StringUtils.hasText(salesInstitutionCode)) {
                String saleOrgErpCode = salesInstitutionCode.substring(salesInstitutionCode.length() - 4, salesInstitutionCode.length());
                List<SalesOrgSubComOrgVo> subComOrgVos = this.salesOrgSubComOrgService.findBySaleOrgErpCode(saleOrgErpCode);
                if (!CollectionUtils.isEmpty(subComOrgVos)) {
                    SalesOrgSubComOrgVo salesOrgSubComOrgVo = subComOrgVos.get(0);
                    //增值税及附加税税率
                    vatRate = salesOrgSubComOrgVo.getVatRate();
                }
            }
            if (Objects.isNull(vatRate)) {
                vatRate = new BigDecimal(0.12);
            }
            dto.setTaxesAdditional(vat.multiply(vatRate).setScale(6, BigDecimal.ROUND_UP));
            log.info("增值税1，{}",vat);
            log.info("增值税2，{}",vat);
            log.info("增值税税率2，{}",vatRate);
            log.info("附加税金2,{}",dto.getTaxesAdditional());
            dto.setSpecialDataStatus(ConfirmEunm.TOBECONFIRMED.getCode());
        }
    }

    /**
     * 批量删除销售计划
     *
     * @param ids
     * @author huojia
     * @date 2022/11/1 21:12
     **/
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delete(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            throw new RuntimeException("请选择要删除的数据！");
        }
        List<SalesPlanEntity> salesPlanEntities = this.salesPlanRepository.listByIds(ids);
        if (!CollectionUtils.isEmpty(salesPlanEntities)) {
            salesPlanEntities.forEach(salesPlanEntity -> {
                Validate.isTrue(!DataFromEnum.FOREIGN_SYSTEM.getCode().equals(salesPlanEntity.getDataFromCode()),
                        "通过系统接口同步过来的数据不支持删除！");
            });
            salesPlanRepository.updateDelStatus(ids,DelFlagStatusEnum.DELETE.getCode());
        }
        List<SalesPlanEntity> mqUpdateList = salesPlanEntities.stream().filter(e -> StringUtils.equals(BusinessUnitEnum.SON_COMPANY.getCode(), e.getBusinessUnitCode())).collect(Collectors.toList());
        this.sendMqMsg(mqUpdateList);
        // 业务日志删除
        salesPlanEntities.forEach(salesPlanEntity -> {
            SalesPlanLogEventDto logEventDto = new SalesPlanLogEventDto();
            logEventDto.setOriginal(this.nebulaToolkitService.copyObjectByWhiteList(
                    salesPlanEntity, SalesPlanDto.class, LinkedHashSet.class, ArrayList.class
            ));
            logEventDto.setNewest(null);
            SerializableBiConsumer<SalesPlanLogEventListener, SalesPlanLogEventDto> onDelete =
                    SalesPlanLogEventListener::onDelete;
            this.nebulaNetEventClient.publish(logEventDto, SalesPlanLogEventListener.class, onDelete);
        });
    }

    /**
     * 新增销售计划
     *
     * @param dto
     * @return com.biz.crm.tpm.business.sales.plan.sdk.vo.SalesPlanVo
     * @author huojia
     * @date 2022/11/1 21:13
     **/
    @Override
    @Transactional(rollbackFor = Exception.class)
    public SalesPlanVo create(SalesPlanDto dto) {
        // 根据配置校验必填字段
        this.createValidate(dto);
        dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        //分子转换出hr组织
        dto = this.convertOrg(Lists.newArrayList(dto)).get(0);
        // 保存销售计划
        SalesPlanEntity salesPlanEntity = this.nebulaToolkitService.copyObjectByWhiteList(dto, SalesPlanEntity.class, LinkedHashSet.class, ArrayList.class);
        this.salesPlanRepository.saveOrUpdate(salesPlanEntity);
        dto.setId(salesPlanEntity.getId());
        // 业务日志新增
        SalesPlanLogEventDto logEventDto = new SalesPlanLogEventDto();
        logEventDto.setOriginal(null);
        logEventDto.setNewest(dto);
        SerializableBiConsumer<SalesPlanLogEventListener, SalesPlanLogEventDto> onCreate =
                SalesPlanLogEventListener::onCreate;
        this.nebulaNetEventClient.publish(logEventDto, SalesPlanLogEventListener.class, onCreate);
        return this.nebulaToolkitService.copyObjectByWhiteList(dto, SalesPlanVo.class, LinkedHashSet.class, ArrayList.class);
    }

    /**
     * 批量导入新增销量计划
     *
     * @param importList
     * @author huojia
     * @date 2022/11/1 21:13
     **/
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void importSave(List<SalesPlanDto> importList) {
        if (CollectionUtils.isEmpty(importList)) {
            return;
        }
        importList.forEach(dto -> {
            this.createValidate(dto);
            dto.setTenantCode(TenantUtils.getTenantCode());
            dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            dto.setDataFromCode(DataFromEnum.IMPORT.getCode());
        });
        Collection<SalesPlanEntity> salesGoalEntities = this.nebulaToolkitService.copyCollectionByWhiteList(
                importList, SalesPlanDto.class, SalesPlanEntity.class, LinkedHashSet.class, ArrayList.class
        );

        this.salesPlanRepository.saveBatch(salesGoalEntities);
        // 业务日志创建
        importList.forEach(dto -> {
            SalesPlanLogEventDto logEventDto = new SalesPlanLogEventDto();
            logEventDto.setOriginal(null);
            logEventDto.setNewest(dto);
            SerializableBiConsumer<SalesPlanLogEventListener, SalesPlanLogEventDto> onCreate =
                    SalesPlanLogEventListener::onCreate;
            this.nebulaNetEventClient.publish(logEventDto, SalesPlanLogEventListener.class, onCreate);
        });
    }

    /**
     * 根据条件查询销量计划
     *
     * @param salesPlanDto
     * @return java.util.List<com.biz.crm.tpm.business.sales.plan.sdk.vo.SalesPlanVo>
     * @author huojia
     * @date 2022/11/4 17:41
     **/
    @Override
    public List<SalesPlanVo> listByConditions(SalesPlanDto salesPlanDto) {
        //限制最大四万,数据如果不对,请改成用分页查询接口,不然存在内存泄漏风险
        List<SalesPlanEntity> salesPlanEntities = this.salesPlanMapper.listByConditions(salesPlanDto, 60000);
        if (CollectionUtils.isEmpty(salesPlanEntities)) {
            return Lists.newArrayList();
        }
        return new ArrayList<>(this.nebulaToolkitService.copyCollectionByWhiteList(
                salesPlanEntities, SalesPlanEntity.class, SalesPlanVo.class, LinkedHashSet.class, ArrayList.class
        ));
    }

    /**
     * 根据维度统计回复量
     *
     * @param dto 查询实体
     * @return 统计数据map
     */
    @Override
    public Map<String, BigDecimal> summaryRecoveryAmount(SalesPlanSummaryDto dto) {
        Map<String, BigDecimal> map = new HashMap<>();
        if (StringUtils.isBlank(dto.getDimen()) || null != SummaryDimensionEunm.codeToEnum(dto.getDimen())) {
            return map;
        }
        if (CollectionUtils.isEmpty(dto.getFormatLsit()) || CollectionUtils.isEmpty(dto.getUnitLsit()) || CollectionUtils.isEmpty(dto.getMonthLsit())) {
            return map;
        }
        if (SummaryDimensionEunm.CUSTOMER.getCode().equals(dto.getDimen())) {
            if (CollectionUtils.isEmpty(dto.getCusLsit())) {
                return map;
            }
        } else if (SummaryDimensionEunm.PRODUCT.getCode().equals(dto.getDimen())) {
            if (CollectionUtils.isEmpty(dto.getProLsit())) {
                return map;
            }
        } else if (SummaryDimensionEunm.CUSTOMER_PRODUCT.getCode().equals(dto.getDimen())) {
            if (CollectionUtils.isEmpty(dto.getCusLsit()) || CollectionUtils.isEmpty(dto.getProLsit())) {
                return map;
            }
        }

        if (SummaryDimensionEunm.CUSTOMER.getCode().equals(dto.getDimen())) {
            List<SalesPlanEntity> list = this.salesPlanRepository.summaryRecoveryAmountCus(dto, TenantUtils.getTenantCode());
            if (CollectionUtils.isEmpty(list)) {
                return map;
            }
            map = list.stream().collect(Collectors.toMap(i -> i.getBusinessFormatCode() + i.getBusinessUnitCode() +
                    i.getYearMonthLy() + i.getCustomerCode(), SalesPlanEntity::getRestoreQuantity));
        } else if (SummaryDimensionEunm.PRODUCT.getCode().equals(dto.getDimen())) {
            List<SalesPlanEntity> list = this.salesPlanRepository.summaryRecoveryAmountPro(dto, TenantUtils.getTenantCode());
            if (CollectionUtils.isEmpty(list)) {
                return map;
            }
            map = list.stream().collect(Collectors.toMap(i -> i.getBusinessFormatCode() + i.getBusinessUnitCode() +
                    i.getYearMonthLy() + i.getCustomerCode(), SalesPlanEntity::getRestoreQuantity));
        } else if (SummaryDimensionEunm.CUSTOMER_PRODUCT.getCode().equals(dto.getDimen())) {
            List<SalesPlanEntity> list = this.salesPlanRepository.summaryRecoveryAmountCusAndPro(dto, TenantUtils.getTenantCode());
            if (CollectionUtils.isEmpty(list)) {
                return map;
            }
            map = list.stream().collect(Collectors.toMap(i -> i.getBusinessFormatCode() + i.getBusinessUnitCode() +
                    i.getYearMonthLy() + i.getCustomerCode(), SalesPlanEntity::getRestoreQuantity));
        }
        return map;
    }

    /**
     * 根据选择的维度手动计算折后计划量和回复量
     *
     * @param dto 查询实体
     */
    @Override
    public void manualCalPlanAndReply(SalesPlanReCalPlanAndReplyDto dto) {
        Validate.notBlank(dto.getBusinessFormatCode(), "请选择业态");
        Validate.notBlank(dto.getBusinessUnitCode(), "请选择业务单元");
        //业务单元为主体才继续执行
        if (!BusinessUnitEnum.isDefaultBusinessUnit(dto.getBusinessUnitCode())) {
            return;
        }
        Validate.notNull(dto.getYearAndMonth(), "请选择年月");
        dto.setYearMonthLy(DateUtil.dateToStr(dto.getYearAndMonth(), DateUtil.date_yyyy_MM));
        Validate.notBlank(dto.getSalesOrgCode(), "请选择销售组织");
        Validate.notBlank(dto.getSalesOrgName(), "销售组织名称不能为空");
        //通过选择的维度数据循环查询销售计划数据，并计算折扣率
        SalesOrgVo salesOrgVo = salesOrgVoService.findBySalesOrgCode(dto.getSalesOrgCode());
        Validate.isTrue(null != salesOrgVo && StringUtils.isNotBlank(salesOrgVo.getSalesOrgLevel()), "未查询到销售组织信息");
        //判断层级
        if (SalesOrgLevelTypeEnum.MECHANISM.getCode().equals(salesOrgVo.getSalesOrgLevel())) {
            dto.setSalesInstitutionCode(salesOrgVo.getSalesOrgCode());
            dto.setSalesInstitutionName(salesOrgVo.getSalesOrgName());
        } else if (SalesOrgLevelTypeEnum.DEPARTMENT.getCode().equals(salesOrgVo.getSalesOrgLevel())) {
            dto.setSalesOrgRegionCode(salesOrgVo.getSalesOrgCode());
            dto.setSalesOrgRegionName(salesOrgVo.getSalesOrgName());
            Validate.notBlank(salesOrgVo.getParentCode(), "销售组织层级为销售部门，未查到上级销售机构");
            dto.setSalesInstitutionCode(salesOrgVo.getParentCode());
            dto.setSalesInstitutionName(salesOrgVo.getParentName());
        } else if (SalesOrgLevelTypeEnum.GROUP.getCode().equals(salesOrgVo.getSalesOrgLevel())) {
            dto.setSalesOrgProvinceCode(salesOrgVo.getSalesOrgCode());
            dto.setSalesOrgProvinceName(salesOrgVo.getSalesOrgName());
            Validate.notBlank(salesOrgVo.getParentCode(), "销售组织层级为销售组，未查到上级销售部门");
            dto.setSalesOrgRegionCode(salesOrgVo.getParentCode());
            dto.setSalesOrgRegionName(salesOrgVo.getParentName());
            SalesOrgVo salesOrgVo2 = salesOrgVoService.findBySalesOrgCode(salesOrgVo.getParentCode());
            Validate.notBlank(salesOrgVo.getParentCode(), "销售组织层级为销售组，未查到最上级销售机构");
            dto.setSalesInstitutionCode(salesOrgVo.getParentCode());
            dto.setSalesInstitutionName(salesOrgVo.getParentName());
        }
        //先判断有没有锁
        String key = SalesPlanConstant.MONTH_PLAN_DISCOUNT_LOCK + dto.getBusinessFormatCode() + dto.getBusinessUnitCode() +
                dto.getYearMonthLy() + dto.getSalesInstitutionCode();
        Validate.isTrue(!redisLockService.isLock(key), "此维度的销售计划正在计算当中，请勿重复操作");
        this.salesPlanSyncService.syncCalPlanAndReply(dto);
    }


    /**
     * 组装日志数据
     *
     * @param before   折前金额
     * @param discount 折扣率
     * @return BigDecimal
     **/
    public void packageClass(List<SalesPlanEntity> updateList, List<SalesPlanLogEventDto> logList, SalesPlanEntity record, BigDecimal discount) {
        SalesPlanLogEventDto logDto = new SalesPlanLogEventDto();
        SalesPlanDto old = nebulaToolkitService.copyObjectByBlankList(record, SalesPlanDto.class, null, null);
        logDto.setOriginal(old);
        //计算折后金额
        this.calAfterDiscount(record, discount);
        old.setDiscountPlanAmount(record.getDiscountPlanAmount());
        old.setDiscountRestoreAmount(record.getDiscountRestoreAmount());
        old.setDiscountRate(record.getDiscountRate());
        logDto.setNewest(old);
        updateList.add(record);
        logList.add(logDto);
    }

    /**
     * 计算折后金额
     *
     * @param before   折前金额
     * @param discount 折扣率
     * @return BigDecimal
     **/
    public void calAfterDiscount(SalesPlanEntity record, BigDecimal discount) {
        if (null != record.getPlanAmount()) {
            if (BigDecimal.ZERO.compareTo(record.getPlanAmount()) == 0) {
                record.setDiscountPlanAmount(BigDecimal.ZERO);
            } else {
                record.setDiscountPlanAmount(record.getPlanAmount().multiply(new BigDecimal(100).subtract(discount)).divide(new BigDecimal(100), 4, BigDecimal.ROUND_HALF_UP));
            }
        }
        if (null != record.getRestoreAmount()) {
            if (BigDecimal.ZERO.compareTo(record.getRestoreAmount()) == 0) {
                record.setDiscountRestoreAmount(BigDecimal.ZERO);
            } else {
                record.setDiscountRestoreAmount(record.getRestoreAmount().multiply(new BigDecimal(100).subtract(discount)).divide(new BigDecimal(100), 4, BigDecimal.ROUND_HALF_UP));
            }
        }
        record.setDiscountRate(discount);
    }

    /**
     * 校验必填字段
     *
     * @param dto
     * @author huojia
     * @date 2022/10/26 14:47
     **/
    private void createValidate(SalesPlanDto dto) {
        dto.setId(null);
        dto.setTenantCode(TenantUtils.getTenantCode());
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        dto.setDataFromCode(DataFromEnum.ADD.getCode());
        List<SalesPlanEntity> uniqueness = this.salesPlanRepository.uniqueness(dto);
        Validate.isTrue(CollectionUtils.isEmpty(uniqueness), "唯一性验证不通过,请检查数据是否重复");
        this.buildData(dto);
    }

    @Override
    public List<SalesPlanVo> findSalesPlanSumVo(SalesPlanDto queryDto) {
        if (StringUtils.isEmpty(queryDto.getYearMonthLy()) &&
                CollectionUtil.isEmpty(queryDto.getRegionCodes()) &&
                CollectionUtil.isEmpty(queryDto.getCustomerRetailerCodeList())) {
            return Collections.emptyList();
        }
        queryDto.setTenantCode(TenantUtils.getTenantCode());
        return this.salesPlanMapper.findSalesPlanSumVo(queryDto);
    }

    /**
     * 数据确认
     *
     * @param dto 查询实体
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void confirmData(SalesPlanConfirmDto dto) {
        List<SalesPlanEntity> list = new ArrayList<>();
        if (BooleanEnum.TRUE.getCapital().equals(dto.getAllCheck())) {
            Pageable pageable = Pageable.ofSize(1000);
            SalesPlanDto selectDto = new SalesPlanDto();
            selectDto.setSpecialDataStatus(ConfirmEunm.TOBECONFIRMED.getCode());
            selectDto.setBusinessUnitCode(BusinessUnitEnum.SON_COMPANY.getCode());
/*            FacturerUserDetails loginUserDetails = loginUserService.getLoginDetails(FacturerUserDetails.class);
            Validate.notBlank(loginUserDetails.getOrgCode(),"未查询到当前登录人组织信息");
            selectDto.setOrgCode(loginUserDetails.getOrgCode());*/
            Page<SalesPlanVo> page = this.findByConditions(pageable, selectDto);
            Optional<Long> total = Optional.ofNullable(page).map(Page::getTotal);
            if (total.isPresent()) {
                list.addAll(nebulaToolkitService.copyCollectionByWhiteList(page.getRecords(), SalesPlanVo.class,SalesPlanEntity.class,HashSet.class,ArrayList.class));
                while (page.hasNext()) {
                    pageable = pageable.next();
                    page = this.findByConditions(pageable, selectDto);
                    list.addAll(nebulaToolkitService.copyCollectionByWhiteList(page.getRecords(),SalesPlanVo.class,SalesPlanEntity.class,HashSet.class,ArrayList.class));
                }
            }
        }else {
            Validate.notEmpty(dto.getIds(), "请选择数据");
            list = this.salesPlanRepository.listByIds(dto.getIds());
        }
        Validate.notEmpty(list, "未找到数据");
        AbstractCrmUserIdentity userIdentity = loginUserService.getAbstractLoginUser();
        Validate.notNull(userIdentity, "未获取到当前用户信息");
        for (SalesPlanEntity entity : list) {
            entity.setSpecialDataStatus(ConfirmEunm.COMPLETETHECONFIRMATION.getCode());
            entity.setConfirmedAccount(userIdentity.getAccount());
            entity.setConfirmedName(userIdentity.getRealName());
            entity.setConfirmedOpinion(dto.getConfirmedOpinion());
        }
        List<SalesPlanEntity> mqUpdateList = list.stream().filter(e -> StringUtils.equals(BusinessUnitEnum.SON_COMPANY.getCode(), e.getBusinessUnitCode())).collect(Collectors.toList());
        this.sendMqMsg(mqUpdateList);
        this.salesPlanRepository.updateBatchById(list);
    }

    //更新分子预算预测、自投费用
    private void sendMqMsg(List<SalesPlanEntity> mqUpdateList) {
        if (CollectionUtils.isEmpty(mqUpdateList)) {
            log.error("确认销售计划：没有分子单元销售计划，不更新！");
            return;
        }
        Set<String> salesOrgCodeSet = new HashSet<>();
        List<SalesPlanEntity> updateList = new ArrayList<>();
        for (SalesPlanEntity entity : mqUpdateList) {
            String salesOrgCode = this.getSalesOrgCode(entity);
            if (StringUtils.isEmpty(salesOrgCode)) {
                continue;
            }
            if (StringUtils.isEmpty(entity.getYearMonthLy())) {
                continue;
            }
            salesOrgCodeSet.add(salesOrgCode);
            updateList.add(entity);
        }
        if (CollectionUtils.isEmpty(updateList)) {
            log.error("确认销售计划：销售组织、年月不全，不更新！");
            return;
        }
        List<SalesOrgAllParentVo> salesOrgAllParentVos = salesOrgVoService.findSalesOrgIncludeAllParentByCodes(Lists.newArrayList(salesOrgCodeSet));
        Map<String, SalesOrgAllParentVo> salesOrgMap = salesOrgAllParentVos.stream().collect(Collectors.toMap(SalesOrgAllParentVo::getCurrSalesOrgCode, v -> v, (v1, v2) -> v1));

        //分子公司与销售机构 关系
        List<String> salesInstitutionErpCodeList = salesOrgAllParentVos.stream().map(SalesOrgAllParentVo::getSalesInstitutionErpCode).distinct().collect(Collectors.toList());
        List<SalesOrgSubComOrgVo> list = salesOrgSubComOrgService.listBySalesOrgCodeList(salesInstitutionErpCodeList);
        if (CollectionUtils.isEmpty(list)) {
            log.error("确认销售计划：分子公司与销售机构查询返回空，不更新！");
            return;
        }
        Map<String, Set<String>> relationMap = list.stream().collect(Collectors.groupingBy(SalesOrgSubComOrgVo::getSalesOrgCode, Collectors.mapping(SalesOrgSubComOrgVo::getOrgCode, Collectors.toSet())));

        Set<String> keySet = new HashSet<>();
        for (SalesPlanEntity entity : updateList) {
            String salesOrgCode = this.getSalesOrgCode(entity);
            if (!salesOrgMap.containsKey(salesOrgCode)) {
                log.error("确认销售计划：销售组织{}不存在，不更新！", salesOrgCode);
                continue;
            }

            SalesOrgAllParentVo salesOrgAllParentVo = salesOrgMap.get(salesOrgCode);

            if (!relationMap.containsKey(salesOrgAllParentVo.getSalesInstitutionErpCode())) {
                log.error("确认销售计划：销售机构[{}]分子公司与销售机构关系不存在，不更新！", salesOrgAllParentVo.getSalesInstitutionErpCode());
                continue;
            }

            for (String orgCode : relationMap.get(salesOrgAllParentVo.getSalesInstitutionErpCode())) {
                String key = entity.getBusinessFormatCode() + entity.getBusinessUnitCode() + orgCode + entity.getYearMonthLy();
                keySet.add(key);
            }
        }

        if (CollectionUtils.isEmpty(keySet)) {
            log.error("确认销售计划：更新条件为空，不更新！");
            return;
        }
        MqMessageVo mqMessageVo = new MqMessageVo();
        mqMessageVo.setMsgBody(JSONObject.toJSONString(keySet));
        mqMessageVo.setTag(ProfitGoalDiscountConstant.SALES_PLAN_CAL_BUDGET_FORECAST_TAG);
        rocketMqProducer.sendMqMsg(mqMessageVo);
    }

    private String getSalesOrgCode(SalesPlanEntity entity) {
        if (StringUtils.isEmpty(entity.getSalesInstitutionCode()) && StringUtils.isEmpty(entity.getSalesOrgRegionCode()) && StringUtils.isEmpty(entity.getSalesOrgProvinceCode())) {
            return null;
        }
        if (StringUtils.isNotEmpty(entity.getSalesInstitutionCode())) {
            return entity.getSalesInstitutionCode();
        } else if (StringUtils.isNotEmpty(entity.getSalesOrgRegionCode())) {
            return entity.getSalesOrgRegionCode();
        } else {
            return entity.getSalesOrgProvinceCode();
        }
    }
    @Override
    public List<SalesPlanDto> convertOrg(List<SalesPlanDto> list) {
        if (CollectionUtils.isEmpty(list)) {
            return list;
        }
        List<SalesPlanDto> filerList = list.stream().filter(e -> StringUtils.equals(BusinessUnitEnum.SON_COMPANY.getCode(), e.getBusinessUnitCode())).collect(Collectors.toList());
        List<String> salesInstitutionErpCodeList = filerList.stream().map(SalesPlanDto::getSalesInstitutionErpCode).filter(StringUtils::isNotEmpty).distinct().collect(Collectors.toList());
        if (CollectionUtils.isEmpty(salesInstitutionErpCodeList)) {
            return list;
        }
        List<SalesOrgSubComOrgVo> subComOrgVoList = salesOrgSubComOrgService.listBySalesOrgCodeList(salesInstitutionErpCodeList);
        if (CollectionUtils.isEmpty(subComOrgVoList)) {
            return list;
        }
        Map<String, String> map = subComOrgVoList.stream().filter(e -> StringUtils.isNotEmpty(e.getOrgCode())).collect(Collectors.toMap(SalesOrgSubComOrgVo::getSalesOrgCode, SalesOrgSubComOrgVo::getOrgCode, (v1, v2) -> v1));
        if (CollectionUtils.isEmpty(map)) {
            return list;
        }
        List<OrgVo> orgVos = orgVoService.findByOrgCodes(Lists.newArrayList(map.values()));
        Map<String, String> orgNameMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(orgVos)) {
            orgNameMap = orgVos.stream().collect(Collectors.toMap(OrgVo::getOrgCode, OrgVo::getOrgName, (v1, v2) -> v1));
        }
        for (SalesPlanDto salesPlanDto : list) {
            if (!StringUtils.equals(BusinessUnitEnum.SON_COMPANY.getCode(), salesPlanDto.getBusinessUnitCode())) {
                continue;
            }
            if (StringUtils.isEmpty(salesPlanDto.getSalesInstitutionErpCode())) {
                continue;
            }
            if (!map.containsKey(salesPlanDto.getSalesInstitutionErpCode())) {
                continue;
            }
            salesPlanDto.setOrgCode(map.get(salesPlanDto.getSalesInstitutionErpCode()));
            salesPlanDto.setOrgName(orgNameMap.get(salesPlanDto.getOrgCode()));
        }
        return list;
    }

    @Override
    public Page<SalesPlanVo> findForSalesVolumeMonitoring(Pageable pageable, SalesPlanDto dto) {
        ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(dto)) {
            dto = new SalesPlanDto();
        }
        Page<SalesPlanVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.salesPlanMapper.findForSalesVolumeMonitoring(page, dto);
    }

    /**
     * 分子销量监控
     *
     * @param pageable
     * @param dto
     * @return
     */
    @Override
    public Page<SubSaleMonitorVo> subSaleMonitor(Pageable pageable, SalesPlanDto dto) {
        ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(dto)) {
            dto = new SalesPlanDto();
        }
        dto.setBusinessUnitCode("DY00000009");
        Page<SalesPlanVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        Page<SalesPlanVo> salesPlanVoPage = salesPlanMapper.findForSalesVolumeMonitoring(page, dto);
        List<SalesPlanVo> records = salesPlanVoPage.getRecords();

        Page<SubSaleMonitorVo> resultPage = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        if (CollectionUtils.isEmpty(records)) {
            return resultPage;
        }

        List<SubSaleMonitorVo> list = new ArrayList<>();
        List<TpmSapDaySalesDto> sapDtoList = new ArrayList<>();
        for (SalesPlanVo planVo : records) {
            SubSaleMonitorVo vo = nebulaToolkitService.copyObjectByBlankList(planVo, SubSaleMonitorVo.class, LinkedHashSet.class, ArrayList.class);
            TpmSapDaySalesDto sapDto = new TpmSapDaySalesDto();
            sapDto.setSpartCode(planVo.getBusinessFormatCode());
            sapDto.setYearMonthLy(planVo.getYearMonthLy().replace("-", ""));
            sapDto.setSalesOrgCode(planVo.getSalesInstitutionErpCode());
            sapDto.setMaterialCode(planVo.getProductCode());
            sapDto.setDealerCode(planVo.getCustomerCode().substring(0, 10));
            sapDtoList.add(sapDto);
            vo.setTheoryGrossProfit(vo.getTheoryIncome().subtract(vo.getTheoryCost()));
            vo.setUniqueKey(sapDto.getDealerCode() + sapDto.getSalesOrgCode() + sapDto.getYearMonthLy() + sapDto.getMaterialCode() + sapDto.getSpartCode());
            list.add(vo);
        }
        List<TpmSapDaySalesVo> sapVoList = tpmSapDaySalesService.findListForSubSaleMonitor(sapDtoList);

        if (!CollectionUtils.isEmpty(sapVoList)) {
            Map<String, TpmSapDaySalesVo> sapMap = sapVoList.stream().collect(Collectors.toMap(e -> e.getUniqueKey(), Function.identity(), (o, n) -> n));
            for (SubSaleMonitorVo vo : list) {
                if (!sapMap.containsKey(vo.getUniqueKey())) {
                    continue;
                }
                TpmSapDaySalesVo sapVo = sapMap.get(vo.getUniqueKey());
                vo.setDeliveryOrderNum(sapVo.getDeliveryOrderNum());
                vo.setIncomeTaxIncluded(sapVo.getIncomeTaxIncluded());
                vo.setInvoiceAmt(sapVo.getInvoiceAmt());
                vo.setCostTaxIncluded(sapVo.getCostTaxIncluded());

                vo.setActualTheoryGrossProfit(vo.getIncomeTaxIncluded().subtract(vo.getCostTaxIncluded()));
                if (vo.getPlanQuantity().compareTo(BigDecimal.ZERO) != 0) {
                    vo.setSaleReachRate(new BigDecimal(vo.getDeliveryOrderNum()).divide(vo.getPlanQuantity(), 2, RoundingMode.HALF_UP));
                }
                vo.setSaleDiff(new BigDecimal(vo.getDeliveryOrderNum()).subtract(vo.getPlanQuantity()));
                vo.setDiscountBeforeSaleDiff(vo.getIncomeTaxIncluded().subtract(vo.getTheoryIncome()));
                vo.setDiscountAfterSaleDiff(vo.getInvoiceAmt().subtract(vo.getDiscountedIncome()));
                vo.setTheoryCostDiff(vo.getCostTaxIncluded().subtract(vo.getTheoryCost()));
                vo.setTheoryGrossMarginDiff(vo.getActualTheoryGrossProfit().subtract(vo.getTheoryGrossProfit()));
            }
        }

        resultPage.setTotal(salesPlanVoPage.getTotal());
        resultPage.setCountId(salesPlanVoPage.getCountId());
        resultPage.setCurrent(salesPlanVoPage.getCurrent());
        resultPage.setSize(salesPlanVoPage.getSize());
        resultPage.setPages(salesPlanVoPage.getPages());
        resultPage.setRecords(list);
        return resultPage;
    }

    /**
     * 销管资金池预算项目计算回复量
     *
     * @param dto
     * @return
     */
    @Override
    public List<SalesPlanVo> sumReplayReimburse(SalesPlanDto dto) {
        return salesPlanRepository.sumReplayReimburse(dto);
    }

    /**
     * 其他预留项目预算项目计算回复量
     *
     * @param dto
     * @return
     */
    @Override
    public List<SalesPlanVo> sumReplayRestore(SalesPlanDto dto) {
        return salesPlanRepository.sumReplayRestore(dto);
    }

    /**
     * 其他预留项目预算项目计算计划量
     *
     * @param dto
     * @return
     */
    @Override
    public List<SalesPlanVo> sumReplayPlan(SalesPlanDto dto) {
        return salesPlanRepository.sumReplayPlan(dto);
    }

    @Override
    public BigDecimal findSaleAmount(SalesPlanDto dto) {
        return this.salesPlanMapper.findSaleAmount(dto);
    }

    @Override
    public List<SalesPlanVo> findForSaleAndFeeMonitoring(List<SalesPlanDto> dtos) {
        if(CollectionUtils.isEmpty(dtos)){
            return Lists.newArrayList();
        }
        return this.salesPlanMapper.findForSaleAndFeeMonitoring(dtos);
    }

    public void setSalesOrgCode(SalesPlanEntity entity) {
        if (StringUtils.isNotEmpty(entity.getSalesOrgProvinceCode())) {
            entity.setSalesOrgCode(entity.getSalesOrgProvinceCode());
            entity.setSalesOrgName(entity.getSalesOrgProvinceName());
        } else if (StringUtils.isNotEmpty(entity.getSalesOrgRegionCode())) {
            entity.setSalesOrgCode(entity.getSalesOrgRegionCode());
            entity.setSalesOrgName(entity.getSalesOrgRegionName());
        } else if (StringUtils.isNotEmpty(entity.getSalesInstitutionCode())) {
            entity.setSalesOrgCode(entity.getSalesInstitutionCode());
            entity.setSalesOrgName(entity.getSalesInstitutionName());
        }
    }
}
