package com.biz.crm.tpm.business.sales.plan.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.dto.DiscountRateDto;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.dto.DiscountRatePlanCalDto;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.enums.DiscountRateVersionEnum;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.service.DiscountRateSdkService;
import com.biz.crm.tpm.business.sales.plan.local.entity.SalesPlanEntity;
import com.biz.crm.tpm.business.sales.plan.local.repository.SalesPlanRepository;
import com.biz.crm.tpm.business.sales.plan.local.service.SalesPlanSyncService;
import com.biz.crm.tpm.business.sales.plan.sdk.constant.SalesPlanConstant;
import com.biz.crm.tpm.business.sales.plan.sdk.dto.SalesPlanDto;
import com.biz.crm.tpm.business.sales.plan.sdk.dto.SalesPlanLogEventDto;
import com.biz.crm.tpm.business.sales.plan.sdk.dto.SalesPlanReCalPlanAndReplyDto;
import com.biz.crm.tpm.business.sales.plan.sdk.event.SalesPlanLogEventListener;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author: chenlong
 * @date: 2023/2/18 14:25
 * @description:异步方法接口服务实现类
 */
@Slf4j
@Service("SalesPlanSyncService")
public class SalesPlanSyncServiceImpl implements SalesPlanSyncService {

    @Autowired(required = false)
    private RedisLockService redisLockService;
    @Autowired(required = false)
    private SalesPlanRepository salesPlanRepository;
    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;
    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;
    @Autowired(required = false)
    private DiscountRateSdkService discountRateSdkService;

    /**
     * 根据选择的维度手动计算折后计划量和回复量
     *
     * @param dto 查询实体
     */
    @Override
    @Async
    public void syncCalPlanAndReply(SalesPlanReCalPlanAndReplyDto dto) {
        String lockKey = SalesPlanConstant.MONTH_PLAN_DISCOUNT_LOCK + dto.getBusinessFormatCode() + dto.getBusinessUnitCode() +
                DateUtil.dateToStr(dto.getYearAndMonth(), DateUtil.date_yyyy_MM) + dto.getSalesInstitutionCode();
        //加锁
        boolean hasLock = false;
        try {
            hasLock = redisLockService.tryLock(lockKey, TimeUnit.SECONDS, 30*60);
            if (!hasLock) {
                throw new RuntimeException("此维度的销售计划正在计算当中，请勿重复操作");
            }
            //分页查询销售计划
            Page<SalesPlanEntity> page = new Page<>(1, 500);
            while (true) {
                Page<SalesPlanEntity> pageResult = this.salesPlanRepository.findSalesPlanByConditions(page, dto);
                if (CollectionUtils.isEmpty(pageResult.getRecords())) {
                    break;
                }
                log.info("手动计算折后计划量和回复量,pageNum:{},总条数:{}",pageResult.getCurrent(),pageResult.getTotal());
                //获取客户+产品组合编码，若无产品编码或客户编码直接跳过
                Set<String> cusProCodeSet = new HashSet<>();
                //获取客户编码，若无产品编码或客户编码直接跳过
                Set<String> cusCodeSet = new HashSet<>();
                for (SalesPlanEntity record : pageResult.getRecords()) {
                    if (StringUtils.isNotBlank(record.getCustomerCode()) && StringUtils.isNotBlank(record.getProductCode())) {
                        cusProCodeSet.add(record.getCustomerCode() + record.getProductCode());
                        cusCodeSet.add(record.getCustomerCode());
                    } else if (StringUtils.isNotBlank(record.getCustomerCode())) {
                        cusCodeSet.add(record.getCustomerCode());
                        log.info("手动计算折后计划量和回复量,只有客户编码。销售计划id:{}",record.getId());
                    }else {
                        log.info("手动计算折后计划量和回复量,无客户和产品编码。销售计划id:{}",record.getId());
                    }
                }
                log.info("手动计算折后计划量和回复量,pageNum:{},cusProCodeSet.size:{}",pageResult.getCurrent(),cusProCodeSet.size());
                if (cusProCodeSet.isEmpty() && cusCodeSet.isEmpty()) {
                    continue;
                }
                Map<String, Map<String, DiscountRateDto>> ratesMap1 = new HashMap<>();
                if (!cusProCodeSet.isEmpty()) {
                    DiscountRatePlanCalDto planCalDto1 = new DiscountRatePlanCalDto();
                    planCalDto1.setBusinessFormatCode(dto.getBusinessFormatCode());
                    planCalDto1.setBusinessUnitCode(dto.getBusinessUnitCode());
                    planCalDto1.setYearAndMonth(dto.getYearAndMonth());
                    planCalDto1.setTenantCode(TenantUtils.getTenantCode());
                    planCalDto1.setCusProCodes(new ArrayList<>(cusProCodeSet));
                    List<DiscountRateDto> rates1 = discountRateSdkService.findRateByPlanConditions(planCalDto1);
                    if (!CollectionUtils.isEmpty(rates1)) {
                        ratesMap1 = rates1.stream().collect(Collectors.groupingBy(i -> i.getCustomerCode() + i.getProductCode(), Collectors.toMap(DiscountRateDto::getRateVersion, Function.identity())));
                    }
                }
                Map<String, Map<String, DiscountRateDto>> ratesMap2 = new HashMap<>();
                if (!cusCodeSet.isEmpty()) {
                    DiscountRatePlanCalDto planCalDto2 = new DiscountRatePlanCalDto();
                    planCalDto2.setBusinessFormatCode(dto.getBusinessFormatCode());
                    planCalDto2.setBusinessUnitCode(dto.getBusinessUnitCode());
                    planCalDto2.setYearAndMonth(dto.getYearAndMonth());
                    planCalDto2.setTenantCode(TenantUtils.getTenantCode());
                    planCalDto2.setCusCodes(new ArrayList<>(cusCodeSet));
                    List<DiscountRateDto> rates2 = discountRateSdkService.findRateByPlanConditions(planCalDto2);
                    if (!CollectionUtils.isEmpty(rates2)) {
                        ratesMap2 = rates2.stream().collect(Collectors.groupingBy(DiscountRateDto::getCustomerCode, Collectors.toMap(DiscountRateDto::getRateVersion, Function.identity())));
                    }
                }
                //循环组装数据
                List<SalesPlanEntity> updateList = new ArrayList<>();
                List<SalesPlanLogEventDto> logList = new ArrayList<>();
                for (SalesPlanEntity record : pageResult.getRecords()) {
                    if (StringUtils.isBlank(record.getCustomerCode())) {
                        continue;
                    }
                    //有产品编码
                    if (StringUtils.isNotBlank(record.getProductCode())) {
                        String key = record.getCustomerCode() + record.getProductCode();
                        if (ratesMap1.containsKey(key)) {
                            Map<String, DiscountRateDto> map = ratesMap1.get(key);
                            //先取修正折扣率，没有再取计划折扣率
                            if (map.containsKey(DiscountRateVersionEnum.CORRECTION.getCode())) {
                                this.packageClass(updateList, logList, record, map.get(DiscountRateVersionEnum.CORRECTION.getCode()).getPlanRate());
                            } else if (map.containsKey(DiscountRateVersionEnum.PLAN.getCode())) {
                                this.packageClass(updateList, logList, record, map.get(DiscountRateVersionEnum.PLAN.getCode()).getPlanRate());
                            }
                        }
                        //客户+产品维度没有查到折扣率，再走客户维度查折扣率
                        else if (ratesMap2.containsKey(record.getCustomerCode())) {
                            Map<String, DiscountRateDto> map = ratesMap2.get(record.getCustomerCode());
                            //先取修正折扣率，没有再取计划折扣率
                            if (map.containsKey(DiscountRateVersionEnum.CORRECTION.getCode())) {
                                this.packageClass(updateList, logList, record, map.get(DiscountRateVersionEnum.CORRECTION.getCode()).getPlanRate());
                            } else if (map.containsKey(DiscountRateVersionEnum.PLAN.getCode())) {
                                this.packageClass(updateList, logList, record, map.get(DiscountRateVersionEnum.PLAN.getCode()).getPlanRate());
                            }
                        } else {
                            this.packageClass(updateList, logList, record, BigDecimal.ZERO);
                        }
                    } else {
                        //没有产品编码
                        if (ratesMap2.containsKey(record.getCustomerCode())) {
                            Map<String, DiscountRateDto> map = ratesMap2.get(record.getCustomerCode());
                            //先取修正折扣率，没有再取计划折扣率
                            if (map.containsKey(DiscountRateVersionEnum.CORRECTION.getCode())) {
                                this.packageClass(updateList, logList, record, map.get(DiscountRateVersionEnum.CORRECTION.getCode()).getPlanRate());
                            } else if (map.containsKey(DiscountRateVersionEnum.PLAN.getCode())) {
                                this.packageClass(updateList, logList, record, map.get(DiscountRateVersionEnum.PLAN.getCode()).getPlanRate());
                            }
                        }else {
                            this.packageClass(updateList, logList, record, BigDecimal.ZERO);
                        }
                    }
                }
                //更新数据
                if (!CollectionUtils.isEmpty(updateList)) {
                    this.salesPlanRepository.updateBatchById(updateList);
                    //编辑日志
//                    if (!CollectionUtils.isEmpty(logList)) {
//                        SerializableBiConsumer<SalesPlanLogEventListener, SalesPlanLogEventDto> onUpdate =
//                                SalesPlanLogEventListener::onUpdate;
//                        for (SalesPlanLogEventDto eventDto : logList) {
//                            this.nebulaNetEventClient.publish(eventDto, SalesPlanLogEventListener.class, onUpdate);
//                        }
//                    }
                }
                page.setCurrent(page.getCurrent() + 1);
            }
        } catch (Exception e) {
            log.error("异步方法计算折后金额失败：{}", e.getMessage());
            throw e;
        } finally {
            if (hasLock) {
                redisLockService.unlock(lockKey);
            }
        }
    }

    /**
     * 组装日志数据
     *
     * @param updateList 更新集合
     * @param updateList 日志集合
     * @param record     销售计划
     * @param discount   折扣率
     * @return BigDecimal
     **/
    public void packageClass(List<SalesPlanEntity> updateList, List<SalesPlanLogEventDto> logList, SalesPlanEntity record, BigDecimal discount) {
        SalesPlanLogEventDto logDto = new SalesPlanLogEventDto();
        SalesPlanDto old = nebulaToolkitService.copyObjectByBlankList(record, SalesPlanDto.class, null, null);
        logDto.setOriginal(old);
        //计算折后金额
        this.calAfterDiscount(record, discount);
        old.setDiscountPlanAmount(record.getDiscountPlanAmount());
        old.setDiscountRestoreAmount(record.getDiscountRestoreAmount());
        old.setDiscountRate(record.getDiscountRate());
        logDto.setNewest(old);
        updateList.add(record);
        logList.add(logDto);
    }

    /**
     * 计算折后金额
     *
     * @param record   销售计划
     * @param discount 折扣率
     * @return BigDecimal
     **/
    public void calAfterDiscount(SalesPlanEntity record, BigDecimal discount) {
        if (null != record.getPlanAmount()) {
            if (BigDecimal.ZERO.compareTo(record.getPlanAmount()) == 0) {
                record.setDiscountPlanAmount(BigDecimal.ZERO);
            } else {
                record.setDiscountPlanAmount(record.getPlanAmount().multiply(new BigDecimal(1).subtract(discount)).setScale(2, BigDecimal.ROUND_HALF_UP));
            }
        }
        if (null != record.getRestoreAmount()) {
            if (BigDecimal.ZERO.compareTo(record.getRestoreAmount()) == 0) {
                record.setDiscountRestoreAmount(BigDecimal.ZERO);
            } else {
                record.setDiscountRestoreAmount(record.getRestoreAmount().multiply(new BigDecimal(1).subtract(discount)).setScale(2, BigDecimal.ROUND_HALF_UP));
            }
        }
        record.setShippingDiscountRate(discount);
    }
}
