package com.biz.crm.tpm.business.sales.plan.local.service.process;

import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.biz.crm.mn.common.base.eunm.DataFromEnum;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.sales.plan.sdk.constant.DictTypeCodeConstant;
import com.biz.crm.tpm.business.sales.plan.sdk.dto.SalesPlanDto;
import com.biz.crm.tpm.business.sales.plan.sdk.service.SalesPlanService;
import com.biz.crm.tpm.business.sales.plan.sdk.vo.SalesPlanHeadImportsVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author huojia
 * @date 2022年10月28日 11:40
 */
@Slf4j
@Component
public class SalesPlanHeadImportsProcess implements ImportProcess<SalesPlanHeadImportsVo> {

    @Resource
    private ProductVoService productVoService;

    @Resource
    private SalesOrgVoService salesOrgVoService;

    @Resource
    private CustomerVoService customerVoService;

    @Resource
    private SalesPlanService salesPlanService;

    @Resource
    private DictToolkitService dictToolkitService;

    @Resource
    private NebulaToolkitService nebulaToolkitService;

    private Map<String, String> mdmBusinessFormatMap = new HashMap<>();
    private Map<String, String> mdmBusinessUnitMap = new HashMap<>();
    private Map<String, CustomerVo> customerVoMap = new HashMap<>();
    private Map<String, ProductVo> productVoMap = new HashMap<>();
    private Map<String, SalesOrgVo> salesOrgVoMap = new HashMap<>();

    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, SalesPlanHeadImportsVo> data,
                                        TaskGlobalParamsVo paramsVo,
                                        Map<String, Object> params) {
        Map<Integer, String> errMap = new HashMap<>();
        try {
            Validate.notEmpty(data, "导入数据不能为空！");
            List<SalesPlanDto> importList = this.validate(data, errMap);
            if (errMap.isEmpty()) {
                salesPlanService.importSave(importList);
            }
        } catch (Exception e) {
            log.error("", e);
            errMap.put(1, errMap.getOrDefault(1, "") + " | " + e.getMessage());
        }
        return errMap;
    }

    /**
     * 数据分片
     *
     * @return 分片长度
     */
    @Override
    public Integer getBatchCount() {
        return Integer.MAX_VALUE;
    }

    /**
     * 校验并转换数据
     *
     * @param data
     * @param errMap
     * @return java.util.List<com.biz.crm.tpm.business.sales.plan.sdk.dto.SalesPlanDto>
     * @author huojia
     * @date 2022/10/28 14:31
     **/
    private List<SalesPlanDto> validate(LinkedHashMap<Integer, SalesPlanHeadImportsVo> data, Map<Integer, String> errMap) {
        List<SalesPlanHeadImportsVo> salesPlanHeadImportsVos = new ArrayList<>(data.values());

        // 构建基础数据map
        this.buildBaseMap(salesPlanHeadImportsVos);
        int index = 1;

        for (SalesPlanHeadImportsVo importsVo : salesPlanHeadImportsVos) {

            index++;
            this.validateNull(importsVo,errMap,index);

            // 校验基础数据
            this.buildBaseData(importsVo,errMap,index);

            // 校验数字
            this.buildDecimal(importsVo,errMap,index);
        }

        Collection<SalesPlanDto> salesPlanDtoList = nebulaToolkitService.copyCollectionByWhiteList(salesPlanHeadImportsVos, SalesPlanHeadImportsVo.class, SalesPlanDto.class, LinkedHashSet.class, ArrayList.class);
        return new ArrayList<>(salesPlanDtoList);
    }

    private void validateNull(SalesPlanHeadImportsVo importsVo, Map<Integer, String> errMap, int index) {
        if (StringUtils.isEmpty(importsVo.getVersionNumber())) {
            //throw new NullPointerException("版本号不能为空！");
            validateIsTrue(false,"版本号不能为空！");
        } else {
            validateIsTrue(DateUtil.isValidDateNotLine(importsVo.getVersionNumber()), "版本号格式有误，例20230823");
        }
        if (StringUtils.isEmpty(importsVo.getBusinessFormatCode())) {
            //throw new NullPointerException("业态不能为空！");
            validateIsTrue(false,"业态不能为空！");
        }
        if (StringUtils.isEmpty(importsVo.getBusinessUnitCode())) {
            //throw new NullPointerException("业务单元不能为空！");
            validateIsTrue(false,"业务单元不能为空！");
        }
        if (StringUtils.isEmpty(importsVo.getYearMonthLy())) {
//            throw new NullPointerException("年月不能为空！");
            validateIsTrue(false,"年月不能为空！");
        }
        if (StringUtils.isEmpty(importsVo.getCustomerCode())) {
//            throw new NullPointerException("客户编码不能为空！");
            validateIsTrue(false,"客户编码不能为空！");
        }
        if (StringUtils.isEmpty(importsVo.getProductCode())) {
//            throw new NullPointerException("产品不能为空！");
            validateIsTrue(false,"产品编码不能为空！");
        }
        if (StringUtils.isEmpty(importsVo.getPlanQuantityStr())) {
//            throw new NullPointerException("计划量不能为空！");
            validateIsTrue(false,"计划数量不能为空！");
        }
        if (StringUtils.isEmpty(importsVo.getPlanAmountStr())) {
//            throw new NullPointerException("计划金额不能为空！");
            validateIsTrue(false,"折前计划金额不能为空！");
        }
        if (StringUtils.isEmpty(importsVo.getDiscountPlanAmountStr())) {
//            throw new NullPointerException("折前计划金额不能为空！");
            validateIsTrue(false,"折后计划金额不能为空！");
        }
        if (StringUtils.isEmpty(importsVo.getRestoreQuantityStr())) {
//            throw new NullPointerException("回复数量不能为空！");
            validateIsTrue(false,"回复数量不能为空！");
        }
        if (StringUtils.isEmpty(importsVo.getRestoreAmountStr())) {
//            throw new NullPointerException("折前回复金额不能为空！");
            validateIsTrue(false,"折前回复金额不能为空！");
        }
        if (StringUtils.isEmpty(importsVo.getDiscountRestoreAmountStr())) {
//            throw new NullPointerException("折后计划金额不能为空！");
            validateIsTrue(false,"折后回复金额不能为空！");
        }
        String errorInfo = this.validateGetErrorInfo();
        if (errorInfo != null) {
            if (errMap.containsKey(index)) {
                errMap.put(index, errMap.getOrDefault(index, "") + "," + errorInfo);
            } else {
                errMap.put(index, errorInfo);
            }
        }
    }

    /**
     * 测试方法
     *
     * @param salesPlanHeadImportsVos
     * @author huojia
     * @date 2023/1/10 23:39
     **/
    public void testValidate(List<SalesPlanHeadImportsVo> salesPlanHeadImportsVos) {
        Map<Integer,String> errMap = new HashMap<>();
        int index = 0;
        // 构建基础数据map
        this.buildBaseMap(salesPlanHeadImportsVos);

        salesPlanHeadImportsVos.forEach(importsVo -> {

            // 校验基础数据
            this.buildBaseData(importsVo,errMap, index);

            // 校验数字
            this.buildDecimal(importsVo, errMap, index);

        });
    }

    /**
     * 校验数字
     *
     * @param importsVo
     * @param errMap
     * @param index
     * @author huojia
     * @date 2022/11/10 21:36
     **/
    private void buildDecimal(SalesPlanHeadImportsVo importsVo, Map<Integer, String> errMap, int index) {
        // 计划量
        try {
            if (!StringUtils.isEmpty(importsVo.getPlanQuantityStr())) {
                importsVo.setPlanQuantity(new BigDecimal(importsVo.getPlanQuantityStr()));
            }
        } catch (Exception e) {
//            throw new IllegalArgumentException("折前计划量" + importsVo.getPlanQuantityStr() + "格式错误，请检查！");
            validateIsTrue(false,"折前计划量" + importsVo.getPlanQuantityStr() + "格式错误，请检查！");
        }
        // 计划量
        try {
            if (!StringUtils.isEmpty(importsVo.getPlanAmountStr())) {
                importsVo.setPlanQuantity(new BigDecimal(importsVo.getPlanAmountStr()));
            }
        } catch (Exception e) {
//            throw new IllegalArgumentException("折前计划金额" + importsVo.getPlanAmountStr() + "格式错误，请检查！");
            validateIsTrue(false,"折前计划金额" + importsVo.getPlanAmountStr() + "格式错误，请检查！");
        }
        try {
            if (!StringUtils.isEmpty(importsVo.getDiscountPlanAmountStr())) {
                importsVo.setDiscountPlanAmount(new BigDecimal(importsVo.getDiscountPlanAmountStr()));
            }
        } catch (Exception e) {
//            throw new IllegalArgumentException("折后计划金额" + importsVo.getDiscountPlanAmountStr() + "格式错误，请检查！");
            validateIsTrue(false,"折后计划金额" + importsVo.getDiscountPlanAmountStr() + "格式错误，请检查！");
        }
        // 回复数量
        try {
            if (!StringUtils.isEmpty(importsVo.getRestoreQuantityStr())) {
                importsVo.setRestoreQuantity(new BigDecimal(importsVo.getRestoreQuantityStr()));
            }
        } catch (Exception e) {
//            throw new IllegalArgumentException("回复数量" + importsVo.getRestoreQuantityStr() + "格式错误，请检查！");
            validateIsTrue(false,"回复数量" + importsVo.getRestoreQuantityStr() + "格式错误，请检查！");
        }
        // 回复金额
        try {
            if (!StringUtils.isEmpty(importsVo.getRestoreAmountStr())) {
                importsVo.setRestoreAmount(new BigDecimal(importsVo.getRestoreAmountStr()));
            }
        } catch (Exception e) {
//            throw new IllegalArgumentException("折前回复金额" + importsVo.getRestoreAmountStr() + "格式错误，请检查！");
            validateIsTrue(false,"折前回复金额" + importsVo.getRestoreAmountStr() + "格式错误，请检查！");
        }
        // 回复金额
        try {
            if (!StringUtils.isEmpty(importsVo.getDiscountPlanAmountStr())) {
                importsVo.setDiscountPlanAmount(new BigDecimal(importsVo.getDiscountPlanAmountStr()));
            }
        } catch (Exception e) {
//            throw new IllegalArgumentException("折后回复金额" + importsVo.getDiscountPlanAmountStr() + "格式错误，请检查！");
            validateIsTrue(false,"折后回复金额" + importsVo.getDiscountPlanAmountStr() + "格式错误，请检查！");
        }
        String errorInfo = this.validateGetErrorInfo();
        if (errorInfo != null) {
            if (errMap.containsKey(index)) {
                errMap.put(index, errMap.get(index) + "," + errorInfo);
            } else {
                errMap.put(index, errorInfo);
            }
        }
    }

    /**
     * 校验基础数据
     *
     * @param importsVo
     * @param errMap
     * @param index
     * @author huojia
     * @date 2022/11/10 21:35
     **/
    private void buildBaseData(SalesPlanHeadImportsVo importsVo, Map<Integer, String> errMap, int index) {
        // 设置数据来源
        importsVo.setDataFromCode(DataFromEnum.IMPORT.getCode());
        // 业态
        if (!StringUtils.isEmpty(importsVo.getBusinessFormatCode())) {
            if (!mdmBusinessFormatMap.containsKey(importsVo.getBusinessFormatCode())) {
//                throw new IllegalArgumentException("业态" + importsVo.getBusinessFormatCode() + "错误，请检查！");
                validateIsTrue(false,"业态 + importsVo.getBusinessFormatCode() + 错误，请检查！");
            } else {
                importsVo.setBusinessFormatCode(mdmBusinessFormatMap.get(importsVo.getBusinessFormatCode()));
            }
        }
        // 业务单元
        if (!StringUtils.isEmpty(importsVo.getBusinessUnitCode())) {
            if (!mdmBusinessUnitMap.containsKey(importsVo.getBusinessUnitCode())) {
                validateIsTrue(false,"业务单元" + importsVo.getBusinessUnitCode() + "错误，请检查！");
//                throw new IllegalArgumentException("业务单元" + importsVo.getBusinessUnitCode() + "错误，请检查！");
            } else {
                importsVo.setBusinessUnitCode(mdmBusinessUnitMap.get(importsVo.getBusinessUnitCode()));
            }
        }
        // 年月
        if (!StringUtils.isEmpty(importsVo.getYearMonthLy())) {
            try {
                DateUtil.parseDate(importsVo.getYearMonthLy(), DateUtil.DEFAULT_YEAR_MONTH);
            } catch (Exception ignored) {
                validateIsTrue(false,"年月" + importsVo.getYearMonthLy() + "格式错误，应该为" + DateUtil.DEFAULT_YEAR_MONTH);
//                throw new IllegalArgumentException("年月" + importsVo.getYearMonthLy() + "格式错误，应该为" + DateUtil.DEFAULT_YEAR_MONTH);
            }
        }
        // 销售组织
        if (!StringUtils.isEmpty(importsVo.getSalesInstitutionCode())) {
            if (!salesOrgVoMap.containsKey(importsVo.getSalesInstitutionCode())) {
//                throw new IllegalArgumentException("销售机构" + importsVo.getSalesInstitutionCode() + "错误，请检查！");
                validateIsTrue(false,"销售机构" + importsVo.getSalesInstitutionCode() + "错误，请检查！");
            } else {
                importsVo.setSalesInstitutionName(salesOrgVoMap.get(importsVo.getSalesInstitutionCode()).getSalesOrgName());
                importsVo.setSalesInstitutionErpCode(salesOrgVoMap.get(importsVo.getSalesInstitutionCode()).getErpCode());
            }
        }
        // 销售组织
        if (!StringUtils.isEmpty(importsVo.getSalesOrgRegionCode())) {
            if (!salesOrgVoMap.containsKey(importsVo.getSalesOrgRegionCode())) {
//                throw new IllegalArgumentException("销售组织（大区）" + importsVo.getSalesOrgRegionCode() + "错误，请检查！");
                validateIsTrue(false,"销售组织（大区）" + importsVo.getSalesOrgRegionCode() + "错误，请检查！");
            } else {
                importsVo.setSalesOrgRegionName(salesOrgVoMap.get(importsVo.getSalesOrgRegionCode()).getSalesOrgName());
                importsVo.setSalesOrgRegionErpCode(salesOrgVoMap.get(importsVo.getSalesOrgRegionCode()).getErpCode());
            }
        }
        // 销售组织
        if (!StringUtils.isEmpty(importsVo.getSalesOrgProvinceCode())) {
            if (!salesOrgVoMap.containsKey(importsVo.getSalesOrgProvinceCode())) {
//                throw new IllegalArgumentException("销售组（省区）" + importsVo.getSalesOrgProvinceCode() + "错误，请检查！");
                validateIsTrue(false,"销售组（省区）" + importsVo.getSalesOrgProvinceCode() + "错误，请检查！");
            } else {
                importsVo.setSalesOrgProvinceName(salesOrgVoMap.get(importsVo.getSalesOrgProvinceCode()).getSalesOrgName());
                importsVo.setSalesOrgProvinceErpCode(salesOrgVoMap.get(importsVo.getSalesOrgProvinceCode()).getErpCode());
            }
        }
        // 客户
        if (!StringUtils.isEmpty(importsVo.getCustomerCode())) {
            if (customerVoMap.containsKey(importsVo.getCustomerCode())) {
                importsVo.setCustomerName(customerVoMap.get(importsVo.getCustomerCode()).getCustomerName());
            } else {
//                throw new IllegalArgumentException("客户编码" + importsVo.getCustomerCode() + "错误，请检查！");
                validateIsTrue(false,"客户编码" + importsVo.getCustomerCode() + "错误，请检查！");
            }
        }
        // 产品
        if (!StringUtils.isEmpty(importsVo.getProductCode())) {
            if (productVoMap.containsKey(importsVo.getProductCode())) {
                importsVo.setProductName(productVoMap.get(importsVo.getProductCode()).getProductName());
                importsVo.setTaxRate(productVoMap.get(importsVo.getProductCode()).getRate());
            } else {
                validateIsTrue(false,"产品编码" + importsVo.getProductCode() + "错误，请检查！");
//                throw new IllegalArgumentException("产品编码" + importsVo.getProductCode() + "错误，请检查！");
            }
        }
        String errorInfo = this.validateGetErrorInfo();
        if (errorInfo != null) {
            if (errMap.containsKey(index)) {
                errMap.put(index, errMap.get(index) + "," + errorInfo);
            } else {
                errMap.put(index, errorInfo);
            }
        }
    }

    /**
     * 构建基础数据map
     *
     * @param salesPlanHeadImportsVos
     * @author huojia
     * @date 2022/11/10 21:34
     **/
    private void buildBaseMap(List<SalesPlanHeadImportsVo> salesPlanHeadImportsVos) {
        // 查询数据字典
        mdmBusinessFormatMap = dictToolkitService.findConvertMapByDictTypeCode(DictTypeCodeConstant.MDM_BUSINESS_FORMAT);
        mdmBusinessUnitMap = dictToolkitService.findConvertMapByDictTypeCode(DictTypeCodeConstant.MDM_BUSINESS_UNIT);

        // 查询主数据数据，校验
        Set<String> salesOrgCodeSet = salesPlanHeadImportsVos.stream().map(SalesPlanHeadImportsVo::getSalesInstitutionCode).filter(Objects::nonNull).collect(Collectors.toSet());
        salesOrgCodeSet.addAll(salesPlanHeadImportsVos.stream().map(SalesPlanHeadImportsVo::getSalesOrgRegionCode).filter(Objects::nonNull).collect(Collectors.toSet()));
        salesOrgCodeSet.addAll(salesPlanHeadImportsVos.stream().map(SalesPlanHeadImportsVo::getSalesOrgProvinceCode).filter(Objects::nonNull).collect(Collectors.toSet()));
        Set<String> customerCodeSet = salesPlanHeadImportsVos.stream().map(SalesPlanHeadImportsVo::getCustomerCode).filter(Objects::nonNull).collect(Collectors.toSet());
        Set<String> productCodeSet = salesPlanHeadImportsVos.stream().map(SalesPlanHeadImportsVo::getProductCode).filter(Objects::nonNull).collect(Collectors.toSet());
        Set<String> yearMonthSet = salesPlanHeadImportsVos.stream().map(SalesPlanHeadImportsVo::getYearMonthLy).filter(Objects::nonNull).collect(Collectors.toSet());

        // 销售组织
        if (!CollectionUtils.isEmpty(salesOrgCodeSet)) {
            List<SalesOrgVo> salesOrgVos = salesOrgVoService.findBySalesOrgCodes(new ArrayList<>(salesOrgCodeSet));
            if (!CollectionUtils.isEmpty(salesOrgVos)) {
                salesOrgVoMap = salesOrgVos.stream().collect(Collectors.toMap(SalesOrgVo::getSalesOrgCode, Function.identity()));
            }
        }

        // 客户
        if (!CollectionUtils.isEmpty(customerCodeSet)) {
            List<CustomerVo> customerVos = customerVoService.findBaseByCustomerCodes(new ArrayList<>(customerCodeSet));
            if (!CollectionUtils.isEmpty(customerVos)) {
                customerVoMap = customerVos.stream().collect(Collectors.toMap(CustomerVo::getCustomerCode, Function.identity()));
            }
        }

        // 产品
        if (!CollectionUtils.isEmpty(productCodeSet)) {
            List<ProductVo> productVoList = productVoService.findByCodes(new ArrayList<>(productCodeSet));;
            if (!CollectionUtils.isEmpty(productVoList)) {
                productVoMap = productVoList.stream().collect(Collectors.toMap(ProductVo::getProductCode, Function.identity()));
            }
        }
    }

    @Override
    public Class<SalesPlanHeadImportsVo> findCrmExcelVoClass() {
        return SalesPlanHeadImportsVo.class;
    }

    @Override
    public String getBusinessCode() {
        return "TPM_SALES_PLAN_IMPORT";
    }

    @Override
    public String getBusinessName() {
        return "TPM-销售计划导入";
    }

    @Override
    public String getTemplateCode() {
        return "TPM_SALES_PLAN_HEAD_IMPORT";
    }

    @Override
    public String getTemplateName() {
        return "TPM销售计划主体导入";
    }
}
