package com.biz.crm.tpm.business.scheme.forecast.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.tpm.business.scheme.forecast.sdk.constants.TpmVerticalSchemeForecastConstants;
import com.biz.crm.tpm.business.scheme.forecast.sdk.dto.SchemeForecastDto;
import com.biz.crm.tpm.business.scheme.forecast.sdk.dto.TpmVerticalSchemeForecastBudgetCashConfirmDto;
import com.biz.crm.tpm.business.scheme.forecast.sdk.dto.TpmVerticalSchemeForecastBudgetCashQueryDto;
import com.biz.crm.tpm.business.scheme.forecast.sdk.service.TpmVerticalSchemeForecastService;
import com.biz.crm.tpm.business.scheme.forecast.sdk.vo.SchemeForecastVo;
import com.biz.crm.tpm.business.scheme.forecast.sdk.vo.TpmVerticalSchemeForecastBudgetCashConfirmVo;
import com.biz.crm.tpm.business.scheme.forecast.sdk.vo.TpmVerticalSchemeForecastBudgetCashVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * <p>
 *
 * </p>
 *
 * @author chenshuang
 * @since 2023-06-09
 */
@RestController
@RequestMapping("/v1/vertical/scheme/forecast")
@Slf4j
@Api(tags = "垂直方案预测")
public class TpmVerticalSchemeForecastController {

    @Autowired(required = false)
    private TpmVerticalSchemeForecastService tpmVerticalSchemeForecastService;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    /**
     * 批量更新(重新计算预估可核销金额)
     *
     * @param idList 主键列表
     * @return 修改结果
     */
    @ApiOperation(value = "批量更新(重新计算预估可核销金额)")
    @PatchMapping("updateAuditAmount")
    public Result<?> updateAuditAmount(@ApiParam(name = "idList", value = "主键集合") @RequestBody List<String> idList) {
        try {
            this.tpmVerticalSchemeForecastService.updateAuditAmount(idList);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 删除数据
     *
     * @param idList 主键结合
     * @return 删除结果
     */
    @ApiOperation(value = "删除数据")
    @PostMapping("delete")
    public Result<?> delete(@ApiParam(name = "idList", value = "主键集合") @RequestBody List<String> idList) {
        Validate.notNull(idList, "请选择数据");
        List<String> lockKeys = new ArrayList<>();
        try {
            for (String id : idList) {
                String lockKey = TpmVerticalSchemeForecastConstants.data_operate_lock + id;
                boolean lock = redisLockService.tryLock(lockKey, TimeUnit.SECONDS, 60 * 60, 5);
                if (!lock) {
                    break;
                }
                lockKeys.add(lockKey);
            }
            Validate.isTrue(idList.size() == lockKeys.size(), "数据正在被其他人操作，请稍后...");
            this.tpmVerticalSchemeForecastService.delete(idList);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            lockKeys.forEach(lockKey -> {
                redisLockService.unlock(lockKey);
            });
        }
    }

    /**
     * 确认
     *
     * @param dto 主键列表
     * @return 确认结果
     */
    @ApiOperation(value = "确认")
    @PostMapping("confirm")
    public Result<?> confirm(@ApiParam(name = "dto", value = "TpmVerticalSchemeForecastBudgetCashConfirmDto") @RequestBody TpmVerticalSchemeForecastBudgetCashConfirmDto dto) {
        Validate.notBlank(dto.getId(), "主键id为空");
        String lockKey = TpmVerticalSchemeForecastConstants.data_operate_lock + dto.getId();
        boolean lock = false;
        try {
            lock = redisLockService.tryLock(lockKey, TimeUnit.SECONDS, 60 * 60, 5);
            Validate.isTrue(lock, "数据正在被其他人操作，请稍后...");
            this.tpmVerticalSchemeForecastService.confirm(dto);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (lock){
                redisLockService.unlock(lockKey);
            }
        }
    }

    /**
     * 确认
     *
     * @param dto 主键列表
     * @return 确认结果
     */
    @ApiOperation(value = "确认")
    @PostMapping("confirmBatch")
    public Result<?> confirmBatch(@ApiParam(name = "dto", value = "TpmVerticalSchemeForecastBudgetCashConfirmDto") @RequestBody TpmVerticalSchemeForecastBudgetCashConfirmDto dto) {
        Validate.isTrue(!CollectionUtils.isEmpty(dto.getIds()), "主键id为空");
        boolean lock = true;
        try {
            lock = redisLockService.batchLock(TpmVerticalSchemeForecastConstants.data_operate_lock,dto.getIds(), TimeUnit.SECONDS, 60 * 60);
            Validate.isTrue(lock, "数据正在被其他人操作，请稍后...");
            this.tpmVerticalSchemeForecastService.confirmBatch(dto);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (lock){
                redisLockService.batchUnLock(TpmVerticalSchemeForecastConstants.data_operate_lock,dto.getIds());
            }

        }
    }

    /**
     * 查询兑付详情
     *
     * @param id 主键结合
     * @return 查询结果
     */
    @ApiOperation(value = "查询兑付详情")
    @GetMapping("findConfirmDetail")
    public Result<TpmVerticalSchemeForecastBudgetCashConfirmVo> findConfirmDetail(@RequestParam(value = "id", required = false) String id,
                                                                                  @RequestParam(value = "code", required = false) String code) {
        try {
            return Result.ok(this.tpmVerticalSchemeForecastService.findConfirmDetail(id, code));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    @ApiOperation(value = "根据区域+零售商查兑付明细")
    @GetMapping("findCashPage")
    public Result<Page<TpmVerticalSchemeForecastBudgetCashVo>> findCashPage(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                            @ApiParam(name = "dto", value = "查询参数") TpmVerticalSchemeForecastBudgetCashQueryDto dto) {
        try {
            Page<TpmVerticalSchemeForecastBudgetCashVo> page = this.tpmVerticalSchemeForecastService.findCashPage(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }
}
