package com.biz.crm.tpm.business.scheme.forecast.local.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.common.rocketmq.service.RocketMqProducer;
import com.biz.crm.mn.common.rocketmq.util.RocketMqUtil;
import com.biz.crm.mn.common.rocketmq.vo.MqMessageVo;
import com.biz.crm.tpm.business.activity.plan.sdk.constant.ActivityPlanConstant;
import com.biz.crm.tpm.business.activity.plan.sdk.constant.ActivityPlanPassMqTagConstant;
import com.biz.crm.tpm.business.detailed.forecast.sdk.constant.DetailedPredictionTypeEnum;
import com.biz.crm.tpm.business.scheme.forecast.local.entity.SchemeForecastEntity;
import com.biz.crm.tpm.business.scheme.forecast.local.repository.SchemeForecastRepository;
import com.biz.crm.tpm.business.scheme.forecast.local.util.ActivityDetailPlanPassSchemeForecastUtil;
import com.biz.crm.tpm.business.scheme.forecast.sdk.constants.SchemeForecastConstants;
import com.biz.crm.tpm.business.scheme.forecast.sdk.dto.*;
import com.biz.crm.tpm.business.scheme.forecast.sdk.dto.log.SchemeForecastLogEventDto;
import com.biz.crm.tpm.business.scheme.forecast.sdk.enums.SchemePredictionTypeEnum;
import com.biz.crm.tpm.business.scheme.forecast.sdk.event.SchemeForecastLogEventListener;
import com.biz.crm.tpm.business.scheme.forecast.sdk.service.SchemeForecastService;
import com.biz.crm.tpm.business.scheme.forecast.sdk.vo.SchemeForecastVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 方案预测表(SchemeForecastEntity)表服务实现类
 *
 * @author youjun
 * @date 2022年11月09日 11:24
 */
@Slf4j
@Service("schemeForecastService")
public class SchemeForecastServiceImpl implements SchemeForecastService {

    @Resource
    private SchemeForecastRepository schemeForecastRepository;

    @Resource
    private NebulaToolkitService nebulaToolkitService;

    @Resource
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private RocketMqProducer rocketMqProducer;

    @Autowired(required = false)
    private RedisService redisService;

    @Autowired(required = false)
    private ActivityDetailPlanPassSchemeForecastUtil activityDetailPlanPassSchemeForecastUtil;

    /**
     * 分页查询数据
     *
     * @param pageable          分页对象
     * @param schemeForecastDto 实体对象
     * @return 方案预测单列表
     */
    @Override
    public Page<SchemeForecastVo> findByForecasts(Pageable pageable, SchemeForecastDto schemeForecastDto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(schemeForecastDto)) {
            schemeForecastDto = new SchemeForecastDto();
        }
        return this.schemeForecastRepository.findByForecasts(pageable, schemeForecastDto);
    }

    /**
     * 新增数据
     *
     * @param schemeForecastDto 实体对象
     * @return 新增结果
     */
    @Override
    public void create(SchemeForecastDto schemeForecastDto) {
        SchemeForecastEntity schemeForecastEntity = this.nebulaToolkitService.copyObjectByWhiteList(schemeForecastDto, SchemeForecastEntity.class, null, null);
        schemeForecastEntity.setTenantCode(TenantUtils.getTenantCode());
        schemeForecastEntity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        schemeForecastEntity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());

        // 日志新增
        SchemeForecastLogEventDto logEventDto = new SchemeForecastLogEventDto();
        logEventDto.setOriginal(null);
        logEventDto.setNewest(schemeForecastDto);
        SerializableBiConsumer<SchemeForecastLogEventListener, SchemeForecastLogEventDto> onCreate =
                SchemeForecastLogEventListener::onCreate;
        this.nebulaNetEventClient.publish(logEventDto, SchemeForecastLogEventListener.class, onCreate);
        this.schemeForecastRepository.saveOrUpdate(schemeForecastEntity);
    }

    /**
     * 修改数据
     *
     * @param schemeForecastDto 实体对象
     * @return 修改结果
     */
    @Override
    public void update(SchemeForecastDto schemeForecastDto) {
        SchemeForecastVo schemeForecastVo = this.findById(schemeForecastDto.getId());
        if (ObjectUtils.isEmpty(schemeForecastVo)) {
            throw new RuntimeException("数据不存在，请刷新后重试！");
        }
        SchemeForecastEntity schemeForecastEntity = this.nebulaToolkitService.copyObjectByWhiteList(schemeForecastVo, SchemeForecastEntity.class, null, null);
        schemeForecastEntity.setEstimatedWriteOffAmount(schemeForecastDto.getEstimatedWriteOffAmount());
        schemeForecastEntity.setActualWriteOffAmount(schemeForecastDto.getActualWriteOffAmount());
        this.schemeForecastRepository.saveOrUpdate(schemeForecastEntity);
        // 日志新增
        SchemeForecastDto dto = this.nebulaToolkitService.copyObjectByWhiteList(schemeForecastVo, SchemeForecastDto.class, LinkedHashSet.class, ArrayList.class);
        SchemeForecastLogEventDto logEventDto = new SchemeForecastLogEventDto();
        logEventDto.setOriginal(dto);
        logEventDto.setNewest(schemeForecastDto);
        SerializableBiConsumer<SchemeForecastLogEventListener, SchemeForecastLogEventDto> onUpdate =
                SchemeForecastLogEventListener::onUpdate;
        this.nebulaNetEventClient.publish(logEventDto, SchemeForecastLogEventListener.class, onUpdate);
    }

    /**
     * 删除数据
     *
     * @param idList 主键结合
     * @return 删除结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<String> idList) {
        Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
        List<SchemeForecastEntity> list = this.schemeForecastRepository.findByIds(idList);
        list.forEach(v -> {
            Validate.isTrue(!StringUtils.equals(DetailedPredictionTypeEnum.CONFIRMED.getDictCode(), v.getDetailedPredictionType()),
                    "方案明细编码[%s]已确认！", v.getSchemeItemCode());
            v.setOnlyKey(this.activityDetailPlanPassSchemeForecastUtil.packageOnlyKey(v, DateUtil.getDate(DateUtil.DEFAULT_DATE_ALL_PATTERN)));
        });
        this.schemeForecastRepository.removeByIds(idList);
    }

    /**
     * 通过主键查询单条数据
     *
     * @param id 主键
     * @return 单条数据
     */
    @Override
    public SchemeForecastVo findById(String id) {
        SchemeForecastEntity schemeForecastEntity = this.schemeForecastRepository.getById(id);
        if (schemeForecastEntity == null) {
            return null;
        }
        return nebulaToolkitService.copyObjectByWhiteList(schemeForecastEntity, SchemeForecastVo.class, null, null);
    }

    /**
     * 通过id批量查询
     *
     * @param idList
     * @return
     */
    @Override
    public List<SchemeForecastVo> findByIds(List<String> idList) {
        if (CollectionUtils.isEmpty(idList)) {
            return Lists.newArrayList();
        }
        List<SchemeForecastEntity> entityList = this.schemeForecastRepository.findByIds(idList);
        return (List<SchemeForecastVo>) nebulaToolkitService.copyCollectionByWhiteList(entityList, SchemeForecastEntity.class, SchemeForecastVo.class, HashSet.class, ArrayList.class);
    }

    /**
     * 更新不覆盖数据
     *
     * @param schemeForecastDto 实体对象
     * @return 更新数据
     */
    @Override
    public void updateDetailedForecast(SchemeForecastDto schemeForecastDto) {
        SchemeForecastEntity schemeForecastEntity = this.nebulaToolkitService.copyObjectByWhiteList(schemeForecastDto, SchemeForecastEntity.class, null, null);
        schemeForecastEntity.setTenantCode(TenantUtils.getTenantCode());
        this.schemeForecastRepository.save(schemeForecastEntity);
    }

    /**
     * 批量导入新增
     *
     * @param importList
     * @return 批量导入新增
     */
    @Override
    public void importSave(List<SchemeForecastDto> importList) {
        if (CollectionUtils.isEmpty(importList)) {
            return;
        }
        importList.forEach(dto -> {
            dto.setTenantCode(TenantUtils.getTenantCode());
            dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            dto.setId(UUID.randomUUID().toString().replaceAll("-", ""));
        });
        // 保存方案预测
        Collection<SchemeForecastEntity> schemeForecastEntity = this.nebulaToolkitService.copyCollectionByWhiteList(importList, SchemeForecastDto.class, SchemeForecastEntity.class, LinkedHashSet.class, ArrayList.class);
        this.schemeForecastRepository.saveBatch(new ArrayList<>(schemeForecastEntity));
        // 业务日志创建
        importList.forEach(dto -> {
            SchemeForecastLogEventDto logEventDto = new SchemeForecastLogEventDto();
            logEventDto.setOriginal(null);
            logEventDto.setNewest(dto);
            SerializableBiConsumer<SchemeForecastLogEventListener, SchemeForecastLogEventDto> onCreate =
                    SchemeForecastLogEventListener::onCreate;
            this.nebulaNetEventClient.publish(logEventDto, SchemeForecastLogEventListener.class, onCreate);
        });
    }

    @Override
    public void updateAuditAmount(List<String> idList) {
        Validate.notEmpty(idList, "请求入参不能为空,请检查");
        List<SchemeForecastEntity> forecastEntities = this.schemeForecastRepository.lambdaQuery()
                .in(SchemeForecastEntity::getId, idList).list();
        Validate.notEmpty(forecastEntities, "不存在的实例,请检查数据是否存在");
        forecastEntities.forEach(forecastEntity -> {
            Validate.isTrue(!StringUtils.equals(SchemePredictionTypeEnum.CONFIRMED.getDictCode(), forecastEntity.getDetailedPredictionType()),
                    "方案明细编码[%s]已确认！", forecastEntity.getSchemeItemCode());
        });
        MqMessageVo mqMessageVo = new MqMessageVo();
        String uuid = UUID.randomUUID().toString().replace("-", "");
        List<String> ids = forecastEntities.stream().map(SchemeForecastEntity::getId).distinct().collect(Collectors.toList());
        redisService.hSet(SchemeForecastConstants.REFRESH_IDS_KEY, uuid, JSONObject.toJSONString(ids), 60 * 60 * 48);//定时任务每天跑，缓存2天够了
        mqMessageVo.setMsgBody(uuid);
        mqMessageVo.setTopic(ActivityPlanConstant.TPM_ACTIVITY_PLAN_PROCESS_PASS_TOPIC + RocketMqUtil.mqEnvironment());
        mqMessageVo.setTag(ActivityPlanPassMqTagConstant.SCHEME_FORECAST_REFRESH);
        rocketMqProducer.sendMqMsg(mqMessageVo);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void confirm(List<String> idList) {
        Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
        List<SchemeForecastEntity> list = this.schemeForecastRepository.lambdaQuery()
                .in(SchemeForecastEntity::getId, idList).list();
        list.forEach(v -> {
            Validate.isTrue(!StringUtils.equals(DetailedPredictionTypeEnum.CONFIRMED.getDictCode(), v.getDetailedPredictionType()),
                    "方案明细编码[%s]已确认！", v.getSchemeItemCode());
            v.setDetailedPredictionType(DetailedPredictionTypeEnum.CONFIRMED.getDictCode());
        });

        List<SchemeForecastDto> confirmList = new ArrayList<>();
        for (SchemeForecastEntity entity : list) {
            if (StringUtils.isBlank(entity.getSchemeItemCode())) {
                continue;
            }
            SchemeForecastDto confirmDto = new SchemeForecastDto();
            confirmDto.setSchemeItemCode(entity.getSchemeItemCode());
            confirmDto.setActualWriteOffAmount(Optional.ofNullable(entity.getActualWriteOffAmount()).orElse(BigDecimal.ZERO));
            //补扣金额 confimDto.set....
            confirmList.add(confirmDto);
        }

        this.schemeForecastRepository.updateBatchById(list);
    }

    /**
     * 导入更新保存
     *
     * @param dtoList
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void importUpdate(List<SchemeForecastDto> dtoList) {
        this.schemeForecastRepository.updateBatchById(nebulaToolkitService.copyCollectionByWhiteList(dtoList, SchemeForecastDto.class, SchemeForecastEntity.class, HashSet.class, ArrayList.class));
    }

    /**
     * 根据方案编码查询
     *
     * @param codes
     * @return
     */
    @Override
    public Collection<SchemeForecastVo> findByCodes(Set<String> codes) {
        if (CollectionUtils.isEmpty(codes)){
            return new ArrayList<>();
        }
        List<SchemeForecastEntity> byCodes = schemeForecastRepository.findByCodes(codes);
        return nebulaToolkitService.copyCollectionByWhiteList(byCodes, SchemeForecastEntity.class, SchemeForecastVo.class, HashSet.class, ArrayList.class);
    }

    @Override
    public Page<String> findHeadVerticalCodeList(Page<String> page, SchemePushSchemeForecastDto dto) {
        return schemeForecastRepository.findHeadVerticalCodeList(page, dto);
    }

    @Override
    public Long findAutoRefreshCount(SchemeForecastAutoRefreshDto dto) {
        dto.setTenantCode(TenantUtils.getTenantCode());
        Validate.notBlank(dto.getTenantCode(), "未获取到租户号！");
        return schemeForecastRepository.findAutoRefreshDataCount(dto);
    }

    @Override
    public List<String> findAutoRefreshDataList(Pageable pageable, SchemeForecastAutoRefreshDto dto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        dto.setTenantCode(TenantUtils.getTenantCode());
        Validate.notBlank(dto.getTenantCode(), "未获取到租户号！");
        return schemeForecastRepository.findAutoRefreshDataList(pageable, dto);
    }

    @Override
    public List<String> findAutoCreateDataList(SchemeForecastAutoCreateDto dto) {
        dto.setTenantCode(TenantUtils.getTenantCode());
        Validate.notBlank(dto.getTenantCode(), "未获取到租户号！");
        return schemeForecastRepository.findAutoCreateDataList(dto);
    }

    @Transactional
    @Override
    public Long autoUpdateShowFLag(SchemeForecastAutoCreateDto dto) {
        dto.setTenantCode(TenantUtils.getTenantCode());
        Validate.notBlank(dto.getTenantCode(), "未获取到租户号！");
        List<String> schemeItemCodes = schemeForecastRepository.findAutoUpdateShowFlagDataList(dto);
        if (CollectionUtils.isEmpty(schemeItemCodes)) {
            return 0L;
        }
        return schemeForecastRepository.autoUpdateShowFLag(schemeItemCodes);
    }

    @Override
    public List<SchemeForecastVo> findByPlanItemCodes(Set<String> planItemCodes) {
        if(CollectionUtils.isEmpty(planItemCodes)){
            return Lists.newArrayList();
        }
        List<SchemeForecastEntity> schemeForecastEntities = schemeForecastRepository.findByPlanItemCodes(planItemCodes);
        if(!CollectionUtils.isEmpty(schemeForecastEntities)){
            Collection<SchemeForecastVo> schemeForecastVos = this.nebulaToolkitService.copyCollectionByWhiteList(schemeForecastEntities, SchemeForecastEntity.class, SchemeForecastVo.class, LinkedHashSet.class, ArrayList.class);
            return (List<SchemeForecastVo>) schemeForecastVos;
        }
        return Lists.newArrayList();
    }
}
