package com.biz.crm.tpm.business.scheme.forecast.local.util;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.tpm.business.month.budget.sdk.dto.OperateMonthBudgetDto;
import com.biz.crm.tpm.business.month.budget.sdk.eunm.BudgetOperationTypeEnum;
import com.biz.crm.tpm.business.month.budget.sdk.service.MonthBudgetLockService;
import com.biz.crm.tpm.business.month.budget.sdk.service.MonthBudgetService;
import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetVo;
import com.biz.crm.tpm.business.scheme.forecast.local.entity.TpmVerticalSchemeForecastBudgetCashEntity;
import com.biz.crm.tpm.business.scheme.forecast.sdk.constants.TpmVerticalSchemeForecastConstants;
import com.biz.crm.tpm.business.scheme.forecast.sdk.dto.TpmVerticalSchemeForecastBudgetCashConfirmDto;
import com.biz.crm.tpm.business.scheme.forecast.sdk.dto.TpmVerticalSchemeForecastBudgetCashDto;
import com.biz.crm.tpm.business.scheme.forecast.sdk.vo.TpmVerticalSchemeForecastBudgetCashConfirmVo;
import com.biz.crm.tpm.business.scheme.forecast.sdk.vo.TpmVerticalSchemeForecastBudgetCashVo;
import com.biz.crm.tpm.business.scheme.forecast.sdk.vo.TpmVerticalSchemeForecastVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * <p>
 *
 * </p>
 *
 * @author chenshuang
 * @since 2023-06-09
 */
@Slf4j
@Component
public class TpmVerticalSchemeForecastUtil {

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private MonthBudgetLockService monthBudgetLockService;

    @Autowired(required = false)
    private MonthBudgetService monthBudgetService;

    @Autowired(required = false)
    private GenerateCodeService generateCodeService;

    /**
     * 验证确认数据
     *
     * @param dto
     */
    public void validateConfirmData(TpmVerticalSchemeForecastBudgetCashConfirmDto dto) {
        Validate.notNull(dto.getId(), "主键id不能为空");
        Validate.notNull(dto.getEstimatedWriteOffAmount(), "预估核销金额不能为空");
        Validate.isTrue(dto.getEstimatedWriteOffAmount().compareTo(BigDecimal.ZERO) >= 0, "预估核销金额不能为负数");
        Validate.notEmpty(dto.getBudgetCashDtoList(), "兑付明细不能为空");
        for (TpmVerticalSchemeForecastBudgetCashDto cashDto : dto.getBudgetCashDtoList()) {
            Validate.notBlank(cashDto.getBudgetCode(), "预算编码不能为空");
            Validate.notNull(cashDto.getConfirmAmount(), "确认金额不能为空");
            Validate.isTrue(cashDto.getConfirmAmount().compareTo(BigDecimal.ZERO) >= 0, "确认金额不能为负数");
        }
    }

    /**
     * 构建兑付明细
     *
     * @param dto
     * @param vo
     */
    public List<TpmVerticalSchemeForecastBudgetCashEntity> buildBudgetCashEntityList(TpmVerticalSchemeForecastBudgetCashConfirmDto dto,
                                                                                     TpmVerticalSchemeForecastBudgetCashConfirmVo vo) {
        Map<String, TpmVerticalSchemeForecastBudgetCashDto> dtoMap = dto.getBudgetCashDtoList().stream()
                .collect(Collectors.toMap(TpmVerticalSchemeForecastBudgetCashDto::getBudgetCode, v -> v, (v1, v2) -> v2));
        Map<String, TpmVerticalSchemeForecastBudgetCashVo> voMap = vo.getBudgetCashVoList().stream()
                .collect(Collectors.toMap(TpmVerticalSchemeForecastBudgetCashVo::getBudgetCode, v -> v, (v1, v2) -> v2));

        TpmVerticalSchemeForecastVo forecastVo = vo.getForecastVo();
        List<TpmVerticalSchemeForecastBudgetCashEntity> budgetCashEntityList = new ArrayList<>();

        dtoMap.forEach((k, v) -> {
            Validate.isTrue(voMap.containsKey(k), "预算编码[%s]不属于当前预测明细", k);

            TpmVerticalSchemeForecastBudgetCashEntity entity = nebulaToolkitService.copyObjectByWhiteList(voMap.get(k), TpmVerticalSchemeForecastBudgetCashEntity.class, HashSet.class, ArrayList.class);
            if (StringUtils.isEmpty(entity.getDetailCode())) {
                String detailCode = generateCodeService.generateCode(TpmVerticalSchemeForecastConstants.FORECAST_CASH_CODE_PREFIX, 1, 6, 2, TimeUnit.DAYS).get(0);
                entity.setDetailCode(detailCode);
            }
            entity.setSchemeForecastId(forecastVo.getId());
            entity.setSchemeForecastCode(forecastVo.getSchemeForecastCode());
            entity.setSchemeCode(forecastVo.getSchemeCode());
            entity.setSchemeItemCode(forecastVo.getSchemeItemCode());
            entity.setYearMonthLy(forecastVo.getYearMonthLy());

            entity.setConfirmAmount(v.getConfirmAmount());
            entity.setUsedAmount(BigDecimal.ZERO);
            entity.setBalanceAmount(entity.getConfirmAmount());
            entity.setOverFrozenAmount(BigDecimal.ZERO);
            if (entity.getConfirmAmount().compareTo(entity.getActFrozenAmount()) > 0) {
                entity.setOverFrozenAmount(entity.getConfirmAmount().subtract(entity.getActFrozenAmount()));
            }
            entity.setTenantCode(TenantUtils.getTenantCode());
            entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            budgetCashEntityList.add(entity);
        });

        return budgetCashEntityList;
    }

    /**
     * 冻结或解冻预算
     *
     * @param forecastVo
     * @param budgetCashEntityList
     * @param operationTypeEnum
     * @param validate             是否只校验
     */
    @Transactional(rollbackFor = Exception.class)
    public void unOrFrozenBudget(TpmVerticalSchemeForecastVo forecastVo,
                                 List<TpmVerticalSchemeForecastBudgetCashEntity> budgetCashEntityList,
                                 BudgetOperationTypeEnum operationTypeEnum,
                                 boolean validate) {
        if (CollectionUtils.isEmpty(budgetCashEntityList)) {
            log.info("垂直大区方案兑付：无预算兑付明细");
            return;
        }

        List<String> budgetCodeList = budgetCashEntityList.stream().filter(e -> Objects.nonNull(e.getOverFrozenAmount()) && BigDecimal.ZERO.compareTo(e.getOverFrozenAmount()) < 0)
                .map(TpmVerticalSchemeForecastBudgetCashEntity::getBudgetCode).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(budgetCodeList)) {
            log.info("垂直大区方案兑付：无需操作预算");
            return;
        }

        Validate.isTrue(StringUtils.equals(operationTypeEnum.getCode(), BudgetOperationTypeEnum.FREEZE.getCode())
                || StringUtils.equals(operationTypeEnum.getCode(), BudgetOperationTypeEnum.UNFREEZE.getCode()), "预算操作类型错误");

        boolean locked = monthBudgetLockService.lock(budgetCodeList, TimeUnit.SECONDS, 60 * 2, 5);
        Validate.isTrue(locked, "操作预算失败，获取操作锁失败！");
        try {
            List<MonthBudgetVo> budgetVosList = monthBudgetService.findByCodes(budgetCodeList, EnableStatusEnum.ENABLE.getCode());
            Validate.notEmpty(budgetVosList, "预算不存在或已禁用");
            Map<String, MonthBudgetVo> budgetVoMap = budgetVosList.stream().collect(Collectors.toMap(MonthBudgetVo::getMonthBudgetCode, v -> v, (v1, v2) -> v1));

            List<OperateMonthBudgetDto> operateDtoList = new ArrayList<>();
            for (TpmVerticalSchemeForecastBudgetCashEntity cashEntity : budgetCashEntityList) {
                Validate.isTrue(budgetVoMap.containsKey(cashEntity.getBudgetCode()), "预算[%s]不存在或已禁用", cashEntity.getBudgetCode());

                MonthBudgetVo monthBudgetVo = budgetVoMap.get(cashEntity.getBudgetCode());
                BigDecimal accumulatedAvailableBalance = monthBudgetVo.getAccumulatedAvailableBalance();
                BigDecimal freezeAmount = monthBudgetVo.getFreezeAmount();


                if (StringUtils.equals(operationTypeEnum.getCode(), BudgetOperationTypeEnum.FREEZE.getCode())) {
                    //冻结
                    Validate.isTrue(accumulatedAvailableBalance.compareTo(cashEntity.getOverFrozenAmount()) >= 0, "预算[%s]余额不足，当前余额：%s", cashEntity.getBudgetCode(), accumulatedAvailableBalance);
                } else if (StringUtils.equals(operationTypeEnum.getCode(), BudgetOperationTypeEnum.UNFREEZE.getCode())) {
                    //解冻
                    Validate.isTrue(freezeAmount.compareTo(cashEntity.getOverFrozenAmount()) >= 0, "预算[%s]冻结金额不足，当前冻结：%s", cashEntity.getBudgetCode(), freezeAmount);
                }
                OperateMonthBudgetDto operateMonthBudgetDto = new OperateMonthBudgetDto();
                operateMonthBudgetDto.setMonthBudgetCode(cashEntity.getBudgetCode());
                operateMonthBudgetDto.setBusinessCode(forecastVo.getSchemeForecastCode());
                operateMonthBudgetDto.setOperationAmount(cashEntity.getOverFrozenAmount());
                operateMonthBudgetDto.setOperationType(operationTypeEnum.getCode());
                operateDtoList.add(operateMonthBudgetDto);
            }
            if (!validate) {
                monthBudgetService.operateBudget(operateDtoList);
            }
        } finally {
            monthBudgetLockService.unLock(budgetCodeList);
        }
    }
}
