package com.biz.crm.tpm.business.settlement.manage.local.service.imports;

import com.aliyun.openservices.shade.com.google.common.collect.Sets;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mn.common.base.util.DateStringDealUtil;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.platform.expense.form.sdk.dto.PlatformExpenseFormDto;
import com.biz.crm.tpm.business.platform.expense.form.sdk.service.PlatformExpenseFormService;
import com.biz.crm.tpm.business.platform.product.relation.sdk.dto.PlatformProductRelationDto;
import com.biz.crm.tpm.business.platform.product.relation.sdk.service.PlatformProductRelationService;
import com.biz.crm.tpm.business.platform.product.relation.sdk.vo.PlatformProductRelationVo;
import com.biz.crm.tpm.business.settlement.manage.local.entity.TpmSettlementManage;
import com.biz.crm.tpm.business.settlement.manage.local.repository.TpmSettlementManageRepository;
import com.biz.crm.tpm.business.settlement.manage.sdk.constant.SettlementManageConstant;
import com.biz.crm.tpm.business.settlement.manage.sdk.enums.TpmSettlementSourceEnum;
import com.biz.crm.tpm.business.settlement.manage.sdk.vo.imports.TmallSettlementImportsVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * TPM结算单管理-猫超导入
 *
 * @author duyiran
 * @create 2022-11-02 16:45
 */
@Slf4j
@Component
public class TmallSettlementImportsProcess implements ImportProcess<TmallSettlementImportsVo> {

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private PlatformExpenseFormService platformExpenseFormService;

    @Autowired(required = false)
    private DictToolkitService dictToolkitService;

    @Autowired(required = false)
    PlatformProductRelationService platformProductRelationService;

    @Autowired(required = false)
    private TpmSettlementManageRepository tpmSettlementManageRepository;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    @Override
    public Integer getBatchCount() {
        return 10000;
    }

    /**
     * 是否开启先校验后导入的模式 默认false: false执行旧逻辑 true执行新逻辑(先校验再保存) <br/>
     * 新逻辑需同时实现接口 tryVerify tryConfirm（默认方法调用execute）
     *
     * @return
     */
    @Override
    public boolean importBeforeValidationFlag() {
        return Boolean.TRUE;
    }

    /**
     * 数据校验
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> tryVerify(LinkedHashMap<Integer, TmallSettlementImportsVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Validate.notNull(data, "导入数据不能为空！");
        Validate.isTrue(data.values().size() <= getBatchCount(), "单次导入数据不能超过%s条", getBatchCount());
        Map<Integer, String> errMap = new HashMap<>();
        for (Map.Entry<Integer, TmallSettlementImportsVo> row : data.entrySet()) {
            int rowNum = row.getKey();
            TmallSettlementImportsVo vo = row.getValue();
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getCode()), "业务主单据编码不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getSubCode()), "业务子单据编码不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getOrderType()), "业务单据类型不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getPlatformProductCode()), "后端商品编码不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getPlatformId()), "唯一ID不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getExpenseItem()), "费用类型不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getAmountStr()), "含税金额不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getMdgCode()), "客户编码不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getSalesOrgCode()), "销售机构编码不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getChannelCode()), "渠道编码不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getBusinessFormatCode()), "业态不能为空！");
            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(rowNum, errInfo);
            }
        }
        return errMap;
    }

    /**
     * 数据处理
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, TmallSettlementImportsVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Map<Integer, String> errMap = new HashMap<>();
        try {
            this.baseValidate(data, errMap);
            List<TpmSettlementManage> importList = this.validateRepeatAndSave(data, errMap);
            if (errMap.isEmpty()) {
                tpmSettlementManageRepository.bulkPreservation(importList);
            }
        } catch (Exception e) {
            log.error(e.getMessage());
            throw e;
        }
        return errMap;
    }

    /**
     * 校验数据
     *
     * @param data data
     * @author duyiran
     * @date 2022-10-29 15:04
     **/
    private void baseValidate(LinkedHashMap<Integer, TmallSettlementImportsVo> data, Map<Integer, String> errMap) {
        Map<String, String> mdmBusinessFormatMap = new HashMap<>();
        Map<String, CustomerVo> customerVoMap = new HashMap<>();
        Map<String, PlatformProductRelationVo> platformProductRelationVoMap = new HashMap<>();
        Map<String, PlatformExpenseFormDto> expenseFormDtoMap = new HashMap<>();
        mdmBusinessFormatMap = dictToolkitService.findConvertMapByDictTypeCode(SettlementManageConstant.MDM_BUSINESS_FORMAT);
        Set<String> customerCodeSet = Sets.newHashSet();
        Set<String> expenseItemSet = Sets.newHashSet();
        Set<String> productCodeSet = Sets.newHashSet();
        for (Map.Entry<Integer, TmallSettlementImportsVo> row : data.entrySet()) {
            productCodeSet.add(row.getValue().getPlatformProductCode());
            expenseItemSet.add(row.getValue().getExpenseItem());
            String customerCode = "";
            String mdgCode = row.getValue().getMdgCode();
            customerCode += mdgCode;
            String salesOrgCode = row.getValue().getSalesOrgCode();
            customerCode += salesOrgCode;
            String channelCode = row.getValue().getChannelCode();
            customerCode += channelCode;
            String businessFormatCode = row.getValue().getBusinessFormatCode();
            // 业态
            if (!StringUtils.isEmpty(businessFormatCode)) {
                if (!mdmBusinessFormatMap.containsKey(businessFormatCode)) {
                    this.validateIsTrue(false, "业态" + businessFormatCode + "错误，请检查！");
                } else {
                    row.getValue().setBusinessFormatCode(mdmBusinessFormatMap.get(businessFormatCode));
                }
            }
            customerCode += row.getValue().getBusinessFormatCode();
            customerCodeSet.add(customerCode);
            row.getValue().setCustomerCode(customerCode);
            try {
                row.getValue().setAmount(new BigDecimal(row.getValue().getAmountStr()));
            } catch (Exception e) {
                log.error("导入报错:{}", e.getMessage());
                this.validateIsTrue(false, "含税金额为空或格式错误");
            }
            try {
                row.getValue().setQuantity(new Integer(row.getValue().getQuantityStr()));
                row.getValue().setPrice(new BigDecimal(row.getValue().getPriceStr()));
            } catch (Exception ignore) {}
            try {
                DateStringDealUtil.validateDateStrAndSet(row.getValue().getBusinessTimeStr(), "业务发生时间", true, DateUtil.DEFAULT_DATE_ALL_PATTERN, row.getValue()::setBusinessTime);
            } catch (Exception e) {
                this.validateIsTrue(false, e.getMessage());
            }
            row.getValue().setYearMonthStr(DateUtil.dateToStr(DateUtil.date_yyyy_MM, row.getValue().getBusinessTime()));

            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(row.getKey(), errInfo);
            }
        }
        //客户
        if (!CollectionUtils.isEmpty(customerCodeSet)) {
            List<CustomerVo> customerVos = customerVoService.findByCustomerCodes(new ArrayList<>(customerCodeSet));
            if (!CollectionUtils.isEmpty(customerVos)) {
                customerVoMap = customerVos.stream().collect(Collectors.toMap(CustomerVo::getCustomerCode, Function.identity()));
            }
        }
        // 产品
        if (!CollectionUtils.isEmpty(productCodeSet) && !CollectionUtils.isEmpty(customerCodeSet)) {
            PlatformProductRelationDto platformProductRelationDto = new PlatformProductRelationDto();
            platformProductRelationDto.setPlatformProductCodeSet(productCodeSet);
            platformProductRelationDto.setCustomerCodeSet(customerCodeSet);
            List<PlatformProductRelationVo> platformVoList = platformProductRelationService.findListByConditions(platformProductRelationDto);
            if (!CollectionUtils.isEmpty(platformVoList)) {
                platformProductRelationVoMap = platformVoList.stream().collect(Collectors.toMap(e -> e.getCustomerCode() + e.getPlatformProductCode(), Function.identity(), (e1, e2) -> e1));
            }
        }
        // 费用项
        if (!CollectionUtils.isEmpty(productCodeSet)) {
            List<PlatformExpenseFormDto> expenseFormDtoList = platformExpenseFormService.findListByExpenseNameSet(expenseItemSet);
            if (!CollectionUtils.isEmpty(expenseFormDtoList)) {
                expenseFormDtoMap = expenseFormDtoList.stream().collect(Collectors.toMap(PlatformExpenseFormDto::getExpensesName, Function.identity(), (e1, e2) -> e1));
            }
        }
        for (Map.Entry<Integer, TmallSettlementImportsVo> row : data.entrySet()) {
            CustomerVo customerVo = customerVoMap.get(row.getValue().getCustomerCode());
            PlatformProductRelationVo productVo = platformProductRelationVoMap.get(row.getValue().getCustomerCode() + row.getValue().getPlatformProductCode());
            PlatformExpenseFormDto platformExpenseFormDto = expenseFormDtoMap.get(row.getValue().getExpenseItem());
            if (customerVo != null) {
                row.getValue().setCustomerName(customerVo.getCustomerName());
                row.getValue().setSalesOrgCode(customerVo.getSalesInstitutionErpCode());
                row.getValue().setSalesOrgName(customerVo.getSalesInstitutionName());
                row.getValue().setChannelCode(customerVo.getCustomerChannelCode());
                row.getValue().setChannelName(customerVo.getCustomerChannelName());
            } else {
                this.validateIsTrue(false, "根据MDG客户编码+销售机构编码+渠道编码+业态编码未找到对应客户！");
            }
            if (productVo != null) {
                row.getValue().setProductName(productVo.getProductName());
                row.getValue().setProductCode(productVo.getProductCode());
                row.getValue().setPlatformProductCode(productVo.getPlatformProductCode());
                row.getValue().setPlatformProductName(productVo.getPlatformProductName());
            } else {
                this.validateIsTrue(false, "根据后端商品编码未找到对应产品！");
            }
            if (platformExpenseFormDto != null) {
                row.getValue().setActivityFormCode(platformExpenseFormDto.getActivityFormCode());
                row.getValue().setActivityFormName(platformExpenseFormDto.getActivityFormName());
                row.getValue().setSourceCode(TpmSettlementSourceEnum.T_MALL.getCode());
            } else {
                this.validateIsTrue(false, "根据费用类型未找到对应活动形式！");
            }

            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                if (errMap.containsKey(row.getKey())) {
                    errMap.put(row.getKey(), errMap.get(row.getKey()) + "," + errInfo);
                } else {
                    errMap.put(row.getKey(), errInfo);
                }
            }
        }
    }

    /**
     * 验重，成功后保存
     *
     * @param data data
     */
    private List<TpmSettlementManage> validateRepeatAndSave(LinkedHashMap<Integer, TmallSettlementImportsVo> data, Map<Integer, String> errMap) {
        Map<String, Integer> repeatKeyToIndexMap = Maps.newHashMap();
        data.forEach((index, importsVo) -> {
            //行内验重 repeatKey
            String key = TpmSettlementSourceEnum.T_MALL.getCode() + importsVo.getPlatformId();
            if (repeatKeyToIndexMap.containsKey(key)) {
                Integer lastIndex = repeatKeyToIndexMap.get(key);
                this.validateIsTrue(false, "该行唯一ID与第【" + lastIndex + "】行重复");
            } else {
                repeatKeyToIndexMap.put(key, index);
                importsVo.setVerifyId(key);
            }

            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                if (errMap.containsKey(index)) {
                    errMap.put(index, errMap.get(index) + "," + errInfo);
                } else {
                    errMap.put(index, errInfo);
                }
            }
        });
        List<TpmSettlementManage> importList = (List<TpmSettlementManage>) nebulaToolkitService.copyCollectionByWhiteList(data.values(), TmallSettlementImportsVo.class, TpmSettlementManage.class, LinkedHashSet.class, ArrayList.class);
        List<String> repeatKeyList = new ArrayList<>(repeatKeyToIndexMap.keySet());
        boolean isLock = redisLockService.batchLock(SettlementManageConstant.SETTLEMENT_MANAGE_LOCK, repeatKeyList, TimeUnit.MINUTES, SettlementManageConstant.DEFAULT_LOCK_TIME);
        Validate.isTrue(isLock, "加锁失败");
        try {
            //数据库验重
            Set<String> repeatSet = tpmSettlementManageRepository.repeatValidateBatch(repeatKeyList);
            if (CollectionUtils.isNotEmpty(repeatSet)) {
                data.forEach((index, importsVo) -> {
                    //数据库验重
                    if (repeatSet.contains(importsVo.getVerifyId())) {
                        this.validateIsTrue(false, "唯一ID已有现存数据，请重新填写！");
                    }

                    String errInfo = this.validateGetErrorInfo();
                    if (errInfo != null) {
                        if (errMap.containsKey(index)) {
                            errMap.put(index, errMap.get(index) + "," + errInfo);
                        } else {
                            errMap.put(index, errInfo);
                        }
                    }
                });
            }
        } finally {
            redisLockService.batchUnLock(SettlementManageConstant.SETTLEMENT_MANAGE_LOCK, repeatKeyList);
        }
        return importList;
    }

    /**
     * 获取数据实体
     *
     * @return
     */
    @Override
    public Class<TmallSettlementImportsVo> findCrmExcelVoClass() {
        return TmallSettlementImportsVo.class;
    }

    /**
     * 获取业务对应的模板编码，全局唯一
     *
     * @return
     */
    @Override
    public String getTemplateCode() {
        return "TPM_SETTLEMENT_TMALL_IMPORT";
    }

    /**
     * 获取业务对应的模板描述
     *
     * @return
     */
    @Override
    public String getTemplateName() {
        return "TPM结算单管理-猫超导入";
    }
}
