package com.biz.crm.tpm.business.audit.summary.configure.local.process.imports;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.channel.sdk.service.CustomerChannelVoService;
import com.biz.crm.mdm.business.customer.channel.sdk.vo.CustomerChannelVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mn.common.base.constant.CommonConstant;
import com.biz.crm.tpm.business.audit.summary.configure.local.entity.TpmCustomerSummaryConfigureEntity;
import com.biz.crm.tpm.business.audit.summary.configure.local.repository.TpmCustomerSummaryConfigureRepository;
import com.biz.crm.tpm.business.audit.summary.configure.sdk.dto.TpmCustomerSummaryConfigureImportDto;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 *
 * </p>
 *
 * @author chenshuang
 * @since 2023-04-03
 */
@Slf4j
@Component
public class TpmCustomerSummaryConfigureImportProcess implements ImportProcess<TpmCustomerSummaryConfigureImportDto> {

    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private TpmCustomerSummaryConfigureRepository tpmCustomerSummaryConfigureRepository;

    @Autowired(required = false)
    private CustomerChannelVoService customerChannelVoService;

    @Autowired(required = false)
    private DictToolkitService dictToolkitService;

    @Override
    public Integer getBatchCount() {
        return Integer.MAX_VALUE;
    }

    /**
     * 是否开启先校验后导入的模式 默认false: false执行旧逻辑 true执行新逻辑(先校验再保存) <br/>
     * 新逻辑需同时实现接口 tryVerify tryConfirm（默认方法调用execute）
     *
     * @return
     */
    @Override
    public boolean importBeforeValidationFlag() {
        return Boolean.TRUE;
    }

    /**
     * 数据校验
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> tryVerify(LinkedHashMap<Integer, TpmCustomerSummaryConfigureImportDto> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Map<Integer, String> errMap = new HashMap<>();

        for (Map.Entry<Integer, TpmCustomerSummaryConfigureImportDto> row : data.entrySet()) {
            TpmCustomerSummaryConfigureImportDto v = row.getValue();
            this.validateIsTrue(StringUtils.isNotBlank(v.getBusinessFormatCode()), "计费业态不能为空！");
            this.validateIsTrue(StringUtils.isNotBlank(v.getSalesInstitutionErpCode()), "计费销售机构MDG编码不能为空！");
            this.validateIsTrue(StringUtils.isNotBlank(v.getChannelCode()), "计费渠道不能为空！");
            this.validateIsTrue(StringUtils.isNotBlank(v.getCustomerErpCode()), "计费MDG客户编码不能为空！");
            this.validateIsTrue(StringUtils.isNotBlank(v.getBusinessFormatCodeSummary()), "合并取量业态不能为空！");
            this.validateIsTrue(StringUtils.isNotBlank(v.getSalesInstitutionErpCodeSummary()), "合并取量销售机构MDG编码不能为空！");
            this.validateIsTrue(StringUtils.isNotBlank(v.getChannelCodeSummary()), "合并取量渠道不能为空！");
            this.validateIsTrue(StringUtils.isNotBlank(v.getCustomerErpCodeSummary()), "合并MDG客户编码不能为空！");
            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(row.getKey(), errInfo);
            }
        }

        return errMap;
    }

    @Override
    @Transactional
    public Map<Integer, String> execute(LinkedHashMap<Integer, TpmCustomerSummaryConfigureImportDto> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Validate.notNull(data, "导入数据对象为空！");
        Validate.isTrue(data.values().size() <= getBatchCount(), "单次导入数据不能超过%s条", getBatchCount());
        Map<Integer, String> errMap = new HashMap<>();
        log.info("TPM-客户合并计量配置表导入=====>数量：{}", data.size());
        log.info("TPM-客户合并计量配置表导入=====>获取渠道");
        Map<String, String> channelMap = this.getChannelMap(data);
        log.info("TPM-客户合并计量配置表导入=====>获取业态");
        Map<String, String> mdmBusinessFormatMap = dictToolkitService.findConvertMapByDictTypeCode("mdm_business_format");
        log.info("TPM-客户合并计量配置表导入=====>验证空和数据正确性");
        Set<String> customerCodeSet = this.valNull(data, channelMap, mdmBusinessFormatMap, errMap);
        log.info("TPM-客户合并计量配置表导入=====>数据库验重和数据转换");
        List<TpmCustomerSummaryConfigureEntity> list = this.convertData(data, customerCodeSet, channelMap, mdmBusinessFormatMap, errMap);
        if (errMap.isEmpty()) {
            log.info("TPM-客户合并计量配置表导入=====>开始保存");
            Lists.partition(list, 500).forEach(e -> {
                log.info("TPM-客户合并计量配置表导入=====>保存：{}", e.size());
                tpmCustomerSummaryConfigureRepository.saveBatch(e);
            });
        }
        return errMap;
    }

    @Override
    public Class<TpmCustomerSummaryConfigureImportDto> findCrmExcelVoClass() {
        return TpmCustomerSummaryConfigureImportDto.class;
    }

    @Override
    public String getTemplateCode() {
        return "TPM_CUSTOMER_SUMMARY_CONFIGURE_IMPORT";
    }

    @Override
    public String getTemplateName() {
        return "TPM-客户合并计量配置表导入";
    }

    @Override
    public String getBusinessCode() {
        return "TPM_CUSTOMER_SUMMARY_CONFIGURE_IMPORT";
    }

    @Override
    public String getBusinessName() {
        return "TPM-客户合并计量配置表导入";
    }

    public Map<String, String> getChannelMap(LinkedHashMap<Integer, TpmCustomerSummaryConfigureImportDto> data) {
        Set<String> channelCodeSet = new HashSet<>();
        data.values().forEach(v -> {
            if (StringUtils.isNotEmpty(v.getChannelCode())) {
                channelCodeSet.add(v.getChannelCode());
            }
            if (StringUtils.isNotEmpty(v.getChannelCodeSummary())) {
                channelCodeSet.add(v.getChannelCodeSummary());
            }
        });

        Map<String, String> channelMap = new HashMap<>();
        if (channelCodeSet.size() > 0) {
            List<CustomerChannelVo> channelVos = customerChannelVoService.findByCodes(Lists.newArrayList(channelCodeSet));
            Map<String, String> collect = channelVos.stream()
                    .filter(e -> DelFlagStatusEnum.NORMAL.getCode().equals(e.getDelFlag()) && EnableStatusEnum.ENABLE.getCode().equals(e.getEnableStatus()))
                    .collect(Collectors.toMap(CustomerChannelVo::getCustomerChannelCode, CustomerChannelVo::getCustomerChannelName, (v1, v2) -> v2));
            channelMap.putAll(collect);
        }
        return channelMap;
    }

    public Set<String> valNull(LinkedHashMap<Integer, TpmCustomerSummaryConfigureImportDto> data, Map<String, String> channelMap, Map<String, String> mdmBusinessFormatMap, Map<Integer, String> errMap) {
        Set<String> customerCodeSet = new HashSet<>();
        Set<String> duplicateSet = new HashSet<>();
        data.forEach((index, v) -> {
            if (mdmBusinessFormatMap.containsKey(v.getBusinessFormatCode())) {
                v.setBusinessFormatCode(mdmBusinessFormatMap.get(v.getBusinessFormatCode()));
            } else {
                this.validateIsTrue(false, "计费业态不存在！");
            }
            this.validateIsTrue(channelMap.containsKey(v.getChannelCode()), "计费渠道编码不存在！");

            if (mdmBusinessFormatMap.containsKey(v.getBusinessFormatCodeSummary())) {
                v.setBusinessFormatCodeSummary(mdmBusinessFormatMap.get(v.getBusinessFormatCodeSummary()));
            } else {
                this.validateIsTrue(false, "合并取量业态不存在！");
            }
            this.validateIsTrue(channelMap.containsKey(v.getChannelCodeSummary()), "合并取量渠道编码不存在！");

            String v1 = this.buildCustomerCode(v.getCustomerErpCode(), v.getSalesInstitutionErpCode(), v.getChannelCode(), v.getBusinessFormatCode());
            String v2 = this.buildCustomerCode(v.getCustomerErpCodeSummary(), v.getSalesInstitutionErpCodeSummary(), v.getChannelCodeSummary(), v.getBusinessFormatCodeSummary());
            if (!StringUtils.equals(v1, v2)) {
                customerCodeSet.add(v1);
                customerCodeSet.add(v2);
            } else {
                this.validateIsTrue(false, "计费取量客户不能与合并客户相同！");
            }

            String onlyKey = this.buildOnlyKey(v);
            if (!duplicateSet.contains(onlyKey)) {
                duplicateSet.add(onlyKey);
            } else {
                this.validateIsTrue(false, "计费取量客户与合并客户关系重复！");
            }

            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(index, errInfo);
            }
        });
        return customerCodeSet;
    }

    public List<TpmCustomerSummaryConfigureEntity> convertData(LinkedHashMap<Integer, TpmCustomerSummaryConfigureImportDto> data,
                                                               Set<String> customerCodeSet,
                                                               Map<String, String> channelMap,
                                                               Map<String, String> mdmBusinessFormatMap, Map<Integer, String> errMap) {
        List<CustomerVo> customerVos = customerVoService.findBaseByCustomerCodes(Lists.newArrayList(customerCodeSet));
        Map<String, CustomerVo> customerVoMap = customerVos.stream()
                .filter(e -> DelFlagStatusEnum.NORMAL.getCode().equals(e.getDelFlag()) && EnableStatusEnum.ENABLE.getCode().equals(e.getEnableStatus()))
                .collect(Collectors.toMap(CustomerVo::getCustomerCode, v -> v, (v1, v2) -> v2));

        List<String> onlyKeys = data.values().stream().map(this::buildOnlyKey).collect(Collectors.toList());
        List<String> existOnlyKeys = tpmCustomerSummaryConfigureRepository.lambdaQuery()
                .eq(TpmCustomerSummaryConfigureEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .in(TpmCustomerSummaryConfigureEntity::getOnlyKey, onlyKeys)
                .select(TpmCustomerSummaryConfigureEntity::getOnlyKey)
                .list()
                .stream().map(TpmCustomerSummaryConfigureEntity::getOnlyKey)
                .collect(Collectors.toList());

        List<TpmCustomerSummaryConfigureEntity> list = new ArrayList<>();
        data.forEach((index, v) -> {
            String onlyKey = this.buildOnlyKey(v);
            this.validateIsTrue(!existOnlyKeys.contains(onlyKey), "关系已存在！");

            String v1 = this.buildCustomerCode(v.getCustomerErpCode(), v.getSalesInstitutionErpCode(), v.getChannelCode(), v.getBusinessFormatCode());
            this.validateIsTrue(customerVoMap.containsKey(v1), "计费客户信息不存在！");

            String v2 = this.buildCustomerCode(v.getCustomerErpCodeSummary(), v.getSalesInstitutionErpCodeSummary(), v.getChannelCodeSummary(), v.getBusinessFormatCodeSummary());
            this.validateIsTrue(customerVoMap.containsKey(v2), "合并取量客户信息不存在！");

            TpmCustomerSummaryConfigureEntity entity = new TpmCustomerSummaryConfigureEntity();
            BeanUtils.copyProperties(v, entity);
            entity.setOnlyKey(onlyKey);
            entity.setTenantCode(TenantUtils.getTenantCode());
            entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            entity.setChannelName(channelMap.get(entity.getChannelCode()));
            entity.setChannelNameSummary(channelMap.get(entity.getChannelCodeSummary()));

            CustomerVo vo1 = customerVoMap.get(v1);
            if (ObjectUtils.isNotEmpty(vo1)){
                entity.setCustomerCode(vo1.getCustomerCode());
                entity.setCustomerName(vo1.getCustomerName());
            }
            CustomerVo vo2 = customerVoMap.get(v2);
            if (ObjectUtils.isNotEmpty(vo2)){
                entity.setCustomerCodeSummary(vo2.getCustomerCode());
                entity.setCustomerNameSummary(vo2.getCustomerName());
            }

            list.add(entity);

            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                if (errMap.containsKey(index)) {
                    errMap.put(index, errMap.get(index) + "," + errInfo);
                } else {
                    errMap.put(index, errInfo);
                }
            }
        });
        return list;
    }

    public String buildCustomerCode(String customerErpCode, String salesInstitutionErpCode, String channelCode, String businessFormatCode) {
        return customerErpCode + salesInstitutionErpCode + channelCode + businessFormatCode;
    }

    public String buildOnlyKey(TpmCustomerSummaryConfigureImportDto v) {
        return v.getBusinessFormatCode() + this.buildCustomerCode(v.getCustomerErpCode(), v.getSalesInstitutionErpCode(), v.getChannelCode(), v.getBusinessFormatCode())
                + this.buildCustomerCode(v.getCustomerErpCodeSummary(), v.getSalesInstitutionErpCodeSummary(), v.getChannelCodeSummary(), v.getBusinessFormatCodeSummary());
    }
}
