package com.biz.crm.tpm.business.warning.config.local.job;

import cn.hutool.core.collection.CollectionUtil;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.common.base.util.UuidCrmUtil;
import com.biz.crm.tpm.business.warning.config.local.repository.TpmWarningMonitoringRepository;
import com.biz.crm.tpm.business.warning.config.local.service.TpmWarningMonitoringService;
import com.biz.crm.tpm.business.warning.config.sdk.dto.TpmWarningConfigDto;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import com.bizunited.nebula.task.dto.DynamicTaskSchedulerDto;
import com.bizunited.nebula.task.service.DynamicTaskSchedulerVoService;
import com.bizunited.nebula.task.vo.DynamicTaskSchedulerVo;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 预警监控定时任务
 *
 * @author wanghaojia
 * @date 2023/12/21 22:04
 */
@Component("refreshWarningMonitoringTask")
@Slf4j
public class RefreshWarningMonitoringTask {

    @Autowired(required = false)
    private DynamicTaskSchedulerVoService dynamicTaskSchedulerVoService;

    @Autowired(required = false)
    private TpmWarningMonitoringRepository tpmWarningMonitoringRepository;

    @Autowired(required = false)
    private TpmWarningMonitoringService tpmWarningMonitoringService;

    /**
     * 刷新一次预警配置的定时任务
     */
    @DynamicTaskService(cornExpression = "0 0 0 * * ?", taskDesc = "预警监控定时创建任务")
    public void refreshWarningMonitoring() {
        //生效状态的预警配置任务
        List<TpmWarningConfigDto> runTpmWarningMonitoring = tpmWarningMonitoringService.findRunTpmWarningMonitoring(new TpmWarningConfigDto());

        //生效一天，一天结束后重新加载
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.HOUR_OF_DAY, 23);
        calendar.set(Calendar.MINUTE, 59);
        calendar.set(Calendar.SECOND, 59);
        calendar.set(Calendar.MILLISECOND, 999);

        String dayStr = DateUtil.getDateStrByFormat(calendar.getTime(), DateUtil.DEFAULT_YEAR_MONTH_DAY);

        for (TpmWarningConfigDto tpmWarningConfigDto : runTpmWarningMonitoring) {
            String code = tpmWarningConfigDto.getCode();
            String taskCode = "WARING-MONITORING-" + dayStr + code + UuidCrmUtil.general();
            String taskName = "预警监控-" + dayStr + code + tpmWarningConfigDto.getName();
            dynamicTaskSchedulerVoService.createIgnorePrefix(taskCode, "refreshWarningMonitoringTask", "runTpmWarningMonitoring",
                    tpmWarningConfigDto.getWarningFrequency(), calendar.getTime(), taskName, tpmWarningConfigDto.getCode());
        }
    }

    public void stopWarningMonitoringByTaskDesc(String taskDesc) {
        DynamicTaskSchedulerDto dynamicTaskSchedulerDto = new DynamicTaskSchedulerDto();
        dynamicTaskSchedulerDto.setTaskDesc(taskDesc);
        List<DynamicTaskSchedulerVo> taskList = tpmWarningMonitoringRepository.findWarningMonitoringTaskList(dynamicTaskSchedulerDto);
        stopWarningMonitoring(taskList.stream().map(DynamicTaskSchedulerVo::getTaskCode).collect(Collectors.toList()));
    }

    public void stopWarningMonitoring(List<String> taskCodes) {
        if (CollectionUtil.isEmpty(taskCodes)) {
            return;
        }
        log.info("预警监控停止任务" + taskCodes);
        dynamicTaskSchedulerVoService.stop(taskCodes.toArray(new String[0]));
    }

    /**
     * 重置已经在执行的定时任务
     */
    public void resetWarningMonitoring(TpmWarningConfigDto tpmWarningConfigDto) {
        String dayStr = DateUtil.getDateStrByFormat(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY);
        //先停掉原来的
        String code = tpmWarningConfigDto.getCode();
        stopWarningMonitoringByTaskDesc("预警监控-" + dayStr + code);
        if (!EnableStatusEnum.ENABLE.getCode().equals(tpmWarningConfigDto.getEnableStatus())) {
            return;
        }
        if (!DelFlagStatusEnum.NORMAL.getCode().equals(tpmWarningConfigDto.getDelFlag())) {
            return;
        }
        Date curDate = new Date();
        if (tpmWarningConfigDto.getStartDate().compareTo(curDate) > 0 || tpmWarningConfigDto.getEndDate().compareTo(curDate) < 0) {
            //还没开始或者已经结束了
            return;
        }
        startNewtWarningMonitoring(dayStr, tpmWarningConfigDto);
    }

    public void startNewtWarningMonitoring(String dayStr, TpmWarningConfigDto tpmWarningConfigDto) {
        startNewtWarningMonitoring(dayStr, Lists.newArrayList(tpmWarningConfigDto));
    }

    public void startNewtWarningMonitoring(String dayStr, List<TpmWarningConfigDto> runTpmWarningMonitoring) {
        if (CollectionUtil.isEmpty(runTpmWarningMonitoring)) {
            return;
        }
        //生效一天，一天结束后重新加载
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.HOUR_OF_DAY, 23);
        calendar.set(Calendar.MINUTE, 59);
        calendar.set(Calendar.SECOND, 59);
        calendar.set(Calendar.MILLISECOND, 999);

        for (TpmWarningConfigDto tpmWarningConfigDto : runTpmWarningMonitoring) {
            String code = tpmWarningConfigDto.getCode();
            String taskCode = "WARING-MONITORING-" + dayStr + code + UuidCrmUtil.general();
            String taskName = "预警监控-" + dayStr + code + tpmWarningConfigDto.getName();
            dynamicTaskSchedulerVoService.createIgnorePrefix(taskCode, "refreshWarningMonitoringTask", "runTpmWarningMonitoring",
                    tpmWarningConfigDto.getWarningFrequency(), calendar.getTime(), taskName, tpmWarningConfigDto.getCode());
        }

    }


    @DynamicTaskService(cornExpression = "0 0 * * * ?", taskDesc = "预警监控定时清理任务-删除已停止任务")
    public void clearWarningMonitoring() {
        DynamicTaskSchedulerDto dynamicTaskSchedulerDto = new DynamicTaskSchedulerDto();
        dynamicTaskSchedulerDto.setTaskDesc("预警监控-");
        dynamicTaskSchedulerDto.setWorkingStatus(3);
        List<DynamicTaskSchedulerVo> taskList = tpmWarningMonitoringRepository.findWarningMonitoringTaskList(dynamicTaskSchedulerDto);
        if (CollectionUtil.isEmpty(taskList)) {
            return;
        }
        dynamicTaskSchedulerVoService.deleteByTaskcodes(taskList.stream().map(DynamicTaskSchedulerVo::getTaskCode).toArray(String[]::new));
    }


    /**
     * 定时任务执行
     */
    public void runTpmWarningMonitoring(String warningCode) {
        TpmWarningConfigDto searchDto = new TpmWarningConfigDto();
        searchDto.setCode(warningCode);
        List<TpmWarningConfigDto> runTpmWarningMonitoring = tpmWarningMonitoringService.findRunTpmWarningMonitoring(searchDto);
        //按编码查询应该只有一个的
        tpmWarningMonitoringService.runTpmWarningMonitoring(runTpmWarningMonitoring.get(0));
    }

}
