package com.biz.crm.tpm.business.warning.config.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mn.common.base.eunm.BusinessFormatEnum;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.tpm.business.warning.config.local.entity.TpmWarningCondition;
import com.biz.crm.tpm.business.warning.config.local.entity.TpmWarningConfig;
import com.biz.crm.tpm.business.warning.config.local.entity.TpmWarningReceivingObject;
import com.biz.crm.tpm.business.warning.config.local.job.RefreshWarningMonitoringTask;
import com.biz.crm.tpm.business.warning.config.local.repository.TpmWarningConditionRepository;
import com.biz.crm.tpm.business.warning.config.local.repository.TpmWarningConfigRepository;
import com.biz.crm.tpm.business.warning.config.local.repository.TpmWarningReceivingObjectRepository;
import com.biz.crm.tpm.business.warning.config.sdk.constant.TpmWarningConfigConstant;
import com.biz.crm.tpm.business.warning.config.sdk.dto.TpmWarningConditionDto;
import com.biz.crm.tpm.business.warning.config.sdk.dto.TpmWarningConfigDto;
import com.biz.crm.tpm.business.warning.config.sdk.dto.TpmWarningReceivingObjectDto;
import com.biz.crm.tpm.business.warning.config.sdk.dto.log.TpmWarningConfigLogEventDto;
import com.biz.crm.tpm.business.warning.config.sdk.enums.TpmWarningReceivingObjectTypeEnum;
import com.biz.crm.tpm.business.warning.config.sdk.event.TpmWarningConfigLogEventListener;
import com.biz.crm.tpm.business.warning.config.sdk.service.TpmWarningConfigService;
import com.biz.crm.tpm.business.warning.config.sdk.vo.TpmWarningConditionVo;
import com.biz.crm.tpm.business.warning.config.sdk.vo.TpmWarningConfigDisplayVo;
import com.biz.crm.tpm.business.warning.config.sdk.vo.TpmWarningConfigVo;
import com.biz.crm.tpm.business.warning.config.sdk.vo.TpmWarningReceivingObjectVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

/**
 * TPM-预警配置(TpmWarningConfig)表服务实现类
 *
 * @author duyiran
 * @since 2022-11-03 15:44:39
 */
@Service("tpmWarningConfigService")
public class TpmWarningConfigServiceImpl implements TpmWarningConfigService {

    @Autowired(required = false)
    private TpmWarningConfigRepository tpmWarningConfigRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private TpmWarningConditionRepository tpmWarningConditionRepository;

    @Autowired(required = false)
    private TpmWarningReceivingObjectRepository tpmWarningReceivingObjectRepository;

    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private GenerateCodeService generateCodeService;

    @Autowired(required = false)
    private RefreshWarningMonitoringTask refreshWarningMonitoringTask;

    /**
     * 分页查询数据
     *
     * @param pageable 分页对象
     * @param dto      dto对象
     * @return
     */
    @Override
    public Page<TpmWarningConfigVo> findByConditions(Pageable pageable, TpmWarningConfigDto dto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(dto)) {
            dto = new TpmWarningConfigDto();
        }
        return this.tpmWarningConfigRepository.findByConditions(pageable, dto);
    }

    /**
     * 执行预警
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void executeWarning() {
        List<TpmWarningConfigVo> tpmWarningConfigVoList = this.tpmWarningConfigRepository.findUnderWay();
        if(CollectionUtils.isEmpty(tpmWarningConfigVoList)){
            return;
        }
        List<String> codeList = tpmWarningConfigVoList.stream().map(TpmWarningConfigVo::getCode).collect(Collectors.toList());
        Map<String, List<TpmWarningCondition>> codeToConditionListMap = tpmWarningConditionRepository.findByCodeList(codeList);


    }

    /**
     * 通过主键查询单条数据(前端用)
     *
     * @param id 主键
     * @return 单条数据
     */
    @Override
    public TpmWarningConfigDisplayVo findById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        TpmWarningConfig entity = tpmWarningConfigRepository.getById(id);
        Validate.notNull(entity, "数据不存在，请刷新后重试！");
        TpmWarningConfigDisplayVo tpmWarningConfigVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, TpmWarningConfigDisplayVo.class, LinkedHashSet.class, ArrayList.class);
        List<TpmWarningConditionVo> conditions = tpmWarningConfigRepository.findConditionsByCode(entity.getCode());
        List<TpmWarningReceivingObjectVo> objects = tpmWarningConfigRepository.findObjectsByCode(entity.getCode());
        tpmWarningConfigVo.setConditionList(conditions);
        tpmWarningConfigVo.setPushDimensionList(objects.stream().filter(e -> TpmWarningReceivingObjectTypeEnum.PUSH_DIMENSION.getCode().equals(e.getWarningReceivingObjectType())).collect(Collectors.toList()));
        tpmWarningConfigVo.setOrgs(objects.stream().filter(e -> TpmWarningReceivingObjectTypeEnum.ORG.getCode().equals(e.getWarningReceivingObjectType()) && BooleanEnum.TRUE.getCapital().equals(e.getIsContains())).collect(Collectors.toList()));
        tpmWarningConfigVo.setNonOrgs(objects.stream().filter(e -> TpmWarningReceivingObjectTypeEnum.ORG.getCode().equals(e.getWarningReceivingObjectType()) && BooleanEnum.FALSE.getCapital().equals(e.getIsContains())).collect(Collectors.toList()));
        tpmWarningConfigVo.setPositions(objects.stream().filter(e -> TpmWarningReceivingObjectTypeEnum.POSITION.getCode().equals(e.getWarningReceivingObjectType()) && BooleanEnum.TRUE.getCapital().equals(e.getIsContains())).collect(Collectors.toList()));
        tpmWarningConfigVo.setNonPositions(objects.stream().filter(e -> TpmWarningReceivingObjectTypeEnum.POSITION.getCode().equals(e.getWarningReceivingObjectType()) && BooleanEnum.FALSE.getCapital().equals(e.getIsContains())).collect(Collectors.toList()));
        if (StringUtils.isNotEmpty(tpmWarningConfigVo.getPushDimension())){
            tpmWarningConfigVo.setPushDimensionStrList(Arrays.asList(tpmWarningConfigVo.getPushDimension().split(",")));
        }
        return tpmWarningConfigVo;
    }

    /**
     * 新增数据
     *
     * @param dto dto对象
     * @return 新增结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public TpmWarningConfigVo create(TpmWarningConfigDto dto) {
        this.createValidate(dto);
        TpmWarningConfig entity = nebulaToolkitService.copyObjectByWhiteList(dto, TpmWarningConfig.class, HashSet.class, ArrayList.class);
        //数据库验重
        Validate.isTrue(tpmWarningConfigRepository.repeatValidate(dto), "已存在相同的预警维度，请勿重复提交！");
        //租户编号赋值
        String tenantCode = TenantUtils.getTenantCode();
        entity.setTenantCode(tenantCode);
        String code = generateCodeService.generateCode(TpmWarningConfigConstant.TPM_WARNING_CONFIG_FORMULA_PREFIX, 1, 6, 0L, TimeUnit.DAYS).get(0);
        entity.setCode(code);
        Collection<TpmWarningCondition> conditions = nebulaToolkitService.copyCollectionByWhiteList(dto.getConditionList(), TpmWarningConditionDto.class,
                TpmWarningCondition.class, HashSet.class, ArrayList.class);
        Collection<TpmWarningReceivingObject> objects = nebulaToolkitService.copyCollectionByWhiteList(dto.getObjectList(), TpmWarningReceivingObjectDto.class,
                TpmWarningReceivingObject.class, HashSet.class, ArrayList.class);
        AtomicInteger index = new AtomicInteger(1);
        conditions.forEach(e -> {
            e.setTenantCode(tenantCode);
            e.setCode(code);
            e.setSubCode(code + "-" + index.getAndIncrement());
        });
        objects.forEach(e -> {
            e.setTenantCode(tenantCode);
            e.setCode(code);
        });
        //保存
        this.tpmWarningConfigRepository.save(entity);
        this.tpmWarningConditionRepository.saveBatch(conditions);
        this.tpmWarningReceivingObjectRepository.saveBatch(objects);
        //日志
        TpmWarningConfigLogEventDto logEventDto = new TpmWarningConfigLogEventDto();
        logEventDto.setOriginal(null);
        logEventDto.setNewest(dto);
        SerializableBiConsumer<TpmWarningConfigLogEventListener, TpmWarningConfigLogEventDto> onCreate =
                TpmWarningConfigLogEventListener::onCreate;
        this.nebulaNetEventClient.publish(logEventDto, TpmWarningConfigLogEventListener.class, onCreate);

        return this.nebulaToolkitService.copyObjectByWhiteList(entity, TpmWarningConfigVo.class, LinkedHashSet.class, ArrayList.class);
    }

    /**
     * 修改新据
     *
     * @param dto dto对象
     * @return 修改结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public TpmWarningConfigVo update(TpmWarningConfigDto dto) {
        this.updateValidate(dto);
        //取出旧的记录
        TpmWarningConfig oldEntity = tpmWarningConfigRepository.getById(dto.getId());
        Validate.notNull(oldEntity, "数据不存在，请刷新后重试！");
        Validate.isTrue(DelFlagStatusEnum.NORMAL.getCode().equals(oldEntity.getDelFlag()), "数据不存在，请刷新后重试！");
        String tenantCode = TenantUtils.getTenantCode();
        Validate.isTrue(tenantCode.equals(oldEntity.getTenantCode()), "数据不存在，请刷新后重试！");
        String code = oldEntity.getCode();
        Validate.isTrue(code.equals(dto.getCode()), "数据不存在，请刷新后重试！");
        TpmWarningConfigDto oldDto = this.nebulaToolkitService.copyObjectByWhiteList(oldEntity, TpmWarningConfigDto.class, LinkedHashSet.class, ArrayList.class);
        List<TpmWarningConditionVo> oldConditions = tpmWarningConfigRepository.findConditionsByCode(code);
        List<TpmWarningReceivingObjectVo> oldObjects = tpmWarningConfigRepository.findObjectsByCode(code);
        oldDto.setConditionList(Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(oldConditions, TpmWarningConditionVo.class, TpmWarningConditionDto.class, HashSet.class, ArrayList.class)));
        oldDto.setObjectList(Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(oldObjects, TpmWarningReceivingObjectVo.class, TpmWarningReceivingObjectDto.class, HashSet.class, ArrayList.class)));

        //数据库验重
        Validate.isTrue(tpmWarningConfigRepository.repeatValidate(dto), "已存在相同的预警维度，请勿重复提交！");

        //新的实体,租户编号赋值
        TpmWarningConfig newEntity = nebulaToolkitService.copyObjectByWhiteList(dto, TpmWarningConfig.class, HashSet.class, ArrayList.class);
        Collection<TpmWarningCondition> newConditions = nebulaToolkitService.copyCollectionByWhiteList(
                dto.getConditionList(), TpmWarningConditionDto.class, TpmWarningCondition.class, HashSet.class, ArrayList.class);
        Collection<TpmWarningReceivingObject> newObjects = nebulaToolkitService.copyCollectionByWhiteList(
                dto.getObjectList(), TpmWarningReceivingObjectDto.class, TpmWarningReceivingObject.class, HashSet.class, ArrayList.class);
        AtomicInteger index = new AtomicInteger(1);
        newConditions.forEach(e -> {
            e.setTenantCode(tenantCode);
            e.setCode(code);
            e.setSubCode(code + "-" + index.getAndIncrement());
        });
        newObjects.forEach(e -> {
            e.setTenantCode(tenantCode);
            e.setCode(code);
        });

        //物理删除子表信息
        this.tpmWarningConditionRepository.removeByCode(code, tenantCode);
        this.tpmWarningReceivingObjectRepository.removeByCode(code, tenantCode);

        //保存
        this.tpmWarningConfigRepository.saveOrUpdate(newEntity);
        this.tpmWarningConditionRepository.saveOrUpdateBatch(newConditions);
        this.tpmWarningReceivingObjectRepository.saveOrUpdateBatch(newObjects);

        //日志
        TpmWarningConfigLogEventDto logEventDto = new TpmWarningConfigLogEventDto();
        logEventDto.setOriginal(oldDto);
        logEventDto.setNewest(dto);
        SerializableBiConsumer<TpmWarningConfigLogEventListener, TpmWarningConfigLogEventDto> onUpdate =
                TpmWarningConfigLogEventListener::onUpdate;
        this.nebulaNetEventClient.publish(logEventDto, TpmWarningConfigLogEventListener.class, onUpdate);
        return this.nebulaToolkitService.copyObjectByWhiteList(newEntity, TpmWarningConfigVo.class, LinkedHashSet.class, ArrayList.class);
    }

    /**
     * 删除数据
     *
     * @param idList 主键结合
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public List<TpmWarningConfigVo> delete(List<String> idList) {
        Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
        String tenantCode = TenantUtils.getTenantCode();

        List<TpmWarningConfig> list = tpmWarningConfigRepository.lambdaQuery()
                .in(TpmWarningConfig::getId, idList)
                .eq(TpmWarningConfig::getTenantCode, tenantCode)
                .list();
        for (TpmWarningConfig tpmWarningConfig : list) {
            tpmWarningConfig.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
        }

        tpmWarningConfigRepository.lambdaUpdate()
                .in(TpmWarningConfig::getId, idList)
                .eq(TpmWarningConfig::getTenantCode, tenantCode)
                .set(TpmWarningConfig::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
                .update();
        return (List<TpmWarningConfigVo>) nebulaToolkitService.copyCollectionByBlankList(list,TpmWarningConfig.class,TpmWarningConfigVo.class,HashSet.class,ArrayList.class);
    }

    /**
     * 创建验证
     *
     * @param dto dto
     */
    private void createValidate(TpmWarningConfigDto dto) {
        Validate.notNull(dto, "新增时，对象信息不能为空！");
        dto.setId(null);
        Validate.notBlank(dto.getName(), "新增数据时，预警规则名称不能为空！");
        Validate.notBlank(dto.getBusinessFormatCode(), "新增数据时，业态不能为空！");
        Validate.notBlank(dto.getBusinessUnitCode(), "新增数据时，业务单元不能为空！");
//        if (!StringUtils.equals(dto.getBusinessUnitCode(), BusinessUnitEnum.VERTICAL.getCode())){
//            Validate.notBlank(dto.getActivityFormCode(), "新增数据时，活动形式编码不能为空！");
//            Validate.notBlank(dto.getActivityFormName(), "新增数据时，活动形式名称不能为空！");
//            Validate.notBlank(dto.getSalesOrgCode(), "新增数据时，销售组织编码不能为空！");
//            Validate.notBlank(dto.getSalesOrgName(), "新增数据时，销售组织名称不能为空！");
//        }
        Validate.notNull(dto.getStartDate(), "新增数据时，预警开始日期不能为空！");
        Validate.notNull(dto.getEndDate(), "新增数据时，预警结束日期不能为空！");
        Validate.notBlank(dto.getWarningMode(), "新增数据时，预警方式不能为空！");
        Validate.noNullElements(dto.getConditionList(), "新增数据时，预警条件不能为空！");
        this.validateConditionList(dto,dto.getConditionList());
        Validate.noNullElements(dto.getObjectList(), "新增数据时，预警对象不能为空！");
        this.validateObjectList(dto.getObjectList());
    }

    /**
     * 修改验证
     *
     * @param dto
     */
    private void updateValidate(TpmWarningConfigDto dto) {
        Validate.notNull(dto, "修改时，对象信息不能为空！");
        Validate.notBlank(dto.getId(), "修改数据时，不能为空！");
        Validate.notBlank(dto.getName(), "修改数据时，预警规则名称不能为空！");
        Validate.notBlank(dto.getBusinessFormatCode(), "修改数据时，业态不能为空！");
//        Validate.notBlank(dto.getActivityFormCode(), "修改数据时，活动形式编码不能为空！");
//        Validate.notBlank(dto.getActivityFormName(), "修改数据时，活动形式名称不能为空！");
        Validate.notBlank(dto.getBusinessUnitCode(), "修改数据时，业务单元不能为空！");
//        Validate.notBlank(dto.getSalesOrgCode(), "修改数据时，销售组织编码不能为空！");
//        Validate.notBlank(dto.getSalesOrgName(), "修改数据时，销售组织名称不能为空！");
        Validate.notNull(dto.getStartDate(), "修改数据时，预警开始日期不能为空！");
        Validate.notNull(dto.getEndDate(), "修改数据时，预警结束日期不能为空！");
        Validate.notBlank(dto.getWarningMode(), "修改数据时，预警方式不能为空！");
        Validate.noNullElements(dto.getConditionList(), "修改数据时，预警条件不能为空！");
        this.validateConditionList(dto,dto.getConditionList());
        Validate.noNullElements(dto.getObjectList(), "修改数据时，预警对象不能为空！");
        this.validateObjectList(dto.getObjectList());
    }

    /**
     * 验证预警条件
     *
     * @param conditionList conditionList
     */
    private void validateConditionList(TpmWarningConfigDto dto,List<TpmWarningConditionDto> conditionList) {
        //推送维度转换
        if (!CollectionUtils.isEmpty(dto.getPushDimensionStrList())){
            dto.setPushDimension(String.join(",",dto.getPushDimensionStrList()));
        }
        conditionList.forEach(e -> {
            e.setId(null);
            Validate.notBlank(e.getFormula(), "公式不能为空！");
            Validate.notBlank(e.getFormulaName(), "公式不能为空！");
            Validate.notBlank(e.getWarningLevel(), "预警级别不能为空！");
        });
    }

    /**
     * 验证预警对象
     *
     * @param objectList objectList
     */
    private void validateObjectList(List<TpmWarningReceivingObjectDto> objectList) {
        objectList.forEach(e -> {
            e.setId(null);
            Validate.notBlank(e.getSubCode(), "组织/职位编码不能为空！");
            Validate.notBlank(e.getIsContains(), "是否包含不能为空！");
            Validate.notBlank(e.getWarningReceivingObjectType(), "预警对象类型不能为空！");
        });
    }

}

