package com.biz.crm.tpm.business.warning.config.local.service.internal;

import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.tpm.business.warning.config.local.entity.TpmWarningCondition;
import com.biz.crm.tpm.business.warning.config.local.entity.TpmWarningMonitoring;
import com.biz.crm.tpm.business.warning.config.local.entity.TpmWarningMonitoringLog;
import com.biz.crm.tpm.business.warning.config.local.repository.TpmWarningConditionRepository;
import com.biz.crm.tpm.business.warning.config.local.repository.TpmWarningConfigRepository;
import com.biz.crm.tpm.business.warning.config.local.repository.TpmWarningMonitoringLogRepository;
import com.biz.crm.tpm.business.warning.config.local.repository.TpmWarningMonitoringRepository;
import com.biz.crm.tpm.business.warning.config.local.service.TpmWarningMonitoringService;
import com.biz.crm.tpm.business.warning.config.local.util.WarningMonitoringMathUtil;
import com.biz.crm.tpm.business.warning.config.sdk.dto.TpmWarningConditionDto;
import com.biz.crm.tpm.business.warning.config.sdk.dto.TpmWarningConfigDto;
import com.biz.crm.tpm.business.warning.config.sdk.dto.TpmWarningMonitoringVariable;
import com.biz.crm.tpm.business.warning.config.sdk.enums.TpmWaringMonitoringStatus;
import com.biz.crm.tpm.business.warning.config.sdk.service.AbstractTpmWarningMonitoringVariableRegister;
import com.biz.crm.tpm.business.warning.config.sdk.vo.TpmWarningConditionVo;
import com.biz.crm.tpm.business.warning.config.sdk.vo.TpmWarningConfigVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 监控表预警
 * @author wanghaojia
 * @date 2023/6/14 16:27
 */
@Service
public class TpmWarningMonitoringServiceImpl implements TpmWarningMonitoringService {

    @Autowired(required = false)
    private TpmWarningConfigRepository tpmWarningConfigRepository;

    @Autowired(required = false)
    private TpmWarningConditionRepository tpmWarningConditionRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private List<AbstractTpmWarningMonitoringVariableRegister> variableRegisters;


    @Autowired(required = false)
    private TpmWarningMonitoringRepository tpmWarningMonitoringRepository;


    @Autowired(required = false)
    private TpmWarningMonitoringLogRepository tpmWarningMonitoringLogRepository;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Autowired(required = false)
    private TpmWarningMonitoringAsyncService tpmWarningMonitoringAsyncService;

    @Autowired(required = false)
    private DictDataVoService dictDataVoService;

    @Autowired(required = false)
    private RedisLockService redisLockService;


    @Override
    public List<TpmWarningMonitoringVariable> findTpmWarningMonitoringVariableList(String monitoringTable) {
        List<TpmWarningMonitoringVariable> list = Lists.newArrayList();
        for (AbstractTpmWarningMonitoringVariableRegister variableRegister : variableRegisters) {
            if (variableRegister.monitoringTable().equals(monitoringTable)){
                return variableRegister.allVariableList();
            }
        }
        return list;
    }

    @Override
    public List<TpmWarningConfigDto> findRunTpmWarningMonitoring(TpmWarningConfigDto dto) {
        return tpmWarningConfigRepository.findRunTpmWarningMonitoring(dto);
    }

    @Override
    @Transactional(rollbackOn = Exception.class)
    public void runTpmWarningMonitoring(TpmWarningConfigDto dto) {
        String lockKey = "warning_monitoring:lock:"+dto.getCode();
        boolean lockSuccess = false;
        try {
            lockSuccess = redisLockService.tryLock(lockKey, TimeUnit.HOURS, 2);
            if (!lockSuccess){
                throw new RuntimeException("预警[" + dto.getCode() + "]正在执行，请稍后操作！");
            }
            String monitoringTable = dto.getMonitoringTable();
            AbstractTpmWarningMonitoringVariableRegister variableRegister = null;
            for (AbstractTpmWarningMonitoringVariableRegister register : variableRegisters) {
                if (register.monitoringTable().equals(monitoringTable)){
                    variableRegister = register;
                    break;
                }
            }
            if (null == variableRegister){
                return;
            }
            Map<String, List<TpmWarningCondition>> conditionListMap = tpmWarningConditionRepository.findByCodeList(Lists.newArrayList(dto.getCode()));
            List<TpmWarningCondition> conditionList = conditionListMap.get(dto.getCode());
            if (CollectionUtils.isEmpty(conditionList)){
                return;
            }
            List<TpmWarningConditionDto> conditionDtoList = (List<TpmWarningConditionDto>) nebulaToolkitService.copyCollectionByWhiteList(conditionList, TpmWarningCondition.class, TpmWarningConditionDto.class, HashSet.class, ArrayList.class);
            dto.setConditionList(conditionDtoList);
            for (TpmWarningConditionDto tpmWarningConditionDto : conditionDtoList) {
                //处理下取余和百分号
                if (StringUtils.isEmpty(tpmWarningConditionDto.getFormula())){
                    continue;
                }
                tpmWarningConditionDto.setFormula(WarningMonitoringMathUtil.formatFormulaPercent(tpmWarningConditionDto.getFormula()));
            }
            List<Map<String, Object>> resultMapList = variableRegister.dataList(conditionDtoList);
            if (CollectionUtils.isEmpty(resultMapList)){
                return;//没有预警数据
            }
            List<TpmWarningMonitoring> warningMonitoringList = Lists.newArrayList();
            List<TpmWarningMonitoringLog> warningMonitoringLogList = Lists.newArrayList();
            String runUuid = UUID.randomUUID().toString().replace("-", "");
            List<TpmWarningMonitoringVariable> variableList = variableRegister.allVariableList();
            String tenantCode = TenantUtils.getTenantCode();

            List<Map<String, Object>> warningDataList = Lists.newArrayList();
            for (Map<String, Object> objectMap : resultMapList) {
                String businessCodeField = variableRegister.businessCodeField();
                Object businessCodeObj = objectMap.get(businessCodeField);
                if (null == businessCodeObj){
                    continue;//没有主键不处理
                }
                String businessCode = businessCodeObj.toString();
                int warningVariableCount = 0;
                for (TpmWarningConditionDto tpmWarningCondition : conditionDtoList) {
                    String computeFormula = tpmWarningCondition.getFormula();
                    for (TpmWarningMonitoringVariable tpmWarningMonitoringVariable : variableList) {
                        if (!computeFormula.contains(tpmWarningMonitoringVariable.getVariableField())){
                            continue;
                        }
                        Object valueObj = objectMap.getOrDefault(tpmWarningMonitoringVariable.getVariableField(),0);
                        computeFormula = computeFormula.replace(tpmWarningMonitoringVariable.getVariableField(),valueObj.toString());
                    }
                    Boolean aBoolean = true;
                    if (variableRegister.doMathCompute()){
                        aBoolean = WarningMonitoringMathUtil.computeCondition(computeFormula);
                    }
                    if (aBoolean){
                        warningVariableCount ++ ;
                        tpmWarningCondition.getWarningDataList().add(objectMap);
                        tpmWarningCondition.setWarningCount(Optional.ofNullable(tpmWarningCondition.getWarningCount()).orElse(0)+1);
                        //条件成立才存日志
                        TpmWarningMonitoringLog tpmWarningMonitoringLog = new TpmWarningMonitoringLog();
                        tpmWarningMonitoringLog.setRunUuid(runUuid);
                        tpmWarningMonitoringLog.setWarningCode(dto.getCode());
                        tpmWarningMonitoringLog.setFormula(tpmWarningCondition.getFormula());
                        tpmWarningMonitoringLog.setFormulaName(tpmWarningCondition.getFormulaName());
                        tpmWarningMonitoringLog.setFormulaValue(computeFormula);
                        tpmWarningMonitoringLog.setBusinessCode(businessCode);
                        tpmWarningMonitoringLog.setTenantCode(tenantCode);
                        warningMonitoringLogList.add(tpmWarningMonitoringLog);
                    }
                }
                TpmWarningMonitoring tpmWarningMonitoring = new TpmWarningMonitoring();
                tpmWarningMonitoring.setWarningCode(dto.getCode());
                tpmWarningMonitoring.setWarningName(dto.getName());
                tpmWarningMonitoring.setMonitoringTable(dto.getMonitoringTable());
                tpmWarningMonitoring.setBusinessCode(businessCode);
                tpmWarningMonitoring.setTenantCode(tenantCode);
                if (warningVariableCount > 0){
                    warningDataList.add(objectMap);
                }
                tpmWarningMonitoring.setWarningVariableCount(warningVariableCount);
                warningMonitoringList.add(tpmWarningMonitoring);
            }
            if (!CollectionUtils.isEmpty(warningMonitoringLogList)){
                //日志每次存
//                List<List<TpmWarningMonitoringLog>> partition = Lists.partition(warningMonitoringLogList, 1000);
//                for (List<TpmWarningMonitoringLog> tpmWarningMonitoringLogs : partition) {
//                    tpmWarningMonitoringLogRepository.saveBatch(tpmWarningMonitoringLogs);
//                }
                //有日志说明有报警数据推送消息
                tpmWarningMonitoringAsyncService.pushMsg(loginUserService.getAbstractLoginUser(),dto,variableRegister,warningDataList,warningMonitoringLogList);
            }
            updateWarningMonitoringPartition(dto,warningMonitoringList);
        }finally {
            if (lockSuccess){
                redisLockService.unlock(lockKey);
            }
        }
    }

    private void updateWarningMonitoringPartition(TpmWarningConfigDto dto,List<TpmWarningMonitoring> warningMonitoringList){
        List<List<TpmWarningMonitoring>> partition = Lists.partition(warningMonitoringList, 1000);
        for (List<TpmWarningMonitoring> tpmWarningMonitorings : partition) {
            updateWarningMonitoring(dto,tpmWarningMonitorings);
        }
    }

    private void updateWarningMonitoring(TpmWarningConfigDto dto,List<TpmWarningMonitoring> warningMonitoringList){
        if (CollectionUtils.isEmpty(warningMonitoringList)){
            return;
        }
        List<String> businessCodeList = warningMonitoringList.stream().map(TpmWarningMonitoring::getBusinessCode).collect(Collectors.toList());
        List<TpmWarningMonitoring> existsList = tpmWarningMonitoringRepository.findByBusinessCodeList(dto.getMonitoringTable(),businessCodeList);
        Map<String, TpmWarningMonitoring> existsMap = existsList.stream().collect(Collectors.toMap(TpmWarningMonitoring::getBusinessCode, Function.identity()));
        List<TpmWarningMonitoring> saveList = Lists.newArrayList();
        List<TpmWarningMonitoring> updateList = Lists.newArrayList();
        List<String> deleteList = Lists.newArrayList();
        for (TpmWarningMonitoring tpmWarningMonitoring : warningMonitoringList) {
            if (existsMap.containsKey(tpmWarningMonitoring.getBusinessCode())){
                //已经存在数据了
                TpmWarningMonitoring existsTpmWarningMonitoring = existsMap.get(tpmWarningMonitoring.getBusinessCode());
                existsTpmWarningMonitoring.setWarningCount(existsTpmWarningMonitoring.getWarningCount()+1);
                existsTpmWarningMonitoring.setWarningVariableCount(tpmWarningMonitoring.getWarningVariableCount());
                if (existsTpmWarningMonitoring.getWarningVariableCount() == 0){
                    tpmWarningMonitoring.setWarningStatus(TpmWaringMonitoringStatus.NORMAL.getCode());
                    deleteList.add(existsTpmWarningMonitoring.getId());
                }else{
                    updateList.add(existsTpmWarningMonitoring);
                }
            }else{
                //如果没异常不保存
                if (tpmWarningMonitoring.getWarningVariableCount() == 0){
                    continue;
                }
                tpmWarningMonitoring.setRecoverWarningCount(0);
                tpmWarningMonitoring.setWarningCount(1);
                tpmWarningMonitoring.setWarningStatus(TpmWaringMonitoringStatus.WARNING.getCode());
                saveList.add(tpmWarningMonitoring);
            }
        }
        if (!CollectionUtils.isEmpty(saveList)){
            tpmWarningMonitoringRepository.saveBatch(saveList);
        }
        if (!CollectionUtils.isEmpty(updateList)){
            tpmWarningMonitoringRepository.updateBatchById(updateList);
        }
        if (!CollectionUtils.isEmpty(deleteList)){
            tpmWarningMonitoringRepository.removeByIds(deleteList);
        }
    }

    @Override
    @Transactional(rollbackOn = Exception.class)
    public void disable(String monitoringTable,List<String> businessCodeList) {
        List<TpmWarningMonitoring> list = tpmWarningMonitoringRepository.findByBusinessCodeList(monitoringTable, businessCodeList);
        for (TpmWarningMonitoring tpmWarningMonitoring : list) {
            Validate.isTrue(EnableStatusEnum.ENABLE.getCode().equals(tpmWarningMonitoring.getEnableStatus()),"存在已禁用数据！");
        }
        list.forEach(item -> {
            item.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
        });
        tpmWarningMonitoringRepository.updateBatchById(list);
    }

    @Override
    @Transactional(rollbackOn = Exception.class)
    public void enable(String monitoringTable,List<String> businessCodeList) {
        List<TpmWarningMonitoring> list = tpmWarningMonitoringRepository.findByBusinessCodeList(monitoringTable, businessCodeList);
        for (TpmWarningMonitoring tpmWarningMonitoring : list) {
            Validate.isTrue(EnableStatusEnum.DISABLE.getCode().equals(tpmWarningMonitoring.getEnableStatus()),"存在未禁用数据！");
        }
        list.forEach(item -> {
            item.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            item.setRecoverWarningCount(Optional.ofNullable(item.getRecoverWarningCount()).orElse(0)+1);
        });
        tpmWarningMonitoringRepository.updateBatchById(list);
    }
}
