package com.biz.crm.tpm.business.withholding.summary.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.identity.FacturerUserDetails;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.tpm.business.withholding.summary.local.service.internal.WithholdingSummaryAsyncService;
import com.biz.crm.tpm.business.withholding.summary.local.service.internal.WithholdingSummaryCeStateAutoSyncXxlJob;
import com.biz.crm.tpm.business.withholding.summary.sdk.constant.WithholdingSummaryConstant;
import com.biz.crm.tpm.business.withholding.summary.sdk.dto.WithholdingSummaryDetailDto;
import com.biz.crm.tpm.business.withholding.summary.sdk.dto.WithholdingSummaryDto;
import com.biz.crm.tpm.business.withholding.summary.sdk.dto.WithholdingSummaryJobParamDto;
import com.biz.crm.tpm.business.withholding.summary.sdk.service.WithholdingSummaryService;
import com.biz.crm.tpm.business.withholding.summary.sdk.vo.WithholdingSummaryDetailVo;
import com.biz.crm.tpm.business.withholding.summary.sdk.vo.WithholdingSummaryVo;
import com.biz.crm.mn.common.base.service.RedisLockService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * <p>
 *
 * </p>
 *
 * @author chenshuang
 * @since 2023-02-14
 */
@RestController
@RequestMapping("/v1/withholding/summary")
@Api(tags = "TPM-预提汇总列表接口")
@Slf4j
public class WithholdingSummaryController {

    @Autowired(required = false)
    private WithholdingSummaryService withholdingSummaryService;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Autowired(required = false)
    private WithholdingSummaryAsyncService withholdingSummaryAsyncService;

    @PostMapping(value = "findByConditions")
    @ApiOperation(value = "列表分页查询")
    public Result<Page<WithholdingSummaryVo>> findByConditions(Pageable pageable, @RequestBody WithholdingSummaryDto dto) {
        try {
            return Result.ok(this.withholdingSummaryService.findByConditions(pageable, dto));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @GetMapping(value = "detail")
    @ApiOperation(value = "详情")
    public Result<WithholdingSummaryVo> detail(@RequestParam("id") String id) {
        try {
            return Result.ok(this.withholdingSummaryService.findById(id));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @PostMapping(value = "create")
    @ApiOperation(value = "新增")
    public Result<?> create(@RequestBody WithholdingSummaryDto dto) {
        String lockKey = WithholdingSummaryConstant.WITHHOLDING_SUMMARY_CREATE + dto.getWithholdingFormulaCode() + ":" + dto.getWithholdingYearMonth();
        boolean hasLock = redisLockService.tryLock(lockKey, TimeUnit.SECONDS, 60 * 30);
        Validate.isTrue(hasLock, "预提汇总规则[%s]，预提年月[%s]正在汇总，请勿重复汇总！", dto.getWithholdingFormulaCode(), dto.getWithholdingYearMonth());
        try {
            //异步接口
//            this.withholdingSummaryAsyncService.create(dto, loginUserService.getAbstractLoginUser());
            this.withholdingSummaryService.create(dto);
            return Result.ok();
        } catch (Exception e) {
            //调异步接口时放开，注释掉finally的
//            redisLockService.unlock(lockKey);
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            redisLockService.unlock(lockKey);
        }
    }

    @PostMapping(value = "update")
    @ApiOperation(value = "编辑")
    public Result<?> update(@RequestBody WithholdingSummaryDto dto) {
        try {
            this.withholdingSummaryService.update(dto);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "删除")
    @DeleteMapping("delete")
    public Result<?> delete(@RequestParam("ids") List<String> ids) {
        try {
            this.withholdingSummaryService.delete(ids);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @PostMapping(value = "refresh")
    @ApiOperation(value = "更新")
    public Result<?> refresh(@RequestBody ArrayList<String> ids) {
        List<String> lockKeys = new ArrayList<>();
        try {
            Validate.isTrue(!CollectionUtils.isEmpty(ids), "更新数据时，主键集合不能为空！");
            ids.forEach(id -> {
                String lockKey = WithholdingSummaryConstant.WITHHOLDING_SUMMARY_LOCK + id;
                boolean hasLock = redisLockService.tryLock(lockKey, TimeUnit.SECONDS, 60 * 30);
                Validate.isTrue(hasLock, "预提汇总数据正在被操作中，请稍后！");
                lockKeys.add(lockKey);
            });
            //异步接口：验证和更新不能同时进行，必须验证全部通过后才能进行更新
//            ids.forEach(id -> this.withholdingSummaryService.checkRefresh(id));
//            ids.forEach(id -> this.withholdingSummaryAsyncService.refresh(id, loginUserService.getAbstractLoginUser()));
            ids.forEach(id -> this.withholdingSummaryService.refresh(id));
            return Result.ok();
        } catch (Exception e) {
            //调异步接口时放开，注释掉finally的
//            if (CollectionUtils.isNotEmpty(lockKeys)) {
//                for (String lockKey : lockKeys) {
//                    redisLockService.unlock(lockKey);
//                }
//            }
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (CollectionUtils.isNotEmpty(lockKeys)) {
                for (String lockKey : lockKeys) {
                    redisLockService.unlock(lockKey);
                }
            }
        }
    }

    @PostMapping(value = "upload")
    @ApiOperation(value = "上传")
    public Result<?> upload(@RequestBody ArrayList<String> ids) {
        List<String> lockKeys = new ArrayList<>();
        try {
            Validate.isTrue(!CollectionUtils.isEmpty(ids), "上传数据时，主键集合不能为空！");
            ids.forEach(id -> {
                String lockKey = WithholdingSummaryConstant.WITHHOLDING_SUMMARY_LOCK + id;
                boolean hasLock = redisLockService.tryLock(lockKey, TimeUnit.SECONDS, 60 * 60);
                Validate.isTrue(hasLock, "预提汇总数据正在被操作中，请稍后！");
                lockKeys.add(lockKey);
            });
            FacturerUserDetails loginDetails = this.loginUserService.getLoginDetails(FacturerUserDetails.class);
            ids.forEach(id -> {
                this.withholdingSummaryService.upload(id, loginDetails);
            });
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (CollectionUtils.isNotEmpty(lockKeys)) {
                for (String lockKey : lockKeys) {
                    redisLockService.unlock(lockKey);
                }
            }
        }
    }

    @ApiOperation(value = "获取明细列表缓存分页接口")
    @GetMapping("findCachePageList")
    public Result<Page<WithholdingSummaryDetailVo>> findCachePageList(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                      @ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                                                      @ApiParam(name = "matchCode", value = "策略编码") @RequestParam(required = false) String matchCode) {
        try {
            Page<WithholdingSummaryDetailVo> page = this.withholdingSummaryService.findCachePageList(pageable, cacheKey, matchCode);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "新增一行接口，保存当前页数据后，在缓存中行首插入一条数据并返回数据")
    @PostMapping("addItemCache")
    public Result<Page<WithholdingSummaryDetailVo>> addItemCache(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                 @ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                                                 @ApiParam(value = "当前页数据") @RequestBody List<WithholdingSummaryDetailDto> saveList) {
        try {
            this.withholdingSummaryService.addItemCache(cacheKey, saveList);
            Page<WithholdingSummaryDetailVo> page = this.withholdingSummaryService.findCachePageList(pageable, cacheKey, null);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "多行删除并返回指定页数据接口")
    @PostMapping("deleteCacheList")
    public Result<Page<WithholdingSummaryDetailVo>> deleteCacheList(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                    @ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                                                    @ApiParam(value = "当前页数据，包含要删除的行勾选信息") @RequestBody List<WithholdingSummaryDetailDto> saveList) {
        try {
            this.withholdingSummaryService.deleteItemCache(cacheKey, saveList);
            Page<WithholdingSummaryDetailVo> page = this.withholdingSummaryService.findCachePageList(pageable, cacheKey, null);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "保存当前页数据到缓存并返回指定页数据接口")
    @PostMapping("saveCurrentPageCache")
    public Result<Page<WithholdingSummaryDetailVo>> saveCurrentPageCache(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                         @ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                                                         @ApiParam(value = "当前页数据") @RequestBody List<WithholdingSummaryDetailDto> saveList) {
        try {
            this.withholdingSummaryService.saveCurrentPageCache(cacheKey, saveList);
            Page<WithholdingSummaryDetailVo> page = this.withholdingSummaryService.findCachePageList(pageable, cacheKey, null);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "获取缓存中的所有明细行")
    @PostMapping("findCacheList")
    public Result<List<WithholdingSummaryDetailVo>> findCacheList(@ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey) {
        try {
            return Result.ok(this.withholdingSummaryService.findCacheList(cacheKey));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "同步CE单据状态")
    @PostMapping("autoSyncCeState")
    public Result<?> autoSyncCeState(@RequestBody WithholdingSummaryJobParamDto dto) {
        String lockKey = WithholdingSummaryConstant.WITHHOLDING_SUMMARY_AUTO_LOCK;
        try {
            Pageable page = PageRequest.of(1, 50);
            int index = 1;
            boolean hasLock = redisLockService.tryLock(lockKey, TimeUnit.SECONDS, 60 * 60 * 12);
            if (!hasLock) {
                log.error("同步CE单据状态进行中！");
                return Result.error("同步CE单据状态进行中！");
            }
            while (true) {
                List<WithholdingSummaryVo> list = withholdingSummaryService.findAutoSyncCeStateList(page, dto);
                if (CollectionUtils.isEmpty(list)) {
                    break;
                }
                int k = 1;
                for (WithholdingSummaryVo v : list) {
                    try {
                        withholdingSummaryService.autoSyncCeState(v, true);
                        k++;
                    } catch (Exception e) {
                        log.error("同步CE单据状态失败！第[{}]页，第[{}]条：{}", index, k, e.getMessage());
                    }
                }
                index++;
                page = PageRequest.of(index, 50);
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            redisLockService.unlock(lockKey);
        }
        return Result.ok();
    }

    @Autowired(required = false)
    private WithholdingSummaryCeStateAutoSyncXxlJob withholdingSummaryCeStateAutoSyncXxlJob;

    @ApiOperation("测试同步CE单据状态定时任务")
    @PostMapping("testAutoSyncXxlJob")
    public Result<?> testAutoSyncXxlJob(@RequestBody WithholdingSummaryJobParamDto dto) {
        try {
            withholdingSummaryCeStateAutoSyncXxlJob.autoSyncCeState();
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }
}
