package com.biz.crm.tpm.business.withholding.summary.local.process.imports;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.cost.center.sdk.service.CostCenterVoService;
import com.biz.crm.mdm.business.cost.center.sdk.vo.CostCenterVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.biz.crm.mdm.business.supplier.sdk.service.SupplierVoService;
import com.biz.crm.mdm.business.supplier.sdk.vo.SupplierVo;
import com.biz.crm.tpm.business.activity.form.sdk.dto.ActivityFormDto;
import com.biz.crm.tpm.business.activity.form.sdk.service.ActivityFormService;
import com.biz.crm.tpm.business.activity.form.sdk.vo.ActivityFormVo;
import com.biz.crm.tpm.business.withholding.summary.local.util.WithholdingSummaryUtil;
import com.biz.crm.tpm.business.withholding.summary.sdk.constant.WithholdingSummaryConstant;
import com.biz.crm.tpm.business.withholding.summary.sdk.dto.WithholdingSummaryDetailDto;
import com.biz.crm.tpm.business.withholding.summary.sdk.service.WithholdingSummaryService;
import com.biz.crm.tpm.business.withholding.summary.sdk.vo.WithholdingSummaryDetailImportVo;
import com.biz.crm.tpm.business.withholding.summary.sdk.vo.WithholdingSummaryVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 预提汇总明细导入
 * </p>
 *
 * @author chenshuang
 * @since 2023-02-15
 */
@Slf4j
@Component
public class WithholdingSummaryDetailImportProcess implements ImportProcess<WithholdingSummaryDetailImportVo> {

    @Autowired(required = false)
    private WithholdingSummaryService withholdingSummaryService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private DictToolkitService dictToolkitService;

    @Autowired(required = false)
    private SalesOrgVoService salesOrgVoService;

    @Autowired(required = false)
    private SupplierVoService supplierVoService;

    @Autowired(required = false)
    private CostCenterVoService costCenterVoService;

    @Autowired(required = false)
    private ActivityFormService activityFormService;

    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private WithholdingSummaryUtil withholdingSummaryUtil;

    @Override
    public Integer getBatchCount() {
        return Integer.MAX_VALUE;
    }

    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, WithholdingSummaryDetailImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Validate.notNull(params.get("cacheKey"), "缓存键，参数未传入");
        Validate.notNull(params.get("matchCode"), "汇总上传编码，参数未传入");
        Validate.notNull(data, "导入数据不能为空！");
        Validate.isTrue(data.values().size() <= getBatchCount(), "单次导入数据不能超过%s条", getBatchCount());
        String cacheKey = (String) params.get("cacheKey");
        String matchCode = (String) params.get("matchCode");
        log.info("预提汇总明细导入-开始执行大批量导入校验-----------------");
        List<WithholdingSummaryDetailDto> dtoList = val(data, matchCode);
        log.info("预提汇总明细导入-执行大批量导入校验结束-----------------");

        log.info("预提汇总明细导入-开始执行大批量导入保存-----------------");
        withholdingSummaryService.saveCurrentPageCache(cacheKey, dtoList);
        log.info("预提汇总明细导入-结束执行大批量导入保存-----------------");
        return null;
    }

    @Override
    public Class<WithholdingSummaryDetailImportVo> findCrmExcelVoClass() {
        return WithholdingSummaryDetailImportVo.class;
    }

    @Override
    public String getTemplateCode() {
        return "WITHHOLDING_SUMMARY_DETAIL_IMPORT";
    }

    @Override
    public String getTemplateName() {
        return "TPM预提汇总明细编辑导入模板";
    }

    @Override
    public String getBusinessCode() {
        return "WITHHOLDING_SUMMARY_DETAIL_IMPORT";
    }

    @Override
    public String getBusinessName() {
        return "TPM预提汇总明细编辑导入模板";
    }

    public List<WithholdingSummaryDetailDto> val(LinkedHashMap<Integer, WithholdingSummaryDetailImportVo> data, String matchCode) {
        WithholdingSummaryVo withholdingSummaryVo = withholdingSummaryService.findMainByCode(matchCode);

        //业务细类
        Map<String, String> businessExpItemMap = dictToolkitService.findConvertMapByDictTypeCode(WithholdingSummaryConstant.TPM_ACCRUED_ACCOUNT);

        //客商类型
        Map<String, String> merchantsMap = dictToolkitService.findConvertMapByDictTypeCode(WithholdingSummaryConstant.TPM_MERCHANTS_TYPE);

        //活动形式
        ActivityFormDto activityForm = new ActivityFormDto();
        activityForm.setBusinessUnitCode(withholdingSummaryVo.getBusinessUnitCode());
        List<ActivityFormVo> activityFormVos = activityFormService.getList(activityForm);
        Map<String, String> activityFormMap = activityFormVos.stream().collect(Collectors.toMap(ActivityFormVo::getActivityFormCode, ActivityFormVo::getActivityFormName, (v1, v2) -> v2));

        //客户信息
        Map<String, CustomerVo> customerMap = getCustomerMap(data);

        //组织信息
        Map<String, String> saleOrgMap = getOrgMap(data);

        //供应商信息
        Map<String, SupplierVo> supplierMap = getSupplierMap(data);

        //成本中心
        Map<String, CostCenterVo> costCenterMap = getCostCenterMap(data);

        List<WithholdingSummaryDetailDto> dtoList = new ArrayList<>();
        for (Map.Entry<Integer, WithholdingSummaryDetailImportVo> entry : data.entrySet()) {
            Integer k = entry.getKey();
            WithholdingSummaryDetailImportVo v = entry.getValue();
            WithholdingSummaryDetailDto dto = nebulaToolkitService.copyObjectByWhiteList(v, WithholdingSummaryDetailDto.class, HashSet.class, ArrayList.class);

            Validate.notBlank(v.getBusinessExpItem(), "第[%s]行，业务细类不能为空！", k);
            Validate.isTrue(businessExpItemMap.containsKey(v.getBusinessExpItem()),
                    "业务细类[%s]在数据字典[%s]不存在！", v.getBusinessExpItem(), WithholdingSummaryConstant.TPM_ACCRUED_ACCOUNT);
            dto.setBusinessExpItem(businessExpItemMap.get(v.getBusinessExpItem()));

            Validate.notBlank(v.getSalesOrgCode(), "第[%s]行，销售组织编码不能为空！", k);
            Validate.isTrue(saleOrgMap.containsKey(v.getSalesOrgCode()), "第[%s]行，销售组织编码[%s]已禁用或删除！", k, v.getSalesOrgCode());
            dto.setSaleOrgName(saleOrgMap.get(v.getSalesOrgCode()));

            if (StringUtils.isNotEmpty(v.getActivityFormCode())) {
                Validate.isTrue(activityFormMap.containsKey(v.getActivityFormCode()), "第[%s]行，活动形式应编号[%s]对应活动形式不存在或未启用！", k, v.getActivityFormCode());
                dto.setActivityFormName(activityFormMap.get(v.getActivityFormCode()));
            }

            if (StringUtils.isNotEmpty(v.getCustomerName())) {
                Validate.notEmpty(v.getCustomerCode(), "第[%s]行，客户编码不能为空！", k);
            }
            if (StringUtils.isNotEmpty(v.getCustomerCode())) {
                Validate.isTrue(customerMap.containsKey(v.getCustomerCode()), "第[%s]行，客户编码[%s]客户信息不存在或未启用！", k, v.getCustomerCode());
                dto.setCustomerName(customerMap.get(v.getCustomerCode()).getCustomerName());
            } else {
                dto.setCustomerName("");
            }

            if (StringUtils.isNotBlank(v.getCustomerClassifyName())) {
                Validate.isTrue(merchantsMap.containsKey(v.getCustomerClassifyName()),
                        "第[%s]行，客商类型[%s]在数据字典[%s]不存在！", k, v.getCustomerClassifyName(), WithholdingSummaryConstant.TPM_MERCHANTS_TYPE);
                dto.setCustomerClassifyCode(merchantsMap.get(v.getCustomerClassifyName()));

                Validate.notBlank(v.getSupplierCode(), "第[%s]行，客商编码不能为空！", k);
                Validate.isTrue(supplierMap.containsKey(v.getSupplierCode()),
                        "第[%s]行，供应商编码[%s]已禁用或删除！", k, v.getSupplierCode());
                SupplierVo supplierVo = supplierMap.get(v.getSupplierCode());
                dto.setSupplierName(supplierVo.getSupplierName());

                if (StringUtils.equals(WithholdingSummaryConstant.TPM_MERCHANTS_TYPE_SUPPLIER, dto.getCustomerClassifyCode())) {
                    dto.setSpecialTag(WithholdingSummaryConstant.SPECIAL_TAG);
                } else {
                    dto.setSpecialTag("");
                }
            } else {
                dto.setSupplierCode("");
                dto.setSupplierName("");
            }

            Validate.notBlank(v.getCostCenter(), "第[%s]行，成本中心不能为空！", k);
            Validate.isTrue(costCenterMap.containsKey(v.getCostCenter()),
                    "第[%s]行，成本中心编码[%s]已禁用或删除！", k, v.getCostCenter());
            CostCenterVo costCenterVo = costCenterMap.get(v.getCostCenter());
            dto.setCostCenterName(costCenterVo.getCostCenterName());

            Validate.isTrue(Objects.nonNull(v.getWithholdingAmount()), "第[%s]行，不含税金额不能为空！", k);
            Validate.isTrue(Objects.nonNull(v.getEstimateTaxAmount()), "第[%s]行，暂估税金不能为空！", k);
            dto.setEstimateTaxAmount(dto.getEstimateTaxAmount().setScale(WithholdingSummaryConstant.AMOUNT_SCALE, RoundingMode.HALF_DOWN));
            dto.setWithholdingAmount(dto.getWithholdingAmount().setScale(WithholdingSummaryConstant.AMOUNT_SCALE, RoundingMode.HALF_DOWN));
            dto.setWithholdingTaxAmount(dto.getWithholdingAmount().add(dto.getEstimateTaxAmount()).setScale(WithholdingSummaryConstant.AMOUNT_SCALE, RoundingMode.HALF_DOWN));
            Validate.isTrue(Objects.nonNull(v.getAttachmentNum()), "第[%s]行，附件张数不能为空！", k);
            if (Objects.nonNull(dto.getQuantity()) && BigDecimal.ZERO.compareTo(dto.getQuantity()) != 0) {
                dto.setPrice(dto.getWithholdingAmount().divide(dto.getQuantity(), WithholdingSummaryConstant.PRICE_SCALE, RoundingMode.HALF_DOWN));
            }
            dtoList.add(dto);
        }
        withholdingSummaryUtil.valUpdateExtend(dtoList, withholdingSummaryVo.getActivityTypeType(), 2);
        return dtoList;
    }

    public Map<String, CustomerVo> getCustomerMap(LinkedHashMap<Integer, WithholdingSummaryDetailImportVo> data) {
        List<String> customerCodes = data.values().stream().filter(e -> StringUtils.isNotEmpty(e.getCustomerCode())).map(WithholdingSummaryDetailImportVo::getCustomerCode).distinct().collect(Collectors.toList());
        Map<String, CustomerVo> customerMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(customerCodes)) {
            List<CustomerVo> supplierVos = customerVoService.findByCustomerCodes(customerCodes);
            customerMap = supplierVos.stream()
                    .filter(e ->
                            StringUtils.equals(EnableStatusEnum.ENABLE.getCode(), e.getEnableStatus())
                                    && StringUtils.equals(DelFlagStatusEnum.NORMAL.getCode(), e.getDelFlag()))
                    .collect(Collectors.toMap(CustomerVo::getCustomerCode, v -> v, (v1, v2) -> v2));
        }
        return customerMap;
    }

    public Map<String, String> getOrgMap(LinkedHashMap<Integer, WithholdingSummaryDetailImportVo> data) {
        List<String> saleOrgCodes = data.values().stream().filter(e -> StringUtils.isNotEmpty(e.getSalesOrgCode())).map(WithholdingSummaryDetailImportVo::getSalesOrgCode).distinct().collect(Collectors.toList());
        Map<String, String> saleOrgMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(saleOrgCodes)) {
            List<SalesOrgVo> saleOrgVos = salesOrgVoService.findBySalesOrgCodes(saleOrgCodes);
            saleOrgMap = saleOrgVos.stream()
                    .filter(e ->
                            StringUtils.equals(EnableStatusEnum.ENABLE.getCode(), e.getEnableStatus())
                                    && StringUtils.equals(DelFlagStatusEnum.NORMAL.getCode(), e.getDelFlag()))
                    .collect(Collectors.toMap(SalesOrgVo::getSalesOrgCode, SalesOrgVo::getSalesOrgName, (v1, v2) -> v2));
        }
        return saleOrgMap;
    }

    public Map<String, SupplierVo> getSupplierMap(LinkedHashMap<Integer, WithholdingSummaryDetailImportVo> data) {
        List<String> supplierCodes = data.values().stream().filter(e -> StringUtils.isNotEmpty(e.getSupplierCode())).map(WithholdingSummaryDetailImportVo::getSupplierCode).distinct().collect(Collectors.toList());
        Map<String, SupplierVo> supplierMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(supplierCodes)) {
            List<SupplierVo> supplierVos = supplierVoService.findBySupplierCodes(supplierCodes);
            supplierMap = supplierVos.stream()
                    .filter(e ->
                            StringUtils.equals(EnableStatusEnum.ENABLE.getCode(), e.getEnableStatus())
                                    && StringUtils.equals(DelFlagStatusEnum.NORMAL.getCode(), e.getDelFlag()))
                    .collect(Collectors.toMap(SupplierVo::getSupplierCode, v -> v, (v1, v2) -> v2));
        }
        return supplierMap;
    }

    public Map<String, CostCenterVo> getCostCenterMap(LinkedHashMap<Integer, WithholdingSummaryDetailImportVo> data) {
        List<String> costCenterCodes = data.values().stream().filter(e -> StringUtils.isNotEmpty(e.getCostCenter())).map(WithholdingSummaryDetailImportVo::getCostCenter).distinct().collect(Collectors.toList());
        Map<String, CostCenterVo> costCenterMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(costCenterCodes)) {
            List<CostCenterVo> costCenterVos = costCenterVoService.findByCodes(costCenterCodes, "Y");
            costCenterMap = costCenterVos.stream()
                    .filter(e ->
                            StringUtils.equals(EnableStatusEnum.ENABLE.getCode(), e.getEnableStatus())
                                    && StringUtils.equals(DelFlagStatusEnum.NORMAL.getCode(), e.getDelFlag()))
                    .collect(Collectors.toMap(CostCenterVo::getCostCenterCode, v -> v, (v1, v2) -> v2));
        }
        return costCenterMap;
    }
}
